/*
Copyright (c) 2011, Michael Kazhdan and Ming Chuang
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/
#ifdef _WIN32
#  define WINDOWS_LEAN_AND_MEAN
#  define NOMINMAX
#  include <windows.h>
#endif
#include <GL/glew.h>
#include <GL/glut.h>

#include <omp.h>
#include <math.h>
#include <float.h>
#include "Time.h"
#include "ply.h"
#include "Mesh.h"
#include "Octree.h"
#include "SparseMatrix.h"
#include "CmdLineParser.h"
#include "FunctionData.h"
#include "PPolynomial.h"
#include "MemoryUsage.h"
#include "PoissonMeshData.h"
#include <stdarg.h>
#include "Util.h"

//////////////
// For CUDA //
//////////////
#include <cuda_runtime.h>
#include <cutil_inline.h>
#include <cutil_gl_inline.h>
#include <cutil_gl_error.h>
#include <cuda_gl_interop.h>
#include <vector_types.h>
#include <cuda.h>

extern "C" 
{
	void cuKernelDual  (float4* pos, unsigned int* d_eIndices, float* d_eValues, float* d_cValues, GLuint num, GLuint eNum);
	void cuKernelPrimal(float4* pos, unsigned int* d_eIndices, float* d_eValues, float* d_cValues, GLuint num, GLuint eNum);
}
////////////////////////////
// Other Global Variables //
////////////////////////////
template< class Real , bool Primal >
struct Tree
{
	static const int dependence;
	static const int supportCount;
	static void (*cuKernel)( float4* , unsigned int* , float* , float* , GLuint , GLuint );
	static typename MeshOctree< Real , Primal >::ParallelSolver< FastPoint3D< Real > > solver;
	static typename MeshOctree< Real , Primal >::TrackingData< FastPoint3D< Real > > trackingData;
	static typename MeshOctree< Real , Primal >::ElementVector< FastPoint3D< Real > > B , X;
	static SparseMatrix< Real , int > E;
	static Point3D< Real > center;
	static Real scale;

	static std::vector< TriangleIndex > inTriangles;
	static std::vector< PlyVertex< Real > > inVertices;
	static std::vector< PlyVertex< Real > > subdividedVertices;
};
template< class Real , bool Primal > Point3D< Real > Tree< Real , Primal >::center;
template< class Real , bool Primal >          Real   Tree< Real , Primal >::scale;
template< class Real , bool Primal > std::vector< TriangleIndex > Tree< Real , Primal >::inTriangles;
template< class Real , bool Primal > std::vector< PlyVertex< Real > > Tree< Real , Primal >::inVertices;
template< class Real , bool Primal > std::vector< PlyVertex< Real > > Tree< Real , Primal >::subdividedVertices;

template<> const int Tree< float , true  >::supportCount = 8;
template<> const int Tree< float , false >::supportCount = 27;
template<> const int Tree< float , true  >::dependence = 1;
template<> const int Tree< float , false >::dependence = 2;
template<> void ( *Tree< float , true  >::cuKernel )( float4* , unsigned int* , float* , float* , GLuint , GLuint ) = cuKernelPrimal;
template<> void ( *Tree< float , false >::cuKernel )( float4* , unsigned int* , float* , float* , GLuint , GLuint ) = cuKernelDual;
template<> const int Tree< double , true  >::supportCount = 8;
template<> const int Tree< double , false >::supportCount = 27;
template<> const int Tree< double , true  >::dependence = 1;
template<> const int Tree< double , false >::dependence = 2;

template<> void ( *Tree< double , true  >::cuKernel )( float4* , unsigned int* , float* , float* , GLuint , GLuint ) = cuKernelPrimal;
template<> void ( *Tree< double , false >::cuKernel )( float4* , unsigned int* , float* , float* , GLuint , GLuint ) = cuKernelDual;

template< class Real , bool Primal > typename MeshOctree< Real , Primal >::ParallelSolver< FastPoint3D< Real > > Tree< Real , Primal >::solver;
template< class Real , bool Primal > typename MeshOctree< Real , Primal >::TrackingData< FastPoint3D< Real > > Tree< Real , Primal >::trackingData;
template< class Real , bool Primal > typename MeshOctree< Real , Primal >::ElementVector< FastPoint3D< Real > > Tree< Real , Primal >::X;
template< class Real , bool Primal > typename MeshOctree< Real , Primal >::ElementVector< FastPoint3D< Real > > Tree< Real , Primal >::B;
template< class Real , bool Primal > typename SparseMatrix< Real , int > Tree< Real , Primal >::E;

int commentNum=0;
char **comments;
int file_type;

#include "glStage.h"
glStage myStage;

//////////////
// Comments //
//////////////
cmdLineString In( "in" ) , Logs( "logs" );
cmdLineStringArray< 3 > Out( "out" );
cmdLineReadable Verbose( "verbose" ) , FullVerbose( "fullVerbose" ) , ShowError( "showError" ) , Progress( "progress" ) , Dual( "dual" ) , VCycle( "vCycle" ) , Clear( "clear" );
cmdLineInt Depth( "depth" , 8 ) , Iters( "iters" , 2 ) , Subdivide( "subdivide" , 0 ) , VCycles( "vCycles" , 1 ) , ParallelDepth( "parallelDepth" , 5 ) , Threads( "threads" , omp_get_num_procs() ) , GSBlockSize( "gsBlockSize" , 2 );
cmdLineFloat CleanTree( "cutOff" , 1e-8 ) , StepSize( "stepSize" , 0.0001);
cmdLineReadable CPUEvaluation( "cpuEvaluation" ) , LockBoundary( "lockBoundary" ) , Conformal( "conformal" ) , Authalic( "authalic" );


cmdLineReadable* params[]=
{
	&In , &Logs,
	&Out ,
	&Verbose , &FullVerbose , &ShowError , &Progress , &Dual , &VCycle , &Clear ,
	&Depth , &Iters , &Subdivide , &VCycles , &ParallelDepth , &Threads , &GSBlockSize ,
	&CleanTree , &StepSize ,
	&CPUEvaluation , &LockBoundary ,
	&Conformal , &Authalic ,
};

void ShowUsage( char* ex )
{
	printf( "Usage: %s\n" , ex );
	printf( "\t--%s <input triangle mesh>\n" , In.name );
	printf( "\t\t The triangle mesh to be evolved under mean-curvature flow.\n" );

	printf( "\t[--%s <maximum reconstruction depth>=%d]\n" , Depth.name , Depth.value );
	printf( "\t\t Running at depth d corresponds to solving on a 2^d x 2^d x 2^d\n" );
	printf( "\t\t voxel grid.\n");

	printf( "\t[--%s <solver iterations>=%d]\n" , Iters.name , Iters.value );
	printf( "\t\t The number of solver iterations within each resolution\n" );

	printf( "\t[--%s <smoothing step size>=%f]\n" , StepSize.name , StepSize.value );
	printf( "\t\t The time-step used for integrating the flow.\n" );

	printf( "\t[--%s <subdivision iterations>=%d]\n" , Subdivide.name , Subdivide.value );
	printf( "\t\t The number of 1-to-4 triangle subdivision passes.\n" );

	printf( "\t[--%s <number of threads>=%d]\n" , Threads.name , Threads.value );
	printf( "\t\t The number of threads to be used for parallelizing the solver.\n" );

	printf( "\t[--%s]\n" , Verbose.name );
	printf( "\t[--%s]\n" , Progress.name );
	printf( "\t[--%s]\n" , Dual.name );
	printf( "\t[--%s]\n" , LockBoundary.name );
}

template< class Real , bool Primal >
int Execute( int argc , char* argv[] )
{
	int paramNum=sizeof(params)/sizeof(cmdLineReadable*);
	comments=new char*[paramNum+10];
	for(int i=0 ; i<paramNum+10 ; i++ ) comments[i] = new char[1024];
	char valueString[1024];
	for(int i=0;i<paramNum;i++)
		if(params[i]->set)
		{
			params[i]->writeValue(valueString);
			DumpOutput( Logs.value , Verbose.set , comments[commentNum++] , "\t--%s %s\n" , params[i]->name , valueString );
		}
	if( Threads.value<1 ) Threads.value = 1;
	if( ParallelDepth.value>Depth.value )
	{
		ParallelDepth.value = Depth.value;
		Threads.value = 1;
	}

	double t;
	std::vector< std::pair< int , int > > inEdges;
	std::vector< OrientedPoint3D< Real > > surfaceSamples;
	std::vector< int > sampleIndices, vIndices;
	std::vector< int > tIndices , eIndices;

	// Read in the mesh
	{
		if( StepSize.value<0 ) StepSize.value /= -Tree< Real , Primal >::scale*Tree< Real , Primal >::scale;

		PlyReadTriangles( In.value , Tree< Real , Primal >::inVertices , Tree< Real , Primal >::inTriangles , PlyVertex< Real >::ReadProperties , PlyVertex< Real >::ReadComponents , file_type );
		FitVertices< Real , PlyVertex< Real > >( Tree< Real , Primal >::inVertices , Point3D< Real >( 0.5 , 0.5 , 0.5 ) , 1. , Tree< Real , Primal >::center , Tree< Real , Primal >::scale );
		if( LockBoundary.set )
		{
			double t = Time();
			SetBoundaryEdges( Tree< Real , Primal >::inTriangles , inEdges );
			printf( "Got %d boundary edges in:\t%f (seconds)\n" , inEdges.size() , Time()-t );
		}

		myStage.edges.resize( inEdges.size() );
		for( int i=0 ; i<inEdges.size() ; i++ ) myStage.edges[i] = inEdges[i];
		myStage.triangles.resize( Tree< Real , Primal >::inTriangles.size() );
		Tree< Real , Primal >::subdividedVertices.resize ( Tree< Real , Primal >::inVertices.size()  );
		for( int i=0 ; i<Tree< Real , Primal >::inTriangles.size() ; i++ ) myStage.triangles[i] = Tree< Real , Primal >::inTriangles[i];
		for( int i=0 ; i<Tree< Real , Primal >::inVertices.size()  ; i++ ) Tree< Real , Primal >::subdividedVertices[i] = Tree< Real , Primal >::inVertices[i];

		// Subdivide the output for denser sampling
		tIndices.resize( myStage.triangles.size() );
		eIndices.resize( myStage.edges.size() );
		for( int i=0 ; i<myStage.triangles.size() ; i++ ) tIndices[i] = i;
		for( int i=0 ; i<myStage.edges.size() ; i++ ) eIndices[i] = i;
		for( int s=0 ; s<Subdivide.value ; s++ )
		{
			std::vector< int > tempTIndices , tempEIndices;
			SubdivideTrianglesAndEdges( Tree< Real , Primal >::subdividedVertices , myStage.triangles , myStage.edges , tempTIndices , tempEIndices );
			for( int i=0 ; i<tempTIndices.size() ; i++ ) tempTIndices[i] = tIndices[ tempTIndices[i] ];
			for( int i=0 ; i<tempEIndices.size() ; i++ ) tempEIndices[i] = eIndices[ tempEIndices[i] ];
			tIndices = tempTIndices;
			eIndices = tempEIndices;
		}
		// Set the index of each vertex to be equal to one of the original triangles containing it.
		// (Since we assume a continuous function, we don't care which triangle.)
		vIndices.resize( Tree< Real , Primal >::subdividedVertices.size() );
		for( int i=0 ; i<myStage.triangles.size() ; i++ ) for( int c=0 ; c<3 ; c++ ) vIndices[ myStage.triangles[i][c] ] = tIndices[ i ];
	}// end of Reading Mesh

	// Set up the octree
	{
		MeshOctree< Real , Primal > tree;
		t=Time();
		// Initialize the tree
		tree.setTree( Tree< Real , Primal >::subdividedVertices , myStage.triangles , Depth.value , CleanTree.value , Progress.set , Threads.value );
		// Initialize the linear solver
		tree.setSolver( Tree< Real , Primal >::solver , ParallelDepth.value , Threads.value ,  GSBlockSize.value , true );

		Tree< Real , Primal >::solver.cycles = VCycles.value;
		Tree< Real , Primal >::solver.gsIters = Iters.value;
		Tree< Real , Primal >::solver.vCycle = VCycle.set;

		DumpOutput( Verbose.set , comments[commentNum++],"#             Tree set in: %9.1f (s), %9.1f (MB)\n" , Time()-t , tree.maxMemoryUsage );
		DumpOutput( Logs.value ,  Verbose.set , comments[commentNum++] , "    Dimension/Elements: %d / %d\n" , tree.basisDimension() , tree.elements() );
		if( Verbose.set ) printf( "    Vertices/Triangles: %d / %d\n"  , (int)Tree< Real , Primal >::inVertices.size() , (int)Tree< Real , Primal >::inTriangles.size() );
		if( Verbose.set ) printf( "            Tree Depth: %d\n" , Depth.value );

		// Initialize temporary solution and constraint vectors
		Tree< Real , Primal >::B.resize( Tree< Real , Primal >::solver.eM().size() );
		Tree< Real , Primal >::X.resize( Tree< Real , Primal >::solver.eM().size() );

		t = Time();
		// Set the tracking data
		tree.setMatrixTrackingData( Tree< Real , Primal >::trackingData , Progress.set );
		if( Verbose.set ) printf( "Set matrix tracking data in %f\n" , Time()-t );

		// Set the initial embedding
		tree.setIdentityCoefficients( Tree< Real , Primal >::solver.x() );
		Tree< Real , Primal >::solver.pX().SetFromArray( Tree< Real , Primal >::solver.x() );

		// Lock the values for basis functions supported on the boundary
		{
			std::vector< int > fixedIndices;
			std::vector< std::pair< int , FastPoint3D< Real > > > fixedValues;
			tree.setEdgeSupportingIndices( Tree< Real , Primal >::inVertices , inEdges , fixedIndices );
			fixedValues.resize( fixedIndices.size() );
			for( int i=0 ; i<fixedIndices.size() ; i++ ) fixedValues[i] = std::pair< int , FastPoint3D< Real > >( fixedIndices[i] , Tree< Real , Primal >::solver.x()[ fixedIndices[i] ] );
			Tree< Real , Primal >::solver.SetFixedValues( fixedValues );
		}

		myStage.vertices.resize( Tree< Real , Primal >::subdividedVertices.size() );
		for( int i=0 ; i<myStage.vertices.size() ; i++ ) myStage.vertices[i] = Point3D< float >( Tree< Real , Primal >::subdividedVertices[i].point );

		tree.getEvaluationMatrix( Tree< Real , Primal >::subdividedVertices , Tree< Real , Primal >::E );
	}
	return EXIT_SUCCESS;
}

#include "OpenGL.inl"


int main(int argc,char* argv[])
{
	cmdLineParse( argc-1 , &argv[1] , sizeof(params) / sizeof(cmdLineReadable*) , params , 0 );
	if( FullVerbose.set ) Verbose.set = true;
	if( !In.set ) 
	{
		ShowUsage( argv[0] ); 
		return EXIT_FAILURE;
	}
	if( Dual.set ) Execute< double , false >( argc , argv ) , Visualize< double , false >( argc , argv );
	else		   Execute< float  , true  >( argc , argv ) , Visualize< float  , true  >( argc , argv );
}