/*
Copyright (c) 2012, Michael Kazhdan
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/
#ifndef SOLVER_H
#define SOLVER_H

#include <cholmod.h>
#include <omp.h>
#include "SparseMatrix.h"
#include "Vector.h"

#ifdef DLONG
typedef long long SOLVER_LONG;
#define CHOLMOD( name ) cholmod_l_ ## name
#else // !DLONG
typedef       int SOLVER_LONG;
#define CHOLMOD( name ) cholmod_ ## name
#endif // DLONG

#define LOWER_TRIANGULAR 1

template< class Real >
class Solver
{
public:
	virtual bool update( const SparseMatrix< Real , int >& M ) = 0;
	virtual void solve ( const SparseMatrix< Real , int >& M , const Vector< Real >& b , Vector< Real >& x ) = 0;
};
template< class Real >
class CholmodSolver : public Solver< Real >
{
	int dim;
	cholmod_factor* cholmod_L;
	cholmod_dense*  cholmod_b;
	cholmod_sparse* cholmod_M;
	std::vector< bool > flaggedValues;
	void   _init( const SparseMatrix< Real , int >& M );
	bool _update( const SparseMatrix< Real , int >& M );
public:
	static cholmod_common cholmod_C;
	static bool cholmod_C_set;

	CholmodSolver( const SparseMatrix< Real , int >& M );
	CholmodSolver( const SparseMatrix< Real , int >& M , const std::vector< int >& lockedValues );
	~CholmodSolver( void );

	bool update( const SparseMatrix< Real , int >& M );
	bool update( const SparseMatrix< Real , int >& M , const std::vector< int >& lockedValues );

	void solve ( const SparseMatrix< Real , int >& M , const Vector< Real >& b , Vector< Real >& x );

	static void Solve( const SparseMatrix< Real , int >& M , Vector< Real >& x , const Vector< Real >& b )
	{
		CholmodSolver solver( M );
		solver.solve( M , b , x );
	}
	static void Solve( const SparseMatrix< Real , int >& M , Vector< Real >& x , const Vector< Real >& b , const std::vector< int >& lockedValues )
	{
		CholmodSolver solver( M , lockedValues );
		solver.solve( M , b , x );
	}
};
template< class Real > bool CholmodSolver< Real >::cholmod_C_set = false;
template< class Real > cholmod_common CholmodSolver< Real >::cholmod_C;

template< class Real >
CholmodSolver< Real >::CholmodSolver( const SparseMatrix< Real , int >& M )
{
	_init( M );
	update( M );
}
template< class Real >
CholmodSolver< Real >::CholmodSolver( const SparseMatrix< Real , int >& M , const std::vector< int >& lockedValues )
{
	_init( M );
	update( M , lockedValues );
}
template< class Real >
bool CholmodSolver< Real >::update( const SparseMatrix< Real , int >& M )
{
	for( int i=0 ; i<flaggedValues.size() ; i++ ) flaggedValues[i] = false;
	return _update( M );
}
template< class Real >
bool CholmodSolver< Real >::update( const SparseMatrix< Real , int >& M , const std::vector< int >& lockedValues )
{
	for( int i=0 ; i<flaggedValues.size() ; i++ ) flaggedValues[i] = false;
	for( int i=0 ; i<lockedValues.size() ; i++ )
		if( lockedValues[i]<0 || lockedValues[i]>=M.rows )
		{
			fprintf( stderr , "[ERROR] Locked index out of bounds: 0 <= %d < %d\n" , lockedValues[i] , M.rows );
			exit( 0 );
		}
		else flaggedValues[ lockedValues[i] ] = true;
	return _update( M );
}
template< class Real >
void CholmodSolver< Real >::_init( const SparseMatrix< Real , int >& M )
{
	{
		if( !cholmod_C_set ) CHOLMOD(start)( &cholmod_C );
		cholmod_C_set = true;
	}
	dim = M.rows;

	flaggedValues.resize( dim , false );

#if LOWER_TRIANGULAR
	int maxEntries = ( M.Entries()-M.rows ) / 2 + M.rows;
	cholmod_M = CHOLMOD(allocate_sparse)( dim , dim , maxEntries , 0 , 1 , -1 , CHOLMOD_REAL , &cholmod_C );
#else // !LOWER_TRIANGULAR
	int maxEntries = M.Entries();
	cholmod_M = CHOLMOD(allocate_sparse)( dim , dim , maxEntries , 0 , 1 ,  0 , CHOLMOD_REAL , &cholmod_C );
#endif // LOWER_TRIANGULAR
	cholmod_M->i = malloc( sizeof( SOLVER_LONG ) * maxEntries );
	cholmod_M->x = malloc( sizeof( double ) * maxEntries );

	SOLVER_LONG *_p = (SOLVER_LONG*)cholmod_M->p;
	SOLVER_LONG *_i = (SOLVER_LONG*)cholmod_M->i;

	int off = 0;
	dim = 0;

	for( int i=0 ; i<M.rows ; i++ )
	{
		_p[dim++] = off;
		for( int j=0 ; j<M.rowSizes[i] ; j++ )
#if LOWER_TRIANGULAR
			if( M[i][j].N>=i )
#endif // LOWER_TRIANGULAR
				_i[off++] = M[i][j].N;
	}
	_p[dim] = off;

	cholmod_L = CHOLMOD(analyze)( cholmod_M , &cholmod_C );
	cholmod_b = CHOLMOD(allocate_dense)( dim , 1 , dim , cholmod_M->xtype , &cholmod_C );
}
template< class Real >
bool CholmodSolver< Real >::_update( const SparseMatrix< Real , int >& M )
{
	double *_x = (double*)cholmod_M->x;
	int off = 0;

	SOLVER_LONG *_p = (SOLVER_LONG*)cholmod_M->p;
#pragma omp parallel for
	for( int i=0 ; i<M.rows ; i++ )
	{
		int off = _p[i];
		for( int j=0 ; j<M.rowSizes[i] ; j++ )
#if LOWER_TRIANGULAR
			if( M[i][j].N>=i )
#endif // LOWER_TRIANGULAR
				if( flaggedValues[i] || flaggedValues[ M[i][j].N ] )
				{
					if( M[i][j].N==i ) _x[off++] = 1.;
					else               _x[off++] = 0.;
				}
				else _x[off++] = double( M[i][j].Value );
	}

	cholmod_C.print = 0;
	CHOLMOD(factorize)( cholmod_M , cholmod_L , &cholmod_C );
	if( cholmod_C.status==CHOLMOD_NOT_POSDEF )
	{
		fprintf( stderr , "[WARNING] Matrix not positive-definite\n" );
		return false;
	}
	else if( cholmod_C.status==CHOLMOD_OUT_OF_MEMORY )
	{
		fprintf( stderr , "[WARNING] CHOLMOD ran out of memory\n" );
		return false;
	}
	else if( cholmod_C.status!=CHOLMOD_OK )
	{
		fprintf( stderr , "[WARNING] CHOLMOD status not OK: %d\n" , cholmod_C.status );
		return false;
	}
	return true;
}
template< class Real >
CholmodSolver< Real >::~CholmodSolver( void )
{
	if( cholmod_L ) CHOLMOD(free_factor)( &cholmod_L , &cholmod_C ) , cholmod_L = NULL;
	if( cholmod_b ) CHOLMOD(free_dense )( &cholmod_b , &cholmod_C ) , cholmod_b = NULL;
	if( cholmod_M ) CHOLMOD(free_sparse)( &cholmod_M , &cholmod_C ) , cholmod_M = NULL;
}

template< class Real >
void CholmodSolver< Real >::solve( const SparseMatrix< Real , int >& M , const Vector< Real >& b , Vector< Real >& x )
{
	double* _b = (double*)cholmod_b->x;
	for( int i=0 ; i<dim ; i++ ) _b[i] = b[i];
	for( int i=0 ; i<b.size() ; i++ ) if( flaggedValues[i] )
		for( int j=0 ; j<M.rowSizes[i] ; j++ ) if( M[i][j].N!=i ) _b[ M[i][j].N ] -= M[i][j].Value * x[i];
	for( int i=0 ; i<b.size() ; i++ ) if( flaggedValues[i] ) _b[i] = x[i];

	cholmod_dense* cholmod_x = CHOLMOD(solve)( CHOLMOD_A , cholmod_L , cholmod_b , &cholmod_C );
	double* _x = (double*)cholmod_x->x;
	for( int i=0 ; i<dim ; i++ ) x[i] = _x[i];

	CHOLMOD(free_dense)( &cholmod_x , &cholmod_C );
}
#endif // SOLVER_H