/*
Copyright (c) 2012, Michael Kazhdan
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/
#ifndef ARRAY_INCLUDED
#define ARRAY_INCLUDED

#define ARRAY_MEMORY_DEBUG 0
#define SUPPORT_ALIGNMENT 0
#define STORE_INFO 1
#define ASSERT( x ) { if( !( x ) ) __debugbreak(); }

// Code from http://stackoverflow.com
inline void* aligned_malloc( size_t size , int align )
{
	void* mem = malloc( size + align + sizeof( void* ) );
	char* amem = ( (char*) mem ) + sizeof( void* );
//	amem += align - ( ( size_t ) amem & (align-1) );
	amem += ~( ( size_t ) amem & (align-1) );
	( ( void** ) amem )[-1] = mem;
	return amem;
}
inline void aligned_free( void* mem ) { free( ( ( void** )mem )[-1] ); }

#if ARRAY_MEMORY_DEBUG
#include "Array.inl"
#define      Pointer( ... )      Array< __VA_ARGS__ >
#define ConstPointer( ... ) ConstArray< __VA_ARGS__ >
template< class C > void        VFreePointer( Array< C >& a )                                     { a.Free( ); }
template< class C > void         FreePointer( Array< C >& a )                                     { a.Free( ); }
#if SUPPORT_ALIGNMENT
template< class C > Array< C > VAllocPointer( int size , int alignment=1 , const char* name=NULL ){ return Array< C >( size , alignment , false , true  , name ); }
template< class C > Array< C >  AllocPointer( int size , int alignment=1 , const char* name=NULL ){ return Array< C >( size , alignment , false , false , name ); }
#else // !SUPPORT_ALIGNMENT
template< class C > Array< C > VAllocPointer( int size , const char* fileName=NULL ){ return Array< C >( size , false , true  , name ); }
#if STORE_INFO
template< class C > inline Array< C > AllocPointer( int size , const char* fileName=NULL , int lineNum=-1 ){ return Array< C >( size , false , false , fileName , lineNum ); }
#else // !STORE_INFO
template< class C > Array< C >  AllocPointer( int size , const char* name=NULL ){ return Array< C >( size , false , false , name ); }
#endif // STORE_INFO
#endif // SUPPORT_ALIGNMENT
template< class C > Array< C > NullPointer( void )                                              { return Array< C >( ); }
template< class C > C*      PointerAddress( Array< C >& a )                                     { return a.pointer(); }
template< class C >      Array< C >       GetPointer(       C& c )                              { return      Array< C >::FromPointer( &c , 1 ); }
template< class C > ConstArray< C >  GetConstPointer( const C& c )                              { return ConstArray< C >::FromPointer( &c , 1 ); }
#else // !ARRAY_MEMORY_DEBUG
#define      Pointer( ... )       __VA_ARGS__*
#define ConstPointer( ... ) const __VA_ARGS__*
#ifdef WIN32
#if SUPPORT_ALIGNMENT
//template< class C > C*   AllocPointer( int size , int alignment=1 , const char* name=NULL ){ return (C*)_aligned_malloc( sizeof(C) * size , alignment ); }
//#define                   FreePointer( ptr ) if( ptr ) _aligned_free( ptr ) , ptr = NULL
template< class C > C*     AllocPointer( int size , int alignment=1 , const char* name=NULL ){ return (C*)aligned_malloc( sizeof(C) * size , alignment ); }
#define                     FreePointer( ptr ) if( ptr ) aligned_free( ptr ) , ptr = NULL
#else // !SUPPORT_ALIGNMENT
#if STORE_INFO
template< class C > C*     AllocPointer( int size , const char* fileName=NULL , int lineNum=-1 ){ return (C*)malloc( sizeof(C) * size ); }
#else // !STORE_INFO
template< class C > C*     AllocPointer( int size , const char* name=NULL ){ return (C*)malloc( sizeof(C) * size ); }
#endif // STORE_INFO
#define                     FreePointer( ptr ) if( ptr ) free( ptr ) , ptr = NULL
#endif // SUPPORT_ALIGNMENT
#else // !WIN32
#if SUPPORT_ALIGNMENT
template< class C > C*   AllocPointer( int size , int alignment=1 , const char* name=NULL ){ return (C*)memalign( sizeof(C) * size , alignment ); }
#else // !SUPPORT_ALIGNMENT
template< class C > C*     AllocPointer( int size , const char* fileName=NULL , int lineNum=-1 ){ return (C*)malloc( sizeof(C) * size ); }
#endif // SUPPORT_ALIGNMENT
#define                   FreePointer( ptr ) if( ptr ) free( ptr ) , ptr = NULL
#endif // WIN32
#define                  VFreePointer( ptr ) if( ptr ) VirtualFree( ptr , 0 , MEM_RELEASE ) , ptr = NULL

#ifdef WIN32
template< class C > C*  VAllocPointer( int size , int alignment=1 , const char* name=NULL ){ return (C*)VirtualAlloc( NULL , size*sizeof(C) , MEM_RESERVE | MEM_COMMIT , PAGE_READWRITE ); }
#endif
template< class C > C*    NullPointer( void ){ return NULL; }
template< class C > C* PointerAddress( C* c ){ return c; }
template< class C >        C*     GetPointer(       C& c ){ return &c; }
template< class C > const C* GetConstPointer( const C& c ){ return &c; }
#endif // ARRAY_MEMORY_DEBUG
#endif // ARRAY_INCLUDED
