/*
Copyright (c) 2005, Michael Kazhdan
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/

#ifndef SET_UP_INCLUDED
#define SET_UP_INCLUDED
#include <vector>
#include <math.h>


/////////////////////
// Data Structures //
/////////////////////
typedef struct Matrix{
	double m[3][3];
} Matrix;
typedef struct Vertex{
	double v[3];
} Vertex;
typedef struct TriangleIndex{
	int idx[3];
} TriangleIndex;
typedef struct XForm{
	Matrix M;
	Vertex v;
} XForm;

///////////////////////////
// Function Declarations //
///////////////////////////
Vertex ZeroVertex(void);
double SquareLength(const Vertex& v);
double DotProduct(const Vertex& v1,const Vertex& v2);
double Length(const Vertex& v);
Vertex Unit(const Vertex& v);
Vertex CrossProduct(const Vertex& v1,const Vertex& v2);

Matrix IdentityMatrix(void);
Matrix Transpose(const Matrix& m);
Vertex Multiply(const Matrix& m,const Vertex& v);
Matrix Multiply(const Matrix& m1,const Matrix& m2);

XForm IdentityXForm(void);
Vertex Multiply(const XForm& xForm,const Vertex& v);
XForm Multiply(const XForm& xForm1,const XForm& xForm2);
XForm Multiply(const XForm& xForm,const Matrix& M);
XForm Multiply(const Matrix& M,const XForm& xForm);

void DecomposeSymmetric(const Matrix& M,Vertex eigenVectors[3],Vertex& eigenValues,const int iters=10);

double Radius(const std::vector<TriangleIndex>& triangles,const std::vector<Vertex>& vertices);
Vertex Center(const std::vector<TriangleIndex>& triangles,const std::vector<Vertex>& vertices);
void Transform(std::vector<Vertex>& vertices,const XForm& xForm);
XForm AnisotropyTransform(const std::vector<TriangleIndex>& triangles,const std::vector<Vertex>& vertices,const int& iterations,const double EPS);

////////////////////////
// Inline Definitions //
////////////////////////
inline Vertex ZeroVertex(void){
	Vertex v;
	v.v[0]=v.v[1]=v.v[2]=0;
	return v;
}
inline double SquareLength(const Vertex& v){return v.v[0]*v.v[0]+v.v[1]*v.v[1]+v.v[2]*v.v[2];}
inline double DotProduct(const Vertex& v1,const Vertex& v2){return v1.v[0]*v2.v[0]+v1.v[1]*v2.v[1]+v1.v[2]*v2.v[2];}
inline double Length(const Vertex& v){return sqrt(SquareLength(v));}
inline Vertex Unit(const Vertex& v){
	Vertex w=v;
	double t=Length(v);
	w.v[0]/=t;
	w.v[1]/=t;
	w.v[2]/=t;
	return w;
}
inline Vertex CrossProduct(const Vertex& v1,const Vertex& v2){
	Vertex v;
	v.v[0]= v1.v[1]*v2.v[2]-v1.v[2]*v2.v[1];
	v.v[1]=-v1.v[0]*v2.v[2]+v1.v[2]*v2.v[0];
	v.v[2]= v1.v[0]*v2.v[1]-v1.v[1]*v2.v[0];
	return v;
}

inline Matrix IdentityMatrix(void){
	Matrix M;
	M.m[0][1]=M.m[0][2]=0;
	M.m[1][0]=M.m[1][2]=0;
	M.m[2][0]=M.m[2][1]=0;
	M.m[0][0]=M.m[1][1]=M.m[2][2]=1;
	return M;
}
inline Matrix Transpose(const Matrix& M){
	Matrix Mt;
	Mt.m[0][0]=M.m[0][0];
	Mt.m[1][0]=M.m[0][1];
	Mt.m[2][0]=M.m[0][2];
	Mt.m[0][1]=M.m[1][0];
	Mt.m[1][1]=M.m[1][1];
	Mt.m[2][1]=M.m[1][2];
	Mt.m[0][2]=M.m[2][0];
	Mt.m[1][2]=M.m[2][1];
	Mt.m[2][2]=M.m[2][2];
	return Mt;
}
inline Vertex Multiply(const Matrix& m,const Vertex& v){
	Vertex w;
	w.v[0]=m.m[0][0]*v.v[0]+m.m[1][0]*v.v[1]+m.m[2][0]*v.v[2];
	w.v[1]=m.m[0][1]*v.v[0]+m.m[1][1]*v.v[1]+m.m[2][1]*v.v[2];
	w.v[2]=m.m[0][2]*v.v[0]+m.m[1][2]*v.v[1]+m.m[2][2]*v.v[2];
	return w;
}
inline Matrix Multiply(const Matrix& m1,const Matrix& m2){
	Matrix m;
	m.m[0][0]=m1.m[0][0]*m2.m[0][0]+m1.m[1][0]*m2.m[0][1]+m1.m[2][0]*m2.m[0][2];
	m.m[0][1]=m1.m[0][1]*m2.m[0][0]+m1.m[1][1]*m2.m[0][1]+m1.m[2][1]*m2.m[0][2];
	m.m[0][2]=m1.m[0][2]*m2.m[0][0]+m1.m[1][2]*m2.m[0][1]+m1.m[2][2]*m2.m[0][2];
	m.m[1][0]=m1.m[0][0]*m2.m[1][0]+m1.m[1][0]*m2.m[1][1]+m1.m[2][0]*m2.m[1][2];
	m.m[1][1]=m1.m[0][1]*m2.m[1][0]+m1.m[1][1]*m2.m[1][1]+m1.m[2][1]*m2.m[1][2];
	m.m[1][2]=m1.m[0][2]*m2.m[1][0]+m1.m[1][2]*m2.m[1][1]+m1.m[2][2]*m2.m[1][2];
	m.m[2][0]=m1.m[0][0]*m2.m[2][0]+m1.m[1][0]*m2.m[2][1]+m1.m[2][0]*m2.m[2][2];
	m.m[2][1]=m1.m[0][1]*m2.m[2][0]+m1.m[1][1]*m2.m[2][1]+m1.m[2][1]*m2.m[2][2];
	m.m[2][2]=m1.m[0][2]*m2.m[2][0]+m1.m[1][2]*m2.m[2][1]+m1.m[2][2]*m2.m[2][2];
	return m;
}

inline XForm IdentityXForm(void){
	XForm xForm;
	xForm.M=IdentityMatrix();
	xForm.v=ZeroVertex();
	return xForm;
}
inline Vertex Multiply(const XForm& xForm,const Vertex& v){
	Vertex w=Multiply(xForm.M,v);
	w.v[0]+=xForm.v.v[0];
	w.v[1]+=xForm.v.v[1];
	w.v[2]+=xForm.v.v[2];
	return w;
}
inline XForm Multiply(const XForm& xForm1,const XForm& xForm2){
	XForm xForm;
	xForm.M=Multiply(xForm1.M,xForm2.M);
	xForm.v=Multiply(xForm1.M,xForm2.v);
	xForm.v.v[0]+=xForm1.v.v[0];
	xForm.v.v[1]+=xForm1.v.v[1];
	xForm.v.v[2]+=xForm1.v.v[2];
	return xForm;
}
inline XForm Multiply(const XForm& xForm,const Matrix& M){
	XForm x;
	x.M=Multiply(xForm.M,M);
	x.v=xForm.v;
	return x;
}
inline XForm Multiply(const Matrix& M,const XForm& xForm){
	XForm x;
	x.M=Multiply(M,xForm.M);
	x.v=Multiply(M,xForm.v);
	return x;
}

#endif
