/*
Copyright (c) 2005, Michael Kazhdan
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/
#include "setUp3D.h"


/////////////////////////////////
// Local Function Declarations //
/////////////////////////////////
double Area		(const Vertex& v1,const Vertex& v2,const Vertex& v3);
Vertex Center	(const Vertex& v1,const Vertex& v2,const Vertex& v3);
Matrix Moments	(const Vertex& v1,const Vertex& v2,const Vertex& v3);

double TransformedRadius	(const std::vector<TriangleIndex>& triangles,const std::vector<Vertex>& vertices,const XForm& xForm);
Vertex TransformedCenter	(const std::vector<TriangleIndex>& triangles,const std::vector<Vertex>& vertices,const XForm& xForm);
Matrix TransformedMoments	(const std::vector<TriangleIndex>& triangles,const std::vector<Vertex>& vertices,const XForm& xForm);

//////////////////////////
// Function Definitions //
//////////////////////////
double Area(const Vertex& v1,const Vertex& v2,const Vertex& v3){
	Vertex p1,p2;
	p1=v2;
	p2=v3;
	p1.v[0]-=v1.v[0];
	p1.v[1]-=v1.v[1];
	p1.v[2]-=v1.v[2];
	p2.v[0]-=v1.v[0];
	p2.v[1]-=v1.v[1];
	p2.v[2]-=v1.v[2];
	return Length(CrossProduct(p1,p2))/2;
}
Vertex Center(const Vertex& v1,const Vertex& v2,const Vertex& v3){
	Vertex v;
	v.v[0]=(v1.v[0]+v2.v[0]+v3.v[0])/3;
	v.v[1]=(v1.v[1]+v2.v[1]+v3.v[1])/3;
	v.v[2]=(v1.v[2]+v2.v[2]+v3.v[2])/3;
	return v;
}
Matrix Moments(const Vertex& v1,const Vertex& v2,const Vertex& v3){
	Matrix M;
	Vertex p,p1,p2;
	int i,j;
	double area;
	double dd,dq1,dq2,q1q2,q1q1,q2q2;
	double fdd,fdq1,fdq2,fq1q1,fq2q2,fq1q2;

	area=Area(v1,v2,v3)*2;

	p.v[0]=v1.v[0];
	p.v[1]=v1.v[1];
	p.v[2]=v1.v[2];

	p1.v[0]=v3.v[0]-v1.v[0];
	p1.v[1]=v3.v[1]-v1.v[1];
	p1.v[2]=v3.v[2]-v1.v[2];

	p2.v[0]=v2.v[0]-v3.v[0];
	p2.v[1]=v2.v[1]-v3.v[1];
	p2.v[2]=v2.v[2]-v3.v[2];
			
	fq2q2 = 1.0/12.0;
	fq1q1 = 1.0/ 4.0;
	fq1q2 = 1.0/ 8.0;
	fdq2  = 1.0/ 6.0;
	fdq1  = 1.0/ 3.0;
	fdd   = 1.0/ 2.0;
	
	for(i=0;i<3;i++){
		for(j=0;j<=i;j++){
			dd	=p.v[i]*p.v[j];
			q1q1=p1.v[i]*p1.v[j];
			q2q2=p2.v[i]*p2.v[j];
			dq1	=p1.v[i]*p.v[j]+p1.v[j]*p.v[i];
			dq2	=p2.v[i]*p.v[j]+p2.v[j]*p.v[i];
			q1q2=p1.v[i]*p2.v[j]+p1.v[j]*p2.v[i];
			M.m[i][j]=
				fq2q2*q2q2+
				fq1q1*q1q1+
				fq1q2*q1q2+
				fdq2 * dq2+
				fdq1 * dq1+
				fdd  *  dd;
			M.m[i][j]*=area;
			M.m[j][i]=M.m[i][j];
		}
	}
	return M;
}
double Radius(const std::vector<TriangleIndex>& triangles,const std::vector<Vertex>& vertices){return TransformedRadius(triangles,vertices,IdentityXForm());}
Vertex Center(const std::vector<TriangleIndex>& triangles,const std::vector<Vertex>& vertices){return TransformedCenter(triangles,vertices,IdentityXForm());}
void Transform(std::vector<Vertex>& vertices,const XForm& xForm){for(int i=0;i<vertices.size();i++){vertices[i]=Multiply(xForm,vertices[i]);}}
Vertex TransformedCenter(const std::vector<TriangleIndex>& triangles,const std::vector<Vertex>& vertices,const XForm& xForm){
	Vertex center=ZeroVertex(),p;
	double a,area;
	
	area=0;
	for(int i=0;i<triangles.size();i++){
		Vertex v[3];
		v[0]=Multiply(xForm,vertices[triangles[i].idx[0]]);
		v[1]=Multiply(xForm,vertices[triangles[i].idx[1]]);
		v[2]=Multiply(xForm,vertices[triangles[i].idx[2]]);
		a=Area(v[0],v[1],v[2]);
		p=Center(v[0],v[1],v[2]);
		center.v[0]+=p.v[0]*a;
		center.v[1]+=p.v[1]*a;
		center.v[2]+=p.v[2]*a;
		area+=a;
	}
	if(area==0){center.v[0]=center.v[1]=center.v[2]=0;}
	else{
		center.v[0]/=area;
		center.v[1]/=area;
		center.v[2]/=area;
	}
	return center;
}
double TransformedRadius(const std::vector<TriangleIndex>& triangles,const std::vector<Vertex>& vertices,const XForm& xForm){
	Matrix M=TransformedMoments(triangles,vertices,xForm);
	return sqrt(M.m[0][0]+M.m[1][1]+M.m[2][2]);
}
Matrix TransformedMoments(const std::vector<TriangleIndex>& triangles,const std::vector<Vertex>& vertices,const XForm& xForm){
	int i,j;
	double area=0;
	Matrix M=IdentityMatrix();
	
	M.m[0][0]=M.m[1][1]=M.m[2][2]=0;
	for(i=0;i<triangles.size();i++){
		Vertex v[3];
		v[0]=Multiply(xForm,vertices[triangles[i].idx[0]]);
		v[1]=Multiply(xForm,vertices[triangles[i].idx[1]]);
		v[2]=Multiply(xForm,vertices[triangles[i].idx[2]]);
		area+=Area(v[0],v[1],v[2]);
		Matrix tempM=Moments(v[0],v[1],v[2]);
		for(j=0;j<3;j++){for(int k=0;k<3;k++){M.m[j][k]+=tempM.m[j][k];}}
	}
	for(i=0;i<3;i++){for(j=0;j<3;j++){M.m[i][j]/=area;}}
	return M;
}
void DecomposeSymmetric(const Matrix& M,Vertex eigenVectors[3],Vertex& eigenValues,const int iters){
	Matrix tempM=M;
	int i,j;

	eigenVectors[0]=ZeroVertex();
	eigenVectors[0].v[0]=1;
	for(i=0;i<iters;i++){eigenVectors[0]=Unit(Multiply(tempM,eigenVectors[0]));}
	eigenValues.v[0]=DotProduct(eigenVectors[0],Multiply(tempM,eigenVectors[0]));
	for(i=0;i<3;i++){
		for(j=0;j<3;j++){
			tempM.m[i][j]-=eigenVectors[0].v[i]*eigenVectors[0].v[j]*eigenValues.v[0];
		}
	}
	eigenVectors[1]=ZeroVertex();
	eigenVectors[1].v[0]=1;
	for(i=0;i<iters;i++){
		eigenVectors[1]=Multiply(tempM,eigenVectors[1]);
		double t=DotProduct(eigenVectors[0],eigenVectors[1]);
		eigenVectors[1].v[0]-=eigenVectors[0].v[0]*t;
		eigenVectors[1].v[1]-=eigenVectors[0].v[1]*t;
		eigenVectors[1].v[2]-=eigenVectors[0].v[2]*t;
		eigenVectors[1]=Unit(eigenVectors[1]);
	}
	eigenValues.v[1]=DotProduct(eigenVectors[1],Multiply(tempM,eigenVectors[1]));
	for(i=0;i<3;i++){
		for(j=0;j<3;j++){
			tempM.m[i][j]-=eigenVectors[1].v[i]*eigenVectors[1].v[j]*eigenValues.v[1];
		}
	}
	eigenVectors[2]=CrossProduct(eigenVectors[0],eigenVectors[1]);
	eigenValues.v[2]=DotProduct(eigenVectors[2],Multiply(tempM,eigenVectors[2]));
	for(i=0;i<3;i++){
		for(j=0;j<3;j++){
			tempM.m[i][j]-=eigenVectors[2].v[i]*eigenVectors[2].v[j]*eigenValues.v[2];
		}
	}
}

XForm AnisotropyTransform(const std::vector<TriangleIndex>& triangles,const std::vector<Vertex>& vertices,const int& iterations,const double EPS){
	XForm xForm=IdentityXForm(),xf=IdentityXForm();
	Vertex center,ctr;
	double radius,rad;

	center=TransformedCenter(triangles,vertices,xForm);
	xf.v.v[0]=-center.v[0];
	xf.v.v[1]=-center.v[1];
	xf.v.v[2]=-center.v[2];
	radius=TransformedRadius(triangles,vertices,xf);

	for(int i=0;i<iterations;i++){
		Vertex ctr;
		Vertex eigenVectors[3];
		Vertex eigenValues;
		Matrix moments,rot,scl;

		// Get the center of the transformed model
		ctr=TransformedCenter(triangles,vertices,xForm);
		xForm.v.v[0]-=ctr.v[0];
		xForm.v.v[1]-=ctr.v[1];
		xForm.v.v[2]-=ctr.v[2];
		// Get the moments matrix of the transformed model
		moments=TransformedMoments(triangles,vertices,xForm);
		// Decompose the matrix
		DecomposeSymmetric(moments,eigenVectors,eigenValues);
		
		double temp=eigenValues.v[0]+eigenValues.v[1]+eigenValues.v[2];
		if		(eigenValues.v[0]/temp < EPS){eigenValues.v[0]=eigenValues.v[1]=eigenValues.v[2]=1;}
		else if	(eigenValues.v[1]/temp < EPS){eigenValues.v[1]=eigenValues.v[2]=1;}
		else if	(eigenValues.v[2]/temp < EPS){eigenValues.v[2]=1;}

		// Generate the rotation and scale matrices
		double det=pow(eigenValues.v[0]*eigenValues.v[1]*eigenValues.v[2],1.0/6);
		for(int j=0;j<3;j++){
			for(int k=0;k<3;k++){rot.m[j][k]=eigenVectors[k].v[j];}
			scl.m[j][j]=sqrt(1.0/eigenValues.v[j])*det;
		}
		xForm=Multiply(rot,xForm);
		xForm=Multiply(scl,xForm);
		xForm=Multiply(Transpose(rot),xForm);
	}

	ctr=TransformedCenter(triangles,vertices,xForm);
	xForm.v.v[0]-=ctr.v[0];
	xForm.v.v[1]-=ctr.v[1];
	xForm.v.v[2]-=ctr.v[2];
	rad=TransformedRadius(triangles,vertices,xForm);
	xf.v=center;
	xf.M.m[0][0]=xf.M.m[1][1]=xf.M.m[2][2]=radius/rad;
	return Multiply(xf,xForm);
}
