/*
Copyright (c) 2005, Michael Kazhdan
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <sys/timeb.h>
#ifndef WIN32
#include <sys/time.h>
#endif
#include "cmdLineParser.h"
#include "ply.h"

const int ITER=5;
const double EPS=10e-8;

double GetTime(void){
#ifdef WIN32
	struct _timeb t;
	_ftime(&t);
	return t.time+t.millitm/1000.0;
#else
	struct timeval t;
	gettimeofday(&t,NULL);
	return t.tv_sec+(double)t.tv_usec/1000000;
#endif
}
void ShowUsage(char* ex){
	printf("Usage %s:\n",ex);
	printf("\t--in   <input points>\n");
	printf("\t--out  <ouput triangle mesh>\n");
	printf("\t[--iter <number of iterations=%d>]\n",ITER);
}

int WriteTriangles(char* fileName,std::vector<TriangleIndex>& triangles,std::vector<Vertex>& vertices,int file_type=PlyDefaultFileType()){
	int ret=0;
	char* extension;
	
	extension=GetFileExtension(fileName);
	if(!strcasecmp(extension,"ply")){ret=PlyWriteTriangles(fileName,triangles,vertices,file_type);}
	else{fprintf(stderr,"Uknown file extension: %s\n",extension);}
	delete[] extension;
	return ret;
}
int ReadTriangles(char* fileName,std::vector<TriangleIndex>& triangles,std::vector<Vertex>& vertices,int& file_type){
	int ret=0;
	char* extension;
	
	extension=GetFileExtension(fileName);
	if(!strcasecmp(extension,"ply")){ret=PlyReadTriangles(fileName,triangles,vertices,file_type);}
	else{fprintf(stderr,"Uknown file extension: %s\n",extension);}
	delete[] extension;
	return ret;
}
int GetCount(int dim,int order){
	int cnt=0;
	if(!order || dim==1){return 1;}
	if(order==1){return dim;}
	for(int i=0;i<dim;i++){cnt+=GetCount(dim-i,order-1);}
	return cnt;
}

int main(int argc,char* argv[]){
	std::vector<TriangleIndex> triangles;
	std::vector<Vertex> vertices;
	char* paramNames[]={"in","out","iter"};
	cmdLineReadable* params[3];
	cmdLineInt Iter;
	cmdLineString In,Out;
	int iter,file_type;
	double t;
	XForm xForm;

	params[0]=&In;
	params[1]=&Out;
	params[2]=&Iter;
	cmdLineParse(argc-1,&argv[1],paramNames,3,params,0);

	if(!In.set){
		printf("No input file specified\n");
		ShowUsage(argv[0]);
		return EXIT_FAILURE;
	}
	if(!Out.set){
		printf("No input file specified\n");
		ShowUsage(argv[0]);
		return EXIT_FAILURE;
	}
	if(Iter.set){iter=Iter.value;}
	else{iter=ITER;}

	if(!ReadTriangles(In.value,triangles,vertices,file_type)){
		fprintf(stderr,"Failed to read triangles from: %s\n",In.value);
		return EXIT_FAILURE;
	}
	t=GetTime();
	xForm=AnisotropyTransform(triangles,vertices,iter,EPS);
	Transform(vertices,xForm);
	t=GetTime()-t;
	if(!WriteTriangles(Out.value,triangles,vertices,file_type)){
		fprintf(stderr,"Failed to write triangles to: %s\n",Out.value);
		return EXIT_FAILURE;
	}
	printf("Total Time: %f\n",t);
	return EXIT_SUCCESS;
}
