#include <stdio.h>
#include <stdlib.h>
#include <setjmp.h>

#ifdef _WIN32
#include <windows.h>
#include "JPEG/jpeglib.h"
#include "JPEG/jerror.h"
#include "JPEG/jmorecfg.h"
#else // !_WIN32
#include <jpeglib.h>
#include <jerror.h>
#include <jmorecfg.h>
#endif // _WIN32

struct my_error_mgr
{
	struct jpeg_error_mgr pub;    // "public" fields
	jmp_buf setjmp_buffer;        // for return to caller
};
typedef struct my_error_mgr * my_error_ptr;


inline
METHODDEF(void)
my_error_exit (j_common_ptr cinfo)
{
	// cinfo->err really points to a my_error_mgr struct, so coerce pointer
	my_error_ptr myerr = (my_error_ptr) cinfo->err;

	// Always display the message.
	// We could postpone this until after returning, if we chose.
	(*cinfo->err->output_message) (cinfo);

	// Return control to the setjmp point
	longjmp(myerr->setjmp_buffer, 1);
}

inline void JPEGWriteColor( const char* fileName , const unsigned char* pixels , int width , int height , int quality )
{
	FILE* fp;
	struct jpeg_compress_struct cInfo;
	struct my_error_mgr jErr;

	fp = fopen( fileName , "wb" );
	if( !fp ) fprintf( stderr , "[ERROR] Failed to open: %s\n" , fileName ) , exit(0);

	cInfo.err = jpeg_std_error( &jErr.pub );
	jpeg_create_compress( &cInfo );

	jpeg_stdio_dest( &cInfo , fp );

	cInfo.image_width = width;			/* image width and height, in pixels */
	cInfo.image_height = height;
	cInfo.input_components = 3;			/* # of color components per pixel */
	cInfo.in_color_space = JCS_RGB;		/* colorspace of input image */

	jpeg_set_defaults( &cInfo );
	jpeg_set_quality( &cInfo , quality , TRUE );

	jpeg_start_compress( &cInfo , TRUE );

	for( int j=0 ; j<height ; j++ )
	{
		JSAMPROW row_pointer[1];
		row_pointer[0] = (unsigned char*)( pixels + j * width * 3 );
		(void) jpeg_write_scanlines( &cInfo , row_pointer , 1 );
	}
	jpeg_finish_compress( &cInfo );
	jpeg_destroy_compress( &cInfo );
	fclose( fp );
}
inline void JPEGWriteGray( const char* fileName , const unsigned char* pixels , int width , int height , int quality )
{
	FILE* fp;
	struct jpeg_compress_struct cInfo;
	struct my_error_mgr jErr;

	fp = fopen( fileName , "wb" );
	if( !fp ) fprintf( stderr , "[ERROR] Failed to open: %s\n" , fileName ) , exit(0);

	cInfo.err = jpeg_std_error( &jErr.pub );
	jpeg_create_compress( &cInfo );

	jpeg_stdio_dest( &cInfo , fp );

	cInfo.image_width = width;			/* image width and height, in pixels */
	cInfo.image_height = height;
	cInfo.input_components = 1;			/* # of color components per pixel */
	cInfo.in_color_space = JCS_GRAYSCALE;		/* colorspace of input image */

	jpeg_set_defaults( &cInfo );
	jpeg_set_quality( &cInfo , quality , TRUE );

	jpeg_start_compress( &cInfo , TRUE );

	unsigned char* _pixels = new unsigned char[ width];
	for( int j=0 ; j<height ; j++ )
	{
		for( int i=0 ; i<width ; i++ ) _pixels[i] = pixels[j*width+i];
		JSAMPROW row_pointer[1];
		row_pointer[0] = _pixels;
		(void) jpeg_write_scanlines( &cInfo , row_pointer , 1 );
	}
	delete[] _pixels;
	jpeg_finish_compress( &cInfo );
	jpeg_destroy_compress( &cInfo );
	fclose( fp );
}

inline bool JPEGReadColor( const char* fileName , unsigned char* pixels , int width , int height )
{
	FILE* fp;
	struct jpeg_decompress_struct cInfo;
	struct my_error_mgr jErr;

	fp = fopen( fileName , "rb" );
	if( !fp ) fprintf( stderr , "[ERROR] Failed to open: %s\n" , fileName ) , exit(0);

	cInfo.err = jpeg_std_error( &jErr.pub );
	jErr.pub.error_exit = my_error_exit;
	if( setjmp( jErr.setjmp_buffer ) )
	{
		jpeg_destroy_decompress( &cInfo );
		fprintf( stderr , "[ERROR] JPEG error occured\n" );
		exit( 0 );
	}

	jpeg_create_decompress( &cInfo );
	jpeg_stdio_src( &cInfo , fp );

	(void) jpeg_read_header( &cInfo , TRUE );
	(void) jpeg_start_decompress( &cInfo );

	if( cInfo.output_components!=3 )
	{
		fprintf( stderr , "[ERROR] Only 3 components per pixel supported: %d != 3\n" , cInfo.output_components );
		exit( 0 );
	}
	if( width!=cInfo.output_width || height!= cInfo.output_height )
	{
		fprintf( stderr , "[ERROR] JPEG dimensions don't match: %d x %d != %d x %d\n" , width , height , cInfo.output_width , cInfo.output_height );
		exit( 0 );
	}
	for( int j=0 ; j<height ; j++ )
	{
		JSAMPROW row_pointers[1];
		row_pointers[0] = pixels + j * width * 3;
		jpeg_read_scanlines( &cInfo , row_pointers, 1 );
	}
	(void) jpeg_finish_decompress( &cInfo );
	jpeg_destroy_decompress( &cInfo );
	fclose( fp );
	return true;
}
inline bool JPEGReadGray( const char* fileName , unsigned char* pixels , int width , int height )
{
	FILE* fp;
	struct jpeg_decompress_struct cInfo;
	struct my_error_mgr jErr;

	fp = fopen( fileName , "rb" );
	if( !fp ) fprintf( stderr , "[ERROR] Failed to open: %s\n" , fileName ) , exit(0);

	cInfo.err = jpeg_std_error( &jErr.pub );
	jErr.pub.error_exit = my_error_exit;
	if( setjmp( jErr.setjmp_buffer ) )
	{
		jpeg_destroy_decompress( &cInfo );
		fprintf( stderr , "[ERROR] JPEG error occured\n" );
		exit( 0 );
	}

	jpeg_create_decompress( &cInfo );
	jpeg_stdio_src( &cInfo , fp );

	(void) jpeg_read_header( &cInfo , TRUE );
	(void) jpeg_start_decompress( &cInfo );

	if( cInfo.output_components!=3 )
	{
		fprintf( stderr , "[ERROR] Only 3 components per pixel supported: %d != 3\n" , cInfo.output_components );
		exit( 0 );
	}
	if( width!=cInfo.output_width || height!= cInfo.output_height )
	{
		fprintf( stderr , "[ERROR] JPEG dimensions don't match: %d x %d != %d x %d\n" , width , height , cInfo.output_width , cInfo.output_height );
		exit( 0 );
	}
	unsigned char* _pixels = new unsigned char[ width*3 ];
	for( int j=0 ; j<height ; j++ )
	{
		JSAMPROW row_pointers[1];
		row_pointers[0] = _pixels;
		jpeg_read_scanlines( &cInfo , row_pointers, 1 );
		for( int i=0 ; i<width ; i++ )
		{
			unsigned int sum = (unsigned int)_pixels[i*3] + (unsigned int)_pixels[i*3+1] + (unsigned int)_pixels[i*3+2];
			pixels[j*width+i] = (unsigned char)( (sum+1)/3 );
		}
	}
	delete[] _pixels;
	(void) jpeg_finish_decompress( &cInfo );
	jpeg_destroy_decompress( &cInfo );
	fclose( fp );
	return true;
}
inline unsigned char* JPEGReadColor( const char* fileName , int& width , int& height )
{
	FILE* fp;
	struct jpeg_decompress_struct cInfo;
	struct my_error_mgr jErr;

	fp = fopen( fileName , "rb" );
	if( !fp ) fprintf( stderr , "[ERROR] Failed to open: %s\n" , fileName ) , exit(0);

	cInfo.err = jpeg_std_error( &jErr.pub );
	jErr.pub.error_exit = my_error_exit;
	if( setjmp( jErr.setjmp_buffer ) )
	{
		jpeg_destroy_decompress( &cInfo );
		fprintf( stderr , "[ERROR] JPEG error occured\n" );
		exit( 0 );
	}

	jpeg_create_decompress( &cInfo );
	jpeg_stdio_src( &cInfo , fp );

	(void) jpeg_read_header( &cInfo , TRUE );
	(void) jpeg_start_decompress( &cInfo );

	if( cInfo.output_components!=3 )
	{
		fprintf( stderr , "[ERROR] Only 3 components per pixel supported: %d != 3\n" , cInfo.output_components );
		exit( 0 );
	}
	width  = cInfo.output_width;
	height = cInfo.output_height;

	unsigned char* pixels = (unsigned char*) malloc( sizeof( unsigned char ) * width * height * 3 );
	if( !pixels )
	{
		fprintf( stderr , "[ERROR] Failed to allocate pixels: %d x %d\n" , width , height );
		exit( 0 );
	}

	for( int j=0 ; j<height ; j++ )
	{
		JSAMPROW row_pointers[1];
		row_pointers[0] = pixels + j * width * 3;
		jpeg_read_scanlines( &cInfo , row_pointers, 1 );
	}
	(void) jpeg_finish_decompress( &cInfo );
	jpeg_destroy_decompress( &cInfo );
	fclose( fp );
	return pixels;
}
inline unsigned char* JPEGReadGray( const char* fileName , int& width , int& height )
{
	FILE* fp;
	struct jpeg_decompress_struct cInfo;
	struct my_error_mgr jErr;

	fp = fopen( fileName , "rb" );
	if( !fp ) fprintf( stderr , "[ERROR] Failed to open: %s\n" , fileName ) , exit(0);

	cInfo.err = jpeg_std_error( &jErr.pub );
	jErr.pub.error_exit = my_error_exit;
	if( setjmp( jErr.setjmp_buffer ) )
	{
		jpeg_destroy_decompress( &cInfo );
		fprintf( stderr , "[ERROR] JPEG error occured\n" );
		exit( 0 );
	}

	jpeg_create_decompress( &cInfo );
	jpeg_stdio_src( &cInfo , fp );

	(void) jpeg_read_header( &cInfo , TRUE );
	(void) jpeg_start_decompress( &cInfo );

	if( cInfo.output_components!=3 )
	{
		fprintf( stderr , "[ERROR] Only 3 components per pixel supported: %d != 3\n" , cInfo.output_components );
		exit( 0 );
	}
	width  = cInfo.output_width;
	height = cInfo.output_height;

	unsigned char* pixels = (unsigned char*) malloc( sizeof( unsigned char ) * width * height );
	if( !pixels )
	{
		fprintf( stderr , "[ERROR] Failed to allocate pixels: %d x %d\n" , width , height );
		exit( 0 );
	}

	unsigned char* _pixels = new unsigned char[ width*3 ];
	for( int j=0 ; j<height ; j++ )
	{
		JSAMPROW row_pointers[1];
		row_pointers[0] = _pixels;
		jpeg_read_scanlines( &cInfo , row_pointers, 1 );
		for( int i=0 ; i<width ; i++ )
		{
			unsigned int sum = (unsigned int)_pixels[i*3] + (unsigned int)_pixels[i*3+1] + (unsigned int)_pixels[i*3+2];
			pixels[j*width+i] = (unsigned char)( (sum+1)/3 );
		}
	}
	delete[] _pixels;
	(void) jpeg_finish_decompress( &cInfo );
	jpeg_destroy_decompress( &cInfo );
	fclose( fp );
	return pixels;
}
inline bool JPEGGetImageSize( const char* fileName , int& width , int& height , int& channelsPerPixel , int& bitsPerChannel )
{
	bitsPerChannel = 8;
	FILE* fp;
	struct jpeg_decompress_struct cInfo;
	struct my_error_mgr	jErr;

	fp = fopen( fileName , "rb" );
	if(!fp)	fprintf(stderr,"Failed to open: %s\n",fileName) , exit(0);

	cInfo.err = jpeg_std_error( &jErr.pub );
	jErr.pub.error_exit = my_error_exit;
	if( setjmp( jErr.setjmp_buffer ) )
	{
		jpeg_destroy_decompress( &cInfo );
		fprintf( stderr , "JPEG error occured\n" );
		return false;
	}
	jpeg_create_decompress( &cInfo );
	jpeg_stdio_src( &cInfo, fp );	
	(void) jpeg_read_header( &cInfo , TRUE );
	width  = cInfo.image_width;
	height = cInfo.image_height;
	channelsPerPixel = cInfo.output_components;

	jpeg_destroy_decompress( &cInfo );
	fclose( fp );
	return true;
}
inline bool JPEGGetImageSize( const char* fileName , int& width , int& height ){ int channelsPerPixel , bitsPerChannel ; return JPEGGetImageSize( fileName , width , height , channelsPerPixel , bitsPerChannel ); }
