#include <Util/Timer.h>
#include <Util/Visualization.h>

#ifndef M_PI
#define M_PI		3.14159265358979323846
#endif // M_PI

struct SurfaceVisualization : public Visualization
{
	std::vector< TriangleIndex > triangles;
	std::vector< Point3D< float > > vertices , colors;
	Camera camera;
	float zoom;
	Point3D< float > translate;
	float scale;
	GLuint vbo , ebo;
	GLfloat lightAmbient[4] , lightDiffuse[4] , lightSpecular[4] , shapeSpecular[4] , shapeSpecularShininess;
	int oldX , oldY , newX , newY;
	float imageZoom , imageOffset[2];
	bool rotating , scaling , panning;
	bool useLight , showEdges , showColor , hasColor;

	SurfaceVisualization( void );
	bool init( const char* fileName , int subdivide=0 , bool ignoreColor=false );
	void initMesh( void );
	void updateMesh( bool newPositions );
	bool select( int x , int y , Point3D< float >& p );

	void idle( void );
	void keyboardFunc( unsigned char key , int x , int y );
	void specialFunc( int key, int x, int y );
	void initMesh( int resX , int resY );
	void display( void );
	void mouseFunc( int button , int state , int x , int y );
	void motionFunc( int x , int y );

	static void     ToggleLightCallBack( Visualization* v , const char* ){ ( (SurfaceVisualization*)v)->useLight    = !( (SurfaceVisualization*)v)->useLight;    }
	static void     ToggleEdgesCallBack( Visualization* v , const char* ){ ( (SurfaceVisualization*)v)->showEdges   = !( (SurfaceVisualization*)v)->showEdges;   }
	static void     ToggleColorCallBack( Visualization* v , const char* ){ ( (SurfaceVisualization*)v)->showColor   = !( (SurfaceVisualization*)v)->showColor;   }
	static void      OutputMeshCallBack( Visualization* v , const char* prompt )
	{
		const SurfaceVisualization* sv = (SurfaceVisualization*)v;
		if( sv->hasColor )
		{
			std::vector< PlyColorVertex< float > > vertices( sv->vertices.size() );
			for( int i=0 ; i<sv->vertices.size() ; i++ ) vertices[i].point = sv->vertices[i] / sv->scale - sv->translate , vertices[i].color = sv->colors[i] * 255.f;
			PlyWriteTriangles( prompt , vertices , sv->triangles , PlyColorVertex< float >::WriteProperties , PlyColorVertex< float >::WriteComponents , PLY_BINARY_NATIVE );
		}
		else
		{
			std::vector< PlyVertex< float > > vertices( sv->vertices.size() );
			for( int i=0 ; i<sv->vertices.size() ; i++ ) vertices[i].point = sv->vertices[i] / sv->scale - sv->translate;
			PlyWriteTriangles( prompt , vertices , sv->triangles , PlyVertex< float >::WriteProperties , PlyVertex< float >::WriteComponents , PLY_BINARY_NATIVE );
		}
	}
	bool setPosition( int x , int y , Point3D< double >& p );
	bool setPosition( int x , int y , Point3D< float >& p );
};
SurfaceVisualization::SurfaceVisualization( void )
{
	zoom = 1.05f;
	lightAmbient [0] = lightAmbient [1] = lightAmbient [2] = 0.25f , lightAmbient [3] = 1.f;
	lightDiffuse [0] = lightDiffuse [1] = lightDiffuse [2] = 0.70f , lightDiffuse [3] = 1.f;
	lightSpecular[0] = lightSpecular[1] = lightSpecular[2] = 1.00f , lightSpecular[3] = 1.f;
	shapeSpecular[0] = shapeSpecular[1] = shapeSpecular[2] = 1.00f , shapeSpecular[3] = 1.f;
	shapeSpecularShininess = 128;
	oldX , oldY , newX , newY;
	imageZoom = 1.f , imageOffset[0] = imageOffset[1] = 0.f;
	rotating = scaling = panning = false;
	useLight = true;
	showEdges = showColor = false;
	vbo = ebo = 0;

	callBacks.push_back( KeyboardCallBack( this , 'l' , "toggle light" , ToggleLightCallBack ) );
	callBacks.push_back( KeyboardCallBack( this , 'c' , "toggle color" , ToggleColorCallBack ) );
	callBacks.push_back( KeyboardCallBack( this , 'e' , "toggle edges" , ToggleEdgesCallBack ) );
	callBacks.push_back( KeyboardCallBack( this , 'o' , "output to file" , "File Name" , OutputMeshCallBack ) );
}
bool SurfaceVisualization::setPosition( int x , int y , Point3D< double >& p )
{
	double _x =(double)x / screenWidth - 0.5 , _y = 1. - (double)y/screenHeight - 0.5;
	_x *= 2. , _y *= 2;
	_x *= zoom , _y *= zoom;
	double r = _x*_x + _y*_y;
	if( r<1 )
	{
		p = camera.forward * ( -sqrt( 1-r ) ) + camera.right * _x + camera.up * _y;
		return true;
	}
	return false;
}
bool SurfaceVisualization::setPosition( int x , int y , Point3D< float >& p )
{
	Point3D< double > _p;
	bool ret = setPosition( x , y , _p );
	p = Point3D< float >( (float)_p[0] , (float)_p[1] , (float)_p[2] );
	return ret;
}

bool SurfaceVisualization::init( const char* fileName , int subdivide , bool ignoreColor )
{
	int file_type;
	std::vector< PlyColorVertex< float > > ply_vertices;
	bool readFlags[ PlyColorVertex< float >::ReadComponents ];
	PlyReadTriangles( fileName , ply_vertices , triangles , PlyColorVertex< float >::ReadProperties , readFlags , PlyColorVertex< float >::ReadComponents , file_type );
	hasColor = ( readFlags[3] && readFlags[4] && readFlags[5] ) || ( readFlags[6] && readFlags[7] && readFlags[8] );
	hasColor &= !ignoreColor;
	showColor = hasColor;
	vertices.resize( ply_vertices.size() );
	for( int i=0 ; i<ply_vertices.size() ; i++ ) vertices[i] = ply_vertices[i].point;

	for( int i=0 ; i<subdivide ; i++ )
	{
		FEM::Mesh inMesh , outMesh;
		inMesh.triangles = &triangles[0] , inMesh.tCount = triangles.size();
		Point3D< float >* tVertices;
		int vCount = FEM::Mesh::LoopSubdivide( inMesh , &vertices[0] , outMesh , &tVertices );
		vertices.resize( vCount );
		for( int i=0 ; i<vCount ; i++ ) vertices[i] = tVertices[i];
		delete[] tVertices;
		triangles.resize( outMesh.tCount );
		for( int i=0 ; i<outMesh.tCount ; i++ ) triangles[i] = outMesh.triangles[i];
		delete[] outMesh.triangles;
	}
	colors.resize( vertices.size() );
	if( hasColor ) for( int i=0 ; i<vertices.size() ; i++ ) colors[i] = ply_vertices[i].color / 255.f;
	else           for( int i=0 ; i<vertices.size() ; i++ ) colors[i] = Point3D< float >( 0.5f , 0.5f , 0.5f );
	scale = 1.f;
	translate = Point3D< float >( 0.f , 0.f , 0.f );
	return hasColor;
}
void SurfaceVisualization::updateMesh( bool newPositions )
{
	Point3D< float > *_vertices = new Point3D< float >[ triangles.size()*9 ];
	Point3D< float > *_normals = _vertices + triangles.size()*3;
	Point3D< float > *_colors = _vertices + triangles.size()*6;

	Point3D< float > center;
	float area = 0.f;
	for( int i=0 ; i<triangles.size() ; i++ )
	{
		Point3D< float > n = Point3D< float >::CrossProduct( vertices[ triangles[i][1] ] - vertices[ triangles[i][0] ] , vertices[ triangles[i][2] ] - vertices[ triangles[i][0] ] );
		Point3D< float > c = ( vertices[ triangles[i][0] ] + vertices[ triangles[i][1] ] + vertices[ triangles[i][2] ] ) / 3.f;
		float a = (float)Length(n);
		center += c*a , area += a;
	}
	center /= area;
	float max = 0.f;
	for( int i=0 ; i<vertices.size() ; i++ ) max = std::max< float >( max , (float)Point3D< float >::Length( vertices[i]-center ) );

	for( int i=0 ; i<triangles.size() ; i++ )
	{
		Point3D< float > n = Point3D< float >::CrossProduct( vertices[ triangles[i][1] ] - vertices[ triangles[i][0] ] , vertices[ triangles[i][2] ] - vertices[ triangles[i][0] ] );
		n /= Length( n );
		for( int j=0 ; j<3 ; j++ )
		{
			_vertices[3*i+j] = ( vertices[ triangles[i][j] ] - center ) / max;
			_colors[3*i+j] = colors[ triangles[i][j] ];
			_normals[3*i+j] = n;
		}
	}
	if( newPositions )
	{
		translate = -center;
		scale = 1.f/max;
		for( int i=0 ; i<vertices.size() ; i++ ) vertices[i] = ( vertices[i]+translate ) * scale;
	}
	glBindBuffer( GL_ARRAY_BUFFER , vbo );
	glBufferData( GL_ARRAY_BUFFER , 9 * triangles.size() * sizeof( Point3D< float > ) , _vertices , GL_DYNAMIC_DRAW );
	glBindBuffer( GL_ARRAY_BUFFER , 0 );

	delete[] _vertices;
	glutPostRedisplay();
}
void SurfaceVisualization::initMesh( void )
{
	TriangleIndex *_triangles = new TriangleIndex[ triangles.size() ];

	for( int i=0 ; i<triangles.size() ; i++ ) for( int j=0 ; j<3 ; j++ ) _triangles[i][j] = 3*i+j;

	glGenBuffers( 1 , &ebo );
	glBindBuffer( GL_ELEMENT_ARRAY_BUFFER , ebo );
	glBufferData( GL_ELEMENT_ARRAY_BUFFER , triangles.size() * sizeof( int ) * 3 , _triangles , GL_STATIC_DRAW );
	glBindBuffer( GL_ELEMENT_ARRAY_BUFFER , 0 );

	glGenBuffers( 1 , &vbo );
	updateMesh( true );

	delete[] _triangles;
}
bool SurfaceVisualization::select( int x , int  y , Point3D< float >& out )
{
	bool ret = false;
	Pointer( float ) depthBuffer = AllocPointer< float >( sizeof(float) * screenWidth * screenHeight );
	glReadPixels( 0 , 0 , screenWidth , screenHeight , GL_DEPTH_COMPONENT , GL_FLOAT , depthBuffer );
	float ar = (float)screenWidth/(float)screenHeight ;
	float _screenWidth , _screenHeight;
	if( screenWidth>screenHeight ) _screenWidth =  screenWidth * ar , _screenHeight = screenHeight;
	else                           _screenWidth =  screenWidth , _screenHeight = screenHeight / ar;
	{
		double _x =(double)x/screenWidth - 0.5 , _y = 1. - (double)y/screenHeight - 0.5 , _z;
		if( screenWidth>screenHeight ) _x *= zoom*ar , _y *= zoom;
		else                           _x *= zoom , _y *= zoom/ar;
		_x *= 2. , _y *= 2;
		int x1 = (int)floor(x) , y1 = (int)floor(y) , x2 = x1+1 , y2 = y1+1;
		float dx = x-x1 , dy = y-y1;
		x1 = std::max< int >( 0.f , std::min< int >( x1 , screenWidth -1 ) );
		y1 = std::max< int >( 0.f , std::min< int >( y1 , screenHeight-1 ) );
		x2 = std::max< int >( 0.f , std::min< int >( x2 , screenWidth -1 ) );
		y2 = std::max< int >( 0.f , std::min< int >( y2 , screenHeight-1 ) );
		_z =
			depthBuffer[ (screenHeight-1-y1)*screenWidth+x1 ] * (1.f-dx) * (1.f-dy) +
			depthBuffer[ (screenHeight-1-y1)*screenWidth+x2 ] * (    dx) * (1.f-dy) +
			depthBuffer[ (screenHeight-1-y2)*screenWidth+x1 ] * (1.f-dx) * (    dy) +
			depthBuffer[ (screenHeight-1-y2)*screenWidth+x2 ] * (    dx) * (    dy) ;
		if( _z<1 ) out = Point3D< float >( camera.forward * ( -1.5 + 3. * _z ) + camera.right * _x + camera.up * _y + camera.position ) , ret = true;
	}
	FreePointer( depthBuffer );
	return ret;
}
void SurfaceVisualization::display( void )
{
	if( !vbo && !ebo ) initMesh();
	glEnable( GL_CULL_FACE );

	glMatrixMode( GL_PROJECTION );
	glLoadIdentity();
	float ar = (float)screenWidth/(float)screenHeight , ar_r = 1.f/ar;
	if( screenWidth>screenHeight ) glOrtho( -ar*zoom , ar*zoom , -zoom , zoom , -1.5 , 1.5 );
	else                           glOrtho( -zoom , zoom , -ar_r*zoom , ar_r*zoom , -1.5 , 1.5 );
	glMatrixMode( GL_MODELVIEW );

	glLoadIdentity();
	camera.draw();

	GLfloat lPosition[4];

	{
		Point3D< float > d = camera.up + camera.right - camera.forward*5;
		lPosition[0] = d[0] , lPosition[1] = d[1] , lPosition[2] = d[2];
	}
	lPosition[3] = 0.0;	
	glLightModeli( GL_LIGHT_MODEL_LOCAL_VIEWER , GL_FALSE );
	glLightModeli( GL_LIGHT_MODEL_TWO_SIDE , GL_TRUE );
	glLightfv( GL_LIGHT0 , GL_AMBIENT , lightAmbient );
	glLightfv( GL_LIGHT0 , GL_DIFFUSE , lightDiffuse );
	glLightfv( GL_LIGHT0 , GL_SPECULAR , lightSpecular );
	glLightfv( GL_LIGHT0 , GL_POSITION , lPosition );
	glEnable( GL_LIGHT0 );
	if( useLight ) glEnable ( GL_LIGHTING );
	else           glDisable( GL_LIGHTING );
	glColorMaterial( GL_FRONT_AND_BACK , GL_AMBIENT_AND_DIFFUSE );
	glEnable( GL_COLOR_MATERIAL );

	glEnable( GL_DEPTH_TEST );
	glMaterialfv( GL_FRONT_AND_BACK , GL_SPECULAR  , shapeSpecular );
	glMaterialf ( GL_FRONT_AND_BACK , GL_SHININESS , shapeSpecularShininess );

	glBindBuffer( GL_ARRAY_BUFFER , vbo );
	glBindBuffer( GL_ELEMENT_ARRAY_BUFFER , ebo );
	glEnableClientState( GL_VERTEX_ARRAY );
	glVertexPointer( 3 , GL_FLOAT , 0 , NULL );
	glNormalPointer(     GL_FLOAT , 0 , (GLubyte*)NULL + sizeof( Point3D< float > ) * triangles.size()*3 );
	glColorPointer ( 3 , GL_FLOAT , 0 , (GLubyte*)NULL + sizeof( Point3D< float > ) * triangles.size()*6 );
	glColor3f( 0.75f , 0.75f , 0.75f );
	glEnableClientState( GL_NORMAL_ARRAY );
	if( showColor ) glEnableClientState( GL_COLOR_ARRAY );
	glDrawElements( GL_TRIANGLES , (GLsizei)(triangles.size()*3) , GL_UNSIGNED_INT , NULL );
	glDisableClientState( GL_NORMAL_ARRAY );
	if( showColor ) glDisableClientState( GL_COLOR_ARRAY );
	glBindBuffer( GL_ARRAY_BUFFER , 0 );
	glBindBuffer( GL_ELEMENT_ARRAY_BUFFER , 0 );

	if( showEdges )
	{
		GLint src , dst;
		glGetIntegerv( GL_BLEND_SRC , &src );
		glGetIntegerv( GL_BLEND_DST , &dst );
		Point3D< float > f = camera.forward / 256;
		glPushMatrix();
		glTranslatef( -f[0] , -f[1] , -f[2] );
		glColor3f( 0.125 , 0.125 , 0.125 );
		glBlendFunc( GL_SRC_ALPHA , GL_ONE_MINUS_SRC_ALPHA );
		glEnable( GL_BLEND );
		glEnable( GL_LINE_SMOOTH );
		glLineWidth( 0.25f );
		glPolygonMode( GL_FRONT_AND_BACK , GL_LINE );
		glBindBuffer( GL_ARRAY_BUFFER , vbo );
		glBindBuffer( GL_ELEMENT_ARRAY_BUFFER , ebo );
		glEnableClientState( GL_VERTEX_ARRAY );
		glVertexPointer( 3 , GL_FLOAT , 0 , NULL );
		glDrawElements( GL_TRIANGLES , (GLsizei)(triangles.size()*3) , GL_UNSIGNED_INT , NULL );
		glBindBuffer( GL_ARRAY_BUFFER , 0 );
		glBindBuffer( GL_ELEMENT_ARRAY_BUFFER , 0 );
		glPolygonMode( GL_FRONT_AND_BACK , GL_FILL );
		glDisable( GL_LINE_SMOOTH );
		glPopMatrix();
		glDisable( GL_BLEND );
		glBlendFunc( src , dst );
	}
}
void SurfaceVisualization::mouseFunc( int button , int state , int x , int y )
{
	newX = x ; newY = y;

	rotating = scaling = panning = false;
	if( button==GLUT_LEFT_BUTTON  )
		if( glutGetModifiers() & GLUT_ACTIVE_CTRL ) panning = true;
		else                                        rotating = true;
	else if( button==GLUT_RIGHT_BUTTON ) scaling = true;
}
void SurfaceVisualization::motionFunc( int x , int y )
{
	oldX = newX , oldY = newY , newX = x , newY = y;

	int imageSize = std::min< int >( screenWidth , screenHeight );
	float rel_x = (newX - oldX) / (float)imageSize * 2;
	float rel_y = (newY - oldY) / (float)imageSize * 2;
	float pRight = -rel_x * zoom , pUp = rel_y * zoom;
	float sForward = rel_y*4;
	float rRight = rel_y , rUp = rel_x;

	if     ( rotating ) camera.rotateUp( rUp ) , camera.rotateRight( rRight );
	else if( scaling  ) zoom *= (float)pow( 0.9 , (double)sForward );
	else if( panning  ) camera.translate( camera.right * pRight + camera.up * pUp );

	glutPostRedisplay();
}

void SurfaceVisualization::idle( void ){ if( !promptCallBack ){ ; } }

void SurfaceVisualization::keyboardFunc( unsigned char key , int x , int y )
{
	switch( key )
	{
	case 'q': camera.rotateUp( -M_PI/128 ) ; break;
	case 'Q': camera.rotateUp( -M_PI/  4 ) ; break;
	case 'w': camera.rotateUp(  M_PI/128 ) ; break;
	case 'W': camera.rotateUp(  M_PI/  4 ) ; break;
	case 'a': camera.rotateRight( -M_PI/128 ) ; break;
	case 'A': camera.rotateRight( -M_PI/  4 ) ; break;
	case 'z': camera.rotateRight(  M_PI/128 ) ; break;
	case 'Z': camera.rotateRight(  M_PI/  4 ) ; break;
	case 's': camera.rotateForward( -M_PI/128 ) ; break;
	case 'S': camera.rotateForward( -M_PI/  4 ) ; break;
	case 'x': camera.rotateForward(  M_PI/128 ) ; break;
	case 'X': camera.rotateForward(  M_PI/  4 ) ; break;
	}
}

void SurfaceVisualization::specialFunc( int key, int x, int y )
{
	float stepSize = 10.f / ( screenWidth + screenHeight );
	if( glutGetModifiers()&GLUT_ACTIVE_CTRL ) stepSize /= 16;
	float panSize = stepSize*2 , scaleSize = stepSize*2;

	switch( key )
	{
	case KEY_UPARROW:    zoom *= 0.98f ; break;
	case KEY_DOWNARROW:  zoom /= 0.98f ; break;
	case KEY_LEFTARROW:  camera.translate(  camera.right * panSize ) ; break;
	case KEY_RIGHTARROW: camera.translate( -camera.right * panSize ) ; break;
	case KEY_PGUP:       camera.translate( -camera.up    * panSize ) ; break;
	case KEY_PGDN:       camera.translate(  camera.up    * panSize ) ; break;
	}
	glutPostRedisplay();
}