#include <Util/LinearSolvers.h>
#include "SurfaceVisualization.inl"

template< class Real >
struct SurfaceSharpeningViewer
{
	static SurfaceVisualization sv;
	static FEM::RiemannianMesh< Real > rMesh;
	static std::vector< Point2D< Real > > flow;
	static Real screenWeight;
	static int cgIterations;
	static int smoothIters;
	static Real flowTime;
	static bool lump;
	static bool verbose;
	static int threads;
	static std::pair< Real , Point3D< float > > selectPoint;
	static int whichMesh;

	static void Init( const char* name , int subdivide , bool ignoreColor , int preSmoothIters );
	static void Idle        ( void );
	static void KeyboardFunc( unsigned char key , int x , int y );
	static void SpecialFunc ( int key, int x, int y );
	static void Display     ( void );
	static void Reshape     ( int w , int h );
	static void MouseFunc   ( int button , int state , int x , int y );
	static void MotionFunc  ( int x , int y );

	static void SetScreen( void );
	static void SetFlow( void );
	static void AdvectSignal( void );
	static void FitToNormals( void );

	static void UpdateFlowTime( void ){ AdvectSignal() , FitToNormals(); }
	static void UpdateFlow( void ){ SetFlow() , UpdateFlowTime(); }
	static void UpdateScreen( void ){ SetScreen() , UpdateFlowTime(); }

	static void SetCGIterationsCallBack( Visualization* v , const char* prompt );
	static void SetVideoCallBack( Visualization* v , const char* prompt );
	static void SetSmoothCallBack( Visualization* v , const char* prompt );
	static void SetFlowTimeCallBack( Visualization* v , const char* prompt );
	static void SetScreenWeightCallBack( Visualization* v , const char* prompt );
	static void ToggleMeshCallBack( Visualization* v , const char* );
protected:
	static Real _videoStart , _videoEnd;
	static int _videoFrame , _videoFrames;
	static bool _hasColor;

	static PreconditionedCGScratch< Real > _cgScratch;
	static SparseMatrix< Real , int > *_fittingM;
	static DiagonalPreconditioner< Real > _fittingPreconditioner;
#if USE_EIGEN
	static EigenCholeskySolver* _eigenCholeskyFitter;
#endif // USE_EIGEN

	static FEM::Mesh::Edge< Real >* _edgeXForms;
	static std::vector< Real > _oldSignal[3] , _newSignal[3];
	static std::vector< int > _valence;
	static std::vector< Point3D< float > > _oldVertices , _newVertices;
	static int _busyCursor;
	static void _SetFittingSolver( Real screen );
	static Real _averageEdgeLength;
};

template< class Real > SurfaceVisualization				SurfaceSharpeningViewer< Real >::sv;
template< class Real > FEM::RiemannianMesh< Real >		SurfaceSharpeningViewer< Real >::rMesh;
template< class Real > std::vector< Point2D< Real > >	SurfaceSharpeningViewer< Real >::flow;
template< class Real > bool								SurfaceSharpeningViewer< Real >::lump = false;
template< class Real > bool								SurfaceSharpeningViewer< Real >::verbose = false;
template< class Real > bool								SurfaceSharpeningViewer< Real >::_hasColor = false;
template< class Real > int								SurfaceSharpeningViewer< Real >::cgIterations = 0;
template< class Real > int								SurfaceSharpeningViewer< Real >::smoothIters = 0;
template< class Real > int								SurfaceSharpeningViewer< Real >::threads = 1;
template< class Real > Real								SurfaceSharpeningViewer< Real >::flowTime = (Real)0;
template< class Real > Real								SurfaceSharpeningViewer< Real >::screenWeight = (Real)1;
template< class Real > std::pair< Real , Point3D< float > > SurfaceSharpeningViewer< Real >::selectPoint = std::pair< Real , Point3D< float > >( 0 , Point3D< float >() );

template< class Real > Real								SurfaceSharpeningViewer< Real >::_videoStart;
template< class Real > Real								SurfaceSharpeningViewer< Real >::_videoEnd;
template< class Real > int								SurfaceSharpeningViewer< Real >::_videoFrame;
template< class Real > int								SurfaceSharpeningViewer< Real >::_videoFrames = 0;
template< class Real > PreconditionedCGScratch< Real >	SurfaceSharpeningViewer< Real >::_cgScratch;
template< class Real > SparseMatrix< Real , int >*		SurfaceSharpeningViewer< Real >::_fittingM = NULL;
template< class Real > DiagonalPreconditioner< Real >	SurfaceSharpeningViewer< Real >::_fittingPreconditioner;
#if USE_EIGEN
template< class Real > EigenCholeskySolver*				SurfaceSharpeningViewer< Real >::_eigenCholeskyFitter = NULL;
#endif // USE_EIGEN

template< class Real > FEM::Mesh::Edge< Real >*			SurfaceSharpeningViewer< Real >::_edgeXForms = NULL;
template< class Real > std::vector< Real >				SurfaceSharpeningViewer< Real >::_oldSignal[3];
template< class Real > std::vector< Real >				SurfaceSharpeningViewer< Real >::_newSignal[3];
template< class Real > std::vector< int >				SurfaceSharpeningViewer< Real >::_valence;
template< class Real > std::vector< Point3D< float > >	SurfaceSharpeningViewer< Real >::_oldVertices;
template< class Real > std::vector< Point3D< float > >	SurfaceSharpeningViewer< Real >::_newVertices;
template< class Real > int								SurfaceSharpeningViewer< Real >::whichMesh = 1;
template< class Real > int								SurfaceSharpeningViewer< Real >::_busyCursor = GLUT_CURSOR_WAIT;
template< class Real > Real								SurfaceSharpeningViewer< Real >::_averageEdgeLength;


// Input format start:end:frames
template< class Real > void SurfaceSharpeningViewer< Real >::SetVideoCallBack( Visualization* , const char* prompt )
{
	char start[512];
	strcpy( start , prompt );
	_videoFrame = _videoFrames = 0;
	char* end = strstr( start , ":" );
	if( !end ){ fprintf( stderr , "[WARNING] Video format is <start>:<end>:<frames>\n" ) ; return; }
	end[0] = 0 , end++;
	char * frames = strstr( end , ":" );
	if( !frames ){ fprintf( stderr , "[WARNING] Video format is <start>:<end>:<frames>\n" ) ; return; }
	frames[0] = 0 , frames++;
	_videoStart = (float)atof(start);
	_videoEnd = (float)atof(end);
	_videoFrames = atoi(frames);
}
template< class Real > void SurfaceSharpeningViewer< Real >::SetCGIterationsCallBack( Visualization* , const char* prompt )
{
	glutSetCursor( _busyCursor );
	cgIterations = std::max< int >( 0 , atoi( prompt ) );
	UpdateScreen();
	UpdateFlow();
	sv.updateMesh( !_hasColor );
	glutSetCursor( GLUT_CURSOR_INHERIT );
}
template< class Real > void SurfaceSharpeningViewer< Real >::SetSmoothCallBack( Visualization* , const char* prompt )
{
	glutSetCursor( _busyCursor );
	smoothIters = atoi( prompt );
	UpdateFlow();
	sv.updateMesh( !_hasColor );
	glutSetCursor( GLUT_CURSOR_INHERIT );
}
template< class Real > void SurfaceSharpeningViewer< Real >::SetFlowTimeCallBack( Visualization* , const char* prompt )
{
	glutSetCursor( _busyCursor );
	flowTime = (float)atof( prompt );
	UpdateFlowTime();
	sv.updateMesh( !_hasColor );
	glutSetCursor( GLUT_CURSOR_INHERIT );
}
template< class Real > void SurfaceSharpeningViewer< Real >::SetScreenWeightCallBack( Visualization* , const char* prompt )
{
	glutSetCursor( _busyCursor );
	screenWeight = (float)atof( prompt );
	UpdateScreen();
	sv.updateMesh( !_hasColor );
	glutSetCursor( GLUT_CURSOR_INHERIT );
}
template< class Real > void SurfaceSharpeningViewer< Real >::ToggleMeshCallBack( Visualization* , const char* )
{
	glutSetCursor( _busyCursor );
	whichMesh = ( whichMesh + 1 ) % 2;
	if( _hasColor )
	{
		if( whichMesh==0 ) for( int c=0 ; c<3 ; c++ ) for( int i=0 ; i<sv.vertices.size() ; i++ ) sv.colors[i][c] = _oldSignal[c][i];
		else               for( int c=0 ; c<3 ; c++ ) for( int i=0 ; i<sv.vertices.size() ; i++ ) sv.colors[i][c] = _newSignal[c][i];
	}
	else
	{
		if( whichMesh==0 ) sv.vertices = _oldVertices;
		else               sv.vertices = _newVertices;
		if( whichMesh==0 ) for( int c=0 ; c<3 ; c++ ) for( int i=0 ; i<sv.vertices.size() ; i++ ) sv.colors[i][c] = ( _oldSignal[c][i] + 1.f ) / 2.f;
		else               for( int c=0 ; c<3 ; c++ ) for( int i=0 ; i<sv.vertices.size() ; i++ ) sv.colors[i][c] = ( _newSignal[c][i] + 1.f ) / 2.f;
	}
	sv.updateMesh( !_hasColor );
	sprintf( sv.info.back() , "Mesh (%s): %lld / %lld" , whichMesh==0 ? "original" : "filtered" , (unsigned long long)sv.vertices.size() , (unsigned long long)sv.triangles.size() );
	glutSetCursor( GLUT_CURSOR_INHERIT );
}

template< class Real > void SurfaceSharpeningViewer< Real >::Init( const char* name , int subdivide , bool ignoreColor , int preSmoothIters )
{
	_hasColor = sv.init( name , subdivide , ignoreColor );
	_oldVertices = sv.vertices;
	_newVertices = sv.vertices;
	rMesh.triangles = &sv.triangles[0];
	rMesh.tCount = sv.triangles.size();
	rMesh.setInnerProduct( &sv.vertices[0] );
	_valence.resize( sv.vertices.size() );
	for( int i=0 ; i<_valence.size() ; i++ ) _valence[i] = 0;
	for( int i=0 ; i<sv.triangles.size() ; i++ ) for( int j=0 ; j<3 ; j++ ) _valence[ sv.triangles[i][j] ]++;
	flow.resize( rMesh.tCount );

	rMesh.makeUnitArea();
	_cgScratch.resize( rMesh.vCount() );
	_edgeXForms = rMesh.template getEdgeXForms< Point3D< float > , Real >( &sv.vertices[0] );

	// Compute the signal, the average, and the gradients
	if( _hasColor )
	{
		for( int c=0 ; c<3 ; c++ )
		{
			_oldSignal[c].resize( sv.colors.size() ) , _newSignal[c].resize( sv.colors.size() );
			for( int i=0 ; i<sv.colors.size() ; i++ ) _oldSignal[c][i] = (Real)sv.colors[i][c];
		}
	}
	else
	{

		std::vector< Point3D< Real > > normals( rMesh.vCount() );
		for( int i=0 ; i<rMesh.tCount ; i++ )
		{
			Point3D< Real > n = Point3D< Real >( Point3D< Real >::CrossProduct( sv.vertices[ sv.triangles[i][1] ] - sv.vertices[ sv.triangles[i][0] ] , sv.vertices[ sv.triangles[i][2] ] - sv.vertices[ sv.triangles[i][0] ] ) );
			for( int j=0 ; j<3 ; j++ ) normals[ rMesh.triangles[i][j] ] += n;
		}
		for( int i=0 ; i<normals.size() ; i++ )
		{
			Real l = (Real)Length( normals[i] );
			if( l ) normals[i] /= l;
			else fprintf( stderr , "[WARNING] Vanishing normal\n" );
		}
		for( int c=0 ; c<3 ; c++ )
		{
			_oldSignal[c].resize( normals.size() ) , _newSignal[c].resize( normals.size() );
			for( int i=0 ; i<normals.size() ; i++ ) _oldSignal[c][i] = normals[i][c];
			for( int i=0 ; i<sv.vertices.size() ; i++ ) sv.colors[i][c] = ( _oldSignal[c][i] + 1.f ) / 2.f;
		}
	}
	Point2D< Real > edges[] = { Point2D< Real >( -1 , 1 ) , Point2D< Real >( 0 , -1 ) , Point2D< Real >( 1 , 0 )  };
	_averageEdgeLength = (Real)0;
	for( int i=0 ; i<rMesh.tCount ; i++ ) for( int j=0 ; j<3 ; j++ ) _averageEdgeLength += (Real)sqrt( Point2D< Real >::Dot( rMesh.g[i] * edges[j] , edges[j] ) );
	_averageEdgeLength /= rMesh.tCount * 3;


	{
		int tCount = (int)rMesh.tCount , vCount = (int)rMesh.vCount();
		Point2D< Real > edges[] = { Point2D< Real >( 1 , 0 ) , Point2D< Real >( -1 , 1 ) , Point2D< Real >( 0 , -1 ) };
		static std::vector< Real > _values;
		static std::vector< Real > _count;
		_values.resize( vCount ) , _count.resize( vCount );

		for( int c=0 ; c<3 ; c++ ) for( int s=0 ; s<preSmoothIters ; s++ )
		{
			Pointer( Real ) values = &_oldSignal[c][0];
#pragma omp parallel for num_threads( threads )
			for( int i=0 ; i<vCount ; i++ ) _values[i] = values[i] , _count[i] = (Real)1.;
#pragma omp parallel for num_threads( threads )
			for( int i=0 ; i<tCount ; i++ ) for( int j=0 ; j<3 ; j++ )
			{
				Real w = exp( - Point2D< Real >::Dot( edges[j] , rMesh.g[i] * edges[j] ) / _averageEdgeLength * _averageEdgeLength );
#pragma omp atomic
				values[ rMesh.triangles[i][j] ] += _values[ rMesh.triangles[i][(j+1)%3] ] * w;
#pragma omp atomic
				_count[ rMesh.triangles[i][j] ] += w;
			}
#pragma omp parallel for num_threads( threads )
			for( int i=0 ; i<vCount ; i++ ) values[i] /= _count[i];
		}
	}

	sv.callBacks.push_back( Visualization::KeyboardCallBack( &sv , 'V' , "video frame info" , "Start:End:Frames" , SetVideoCallBack ) );
	sv.callBacks.push_back( Visualization::KeyboardCallBack( &sv , 't' , "toggle mesh" , ToggleMeshCallBack ) );
	sv.callBacks.push_back( Visualization::KeyboardCallBack( &sv , '4' , "set cg iterations" , "CG Iterations" , SetCGIterationsCallBack ) );
	sv.callBacks.push_back( Visualization::KeyboardCallBack( &sv , '3' , "set screen weight" , "Screen Weight" , SetScreenWeightCallBack ) );
	sv.callBacks.push_back( Visualization::KeyboardCallBack( &sv , '2' , "set smooth iters" , "Smooth Iterations" , SetSmoothCallBack ) );
	sv.callBacks.push_back( Visualization::KeyboardCallBack( &sv , '1' , "set flow time" , "Flow Time" , SetFlowTimeCallBack ) );

	sv.info.resize( 6 );
	for( int i=0 ; i<sv.info.size() ; i++ ) sv.info[i] = new char[512] , sv.info[i][0] = 0;
}
template< class Real > void SurfaceSharpeningViewer< Real >::_SetFittingSolver( Real screen )
{
	if( _fittingM ) delete _fittingM , _fittingM = NULL;
	if( _hasColor ) return;
#if USE_EIGEN
	if( _eigenCholeskyFitter ) delete _eigenCholeskyFitter , _eigenCholeskyFitter = NULL;
	if( cgIterations>0 )
#endif // USE_EIGEN
	{
		_fittingM = new SparseMatrix< Real , int >();
		rMesh.setBasisSystemMatrix( *_fittingM , screen , 1. , lump );
		_fittingPreconditioner.set( *_fittingM );
	}
#if USE_EIGEN
	else
	{
		SparseMatrix< Real , int > M;
		rMesh.setBasisSystemMatrix( M , screen , 1. , lump );
		_eigenCholeskyFitter = new EigenCholeskySolver( M );
	}
#endif // USE_EIGEN
}
template< class Real > void SurfaceSharpeningViewer< Real >::SetScreen( void )
{
	double t = Time();
	screenWeight = std::max< float >( screenWeight , 0. );
	_SetFittingSolver( screenWeight );
	sprintf( sv.info[4] , "Set screen (cg=%d : weight=%.2e): %.2f(s)\n" , cgIterations , screenWeight , Time()-t );
}
template< class Real > void SurfaceSharpeningViewer< Real >::SetFlow( void )
{
	double t = Time();
	int tCount = (int)rMesh.tCount , vCount = (int)rMesh.vCount();
	std::vector< Real > tPotential( tCount , (Real)0 ) , vPotential( vCount );

	smoothIters = std::max< int >( smoothIters , 0 );

	auto smooth_signal = [&] ( Real* values )
	{
		static std::vector< Real > _values;
		static std::vector< Real > _count;
		_values.resize( vCount ) , _count.resize( vCount );

		Point2D< Real > edges[] = { Point2D< Real >( 1 , 0 ) , Point2D< Real >( -1 , 1 ) , Point2D< Real >( 0 , -1 ) };
		for( int s=0 ; s<smoothIters ; s++ )
		{
#pragma omp parallel for num_threads( threads )
			for( int i=0 ; i<vCount ; i++ ) _values[i] = values[i] , _count[i] = (Real)1.;
#pragma omp parallel for num_threads( threads )
			for( int i=0 ; i<tCount ; i++ ) for( int j=0 ; j<3 ; j++ )
			{
				Real w = exp( - Point2D< Real >::Dot( edges[j] , rMesh.g[i] * edges[j] ) / _averageEdgeLength * _averageEdgeLength );
#pragma omp atomic
				values[ rMesh.triangles[i][j] ] += _values[ rMesh.triangles[i][(j+1)%3] ] * w;
#pragma omp atomic
				_count[ rMesh.triangles[i][j] ] += w;
			}
#pragma omp parallel for num_threads( threads )
			for( int i=0 ; i<vCount ; i++ ) values[i] /= _count[i];
		}
	};
	for( int c=0 ; c<3 ; c++ )
	{
		std::vector< Real > scratchSignal = _oldSignal[c];
		smooth_signal( &scratchSignal[0] );
		rMesh.setGradients( &scratchSignal[0] , &flow[0] );
#pragma omp parallel for num_threads( threads )
		for( int i=0 ; i<tCount ; i++ ) tPotential[i] += Point2D< Real >::Dot( flow[i] , rMesh.g[i] * flow[i] );
	}

	// Prolong the square-norms to the vertices (per-vertex)
	rMesh.setProlongation( &tPotential[0] , &vPotential[0] );

	// Set the mesh color
	if( !_hasColor )
	{
		for( int c=0 ; c<3 ; c++ ) for( int i=0 ; i<sv.vertices.size() ; i++ ) sv.colors[i][c] = ( _oldSignal[c][i] + 1.f ) / 2.f;
	}
	// Smooth the flow again, since we are taking another derivative
	smooth_signal( &vPotential[0] );

	// Compute the gradients of the square-norm (per-triangle)
	rMesh.setGradients( &vPotential[0] , &flow[0] );

	// Normalize the scale of the flow in a somewhat ad-hoc manner
	{
		double scl = 1. / sqrt( 1e8 );
		if( _hasColor ) scl *= 4;
		else            scl /= 128;
#pragma omp parallel for num_threads( threads )
		for( int i=0 ; i<rMesh.tCount ; i++ ) flow[i] *= scl;
	}

	sprintf( sv.info[3] , "Set flow (smooth iters=%d): %.2f(s)\n" , smoothIters , Time()-t );
}

template< class Real > void SurfaceSharpeningViewer< Real >::AdvectSignal( void )
{
	double t = Time();
	double eps = 1e-3;
	int tCount = (int)rMesh.tCount , vCount = (int)rMesh.vCount();
	{
		Real coords[3][2] = { { (Real)eps , (Real)eps } , { (Real)(1.-2.*eps) , (Real)eps } , { (Real)eps , (Real)(1.-2.*eps) } };
		for( int i=0 ; i<vCount ; i++ ) for( int c=0 ; c<3 ; c++ ) _newSignal[c][i] = (Real)0;
#pragma omp parallel for num_threads( threads )
		for( int i=0 ; i<tCount ; i++ ) for( int j=0 ; j<3 ; j++ )
		{
			FEM::Mesh::SamplePoint< Real > p;
			p.tIdx = i , p.p[0] = coords[j][0] , p.p[1] = coords[j][1];
			rMesh.flow_( _edgeXForms , &flow[0] , -(Real)flowTime , p , _averageEdgeLength/(Real)10 , 0 );
			for( int c=0 ; c<3 ; c++ )
			{

				Real oldValue = _oldSignal[c][ rMesh.triangles[p.tIdx][0] ] * (1.-p.p[0]-p.p[1]) + _oldSignal[c][ rMesh.triangles[p.tIdx][1] ] * p.p[0] + _oldSignal[c][ rMesh.triangles[p.tIdx][2] ] * p.p[1];
#pragma omp atomic
				_newSignal[c][ rMesh.triangles[i][j] ] += oldValue;
			}
		}
#pragma omp parallel for num_threads( threads )
			for( int i=0 ; i<vCount ; i++ ) for( int c=0 ; c<3 ; c++ ) _newSignal[c][i] /= (Real)_valence[i];
	}
	sprintf( sv.info[2] , "Advected signal values (%.2e): %.2f(s)\n" , flowTime , Time()-t );
	sprintf( sv.info[1] , "" );
}

template< class Real > void SurfaceSharpeningViewer< Real >::FitToNormals( void )
{
	double t = Time();
	if( _hasColor )
	{
		if( whichMesh==1 ) for( int c=0 ; c<3 ; c++ ) for( int i=0 ; i<sv.vertices.size() ; i++ ) sv.colors[i][c] = _newSignal[c][i];
		sprintf( sv.info[0] , "Fit colors: %.2f(s)\n" , Time()-t );
	}
	else
	{
		if( whichMesh==1 ) for( int c=0 ; c<3 ; c++ ) for( int i=0 ; i<sv.vertices.size() ; i++ ) sv.colors[i][c] = ( _newSignal[c][i] + 1.f ) / 2.f;
		int tCount = (int)rMesh.tCount , vCount = (int)rMesh.vCount();
		std::vector< Point2D< Real > > gradients[3];
		std::vector< Point3D< Real > > normals( tCount );
		for( int i=0 ; i<tCount ; i++ )
		{
			for( int j=0 ; j<3 ; j++ ) normals[i] += Point3D< Real >( _newSignal[0][ sv.triangles[i][j] ] , _newSignal[1][ sv.triangles[i][j] ] , _newSignal[2][ sv.triangles[i][j] ] ); 
			Real l = (Real)Length( normals[i] );
			if( l ) normals[i] /= l;
		}

		for( int c=0 ; c<3 ; c++ )
		{
			std::vector< Real > x( vCount );
			for( int i=0 ; i<vCount ; i++ ) x[i] = (Real)_oldVertices[i][c];
			gradients[c].resize( tCount );
			rMesh.setGradients( &x[0] , &gradients[c][0] );
		}
		for( int i=0 ; i<tCount ; i++ )
		{
			Point3D< Real > g1( gradients[0][i][0] , gradients[1][i][0] , gradients[2][i][0] );
			Point3D< Real > g2( gradients[0][i][1] , gradients[1][i][1] , gradients[2][i][1] );
			Point3D< Real > n = normals[i];
			g1 -= n * Point3D< Real >::Dot( g1 , n );
			g2 -= n * Point3D< Real >::Dot( g2 , n );
			gradients[0][i][0] = g1[0] , gradients[1][i][0] = g1[1] , gradients[2][i][0] = g1[2];
			gradients[0][i][1] = g2[0] , gradients[1][i][1] = g2[1] , gradients[2][i][1] = g2[2];
		}

		std::vector< Real > b( vCount ) , x( vCount );
		for( int c=0 ; c<3 ; c++ )
		{
			for( int i=0 ; i<vCount ; i++ ) x[i] = (Real)_oldVertices[i][c];
			rMesh.setConstraints( &x[0] , &gradients[c][0] , lump , &b[0] , screenWeight );

			if( _fittingM ) SolvePreconditionedCG( *_fittingM , _fittingPreconditioner , cgIterations , (int)b.size() , &b[0] , &x[0] , &_cgScratch , 1e-8 , omp_get_num_procs() , verbose );
#if USE_EIGEN
			else _eigenCholeskyFitter->solve( &b[0] , &x[0] );
#endif // USE_EIGEN

			for( int i=0 ; i<vCount ; i++ ) _newVertices[i][c] = (float)x[i];
		}
		if( whichMesh==1 ) sv.vertices = _newVertices;
		sprintf( sv.info[0] , "Fit normals: %.2f(s)\n" , Time()-t );
	}
}
template< class Real > void SurfaceSharpeningViewer< Real >::Idle( void )
{
	bool useLogInterpolation = true;
	if( _videoFrames && _videoFrame<=_videoFrames )
	{
		whichMesh = 1;
		char fileName[512];
		if( _videoFrame )
		{
			sprintf( fileName , "video.%d.jpg" , _videoFrame-1 );
			sv.saveFrameBuffer( fileName , GL_FRONT );
			printf( "Wrote frame[%d / %d]: %s (%f)       \r" , _videoFrame , _videoFrames , fileName , flowTime );
		}
		else printf( "Video: [%.2e , %.2e] / %d\n" , _videoStart , _videoEnd , _videoFrames );
		if( _videoFrame==_videoFrames ) printf( "\n" );

		if( _videoStart>0 && useLogInterpolation )
		{
			double _s = log( _videoStart ) , _e = log( _videoEnd );
			flowTime = exp( _s + ( ( _e - _s ) / (_videoFrames-1) ) * _videoFrame );
		}
		else flowTime = _videoStart + ( ( _videoEnd - _videoStart ) / (_videoFrames-1) ) * _videoFrame;
		UpdateFlowTime();
		sv.updateMesh( !_hasColor );
		glutPostRedisplay();
		_videoFrame++;
	}
	else sv.Idle();
}
template< class Real > void SurfaceSharpeningViewer< Real >::KeyboardFunc( unsigned char key , int x , int y ){ sv.KeyboardFunc( key , x , y ); }
template< class Real > void SurfaceSharpeningViewer< Real >::SpecialFunc( int key , int x , int y ){ sv.SpecialFunc( key , x ,  y ); }
template< class Real > void SurfaceSharpeningViewer< Real >::Display( void ){ sv.Display(); }
template< class Real > void SurfaceSharpeningViewer< Real >::Reshape( int w , int h ){ sv.Reshape( w , h ); }
template< class Real > void SurfaceSharpeningViewer< Real >::MouseFunc( int button , int state , int x , int y )
{
	if( glutGetModifiers() & GLUT_ACTIVE_SHIFT && state==GLUT_DOWN )
		if( sv.select( x , y , selectPoint.second ) )
		{
			if     ( button==GLUT_LEFT_BUTTON  ) selectPoint.first =  flowTime;
			else if( button==GLUT_RIGHT_BUTTON ) selectPoint.first = -flowTime;
		}
		else selectPoint.first = (Real)0;
	else sv.MouseFunc( button , state , x , y );
}
template< class Real > void SurfaceSharpeningViewer< Real >::MotionFunc( int x , int y ){ sv.MotionFunc( x , y ); }