#include <stdio.h>
#include <stdlib.h>
#include <omp.h>
#include <sys/timeb.h>
#ifndef WIN32
#include <sys/time.h>
#endif // WIN32
#include <GL/glew.h>
#include <GL/glut.h>
#include <Util/CmdLineParser.h>
#include <Util/Algebra.h>
#include <Util/MemoryUsage.h>
#include <Util/Ply.h>
#include <Util/FEM.h>
#include <Util/Camera.h>
#include <Util/Visualization.h>
#include "SurfaceSharpeningViewer.h"

double maxMemoryUsage = 0;
double MemoryUsage( void )
{
	double mem = double( MemoryInfo::Usage() ) / (1<<20);
	if( mem>maxMemoryUsage ) maxMemoryUsage = mem;
	return mem;
}
cmdLineParameter< char* > Out( "out" );							// The output mesh
cmdLineParameter< int > CGIterations( "cgIters" , 25 );			// The number of CG iterations to perform for each solve
cmdLineParameter< int > Smooth( "smooth" , 1 );					// The number of smoothing iterations
cmdLineParameter< int > PreSmooth( "preSmooth" , 0 );			// The number of pre-smoothing iterations
cmdLineParameter< float > ScreenWeight( "sWeight" , 1.f );		// Screening weight for fitting normals and positions
cmdLineReadable Verbose( "verbose" );							// Indicates that verbose output should be displayed on the command line
cmdLineParameter< float > FlowTime( "fTime" , 0.f );			// The duration of the flow
cmdLineParameter< int > Subdivide( "subdivide"  , 0 );			// Number of subdivision iterations to be performed
cmdLineParameter< int > Threads( "threads" , omp_get_num_procs() );				// The number of threads to use for parallelization
cmdLineReadable Lump( "lump" );									// Whether lumping should be used for the mass-matrix
cmdLineReadable NoColor( "noColor" );							// Ignore color if it's there
cmdLineReadable Single( "single" );								// Whether single precision floats should be used for computation

cmdLineReadable* params[] =
{
	&Out , &FlowTime , &ScreenWeight , &Verbose ,
	&Lump , &Single , &Subdivide , &CGIterations , &Smooth , &NoColor ,
	&Threads , &PreSmooth ,
	NULL
};

void ShowUsage( const char* ex )
{
	printf( "Usage: %s <input mesh> [params]:\n" , ex );
	printf( "\t[--%s <output file name>]\n" , Out.name );
	printf( "\t[--%s <flow time>=%f]\n" , FlowTime.name , FlowTime.value );
	printf( "\t[--%s <smoothing iterations>=%d]\n" , Smooth.name , Smooth.value );
	printf( "\t[--%s <subdivision iterations>=%d]\n" , Subdivide.name , Subdivide.value );
	printf( "\t[--%s <screening weight>=%f]\n" , ScreenWeight.name , ScreenWeight.value );
	printf( "\t[--%s <conjugate gradient iterations>=%d]\n" , CGIterations.name , CGIterations.value ); 
	printf( "\t[--%s]\n" , Lump.name );
	printf( "\t[--%s]\n" , NoColor.name );
	printf( "\t[--%s]\n" , Single.name );
	printf( "\t[--%s]\n" , Verbose.name );
}
template< class Real >
Point3D< Real > HSV2RGB( Point3D< Real > _hsv )
{
	Point3D< Real > rgb;
	Real hsv[] = { _hsv[0] , _hsv[1] , _hsv[2] };
	int i;
	Real f, p, q, t;
	if( hsv[1] == 0 )
	{
		rgb[0] = rgb[1] = rgb[2] = hsv[2];
		return rgb;
	}
	else if( hsv[1]<0 )
	{
		fprintf( stderr , "[ERROR] Saturation can't be negative\n" );
		return rgb;
	}
	while( hsv[0]<0 ) hsv[0] += 2. * M_PI;
	hsv[0] /= M_PI / 3.;
	i = (int)floor( hsv[0] );
	f = (Real)(hsv[0] - i);
	p = (Real)(hsv[2] * ( 1 - hsv[1] ));
	q = (Real)(hsv[2] * ( 1 - hsv[1] * f ));
	t = (Real)(hsv[2] * ( 1 - hsv[1] * ( 1 - f ) ));
	switch( i ) {
		case 0:
			rgb[0] = hsv[2] , rgb[1] = t , rgb[2] = p;
			break;
		case 1:
			rgb[0] = q , rgb[1] = hsv[2] , rgb[2] = p;
			break;
		case 2:
			rgb[0] = p , rgb[1] = hsv[2] , rgb[2] = t;
			break;
		case 3:
			rgb[0] = p , rgb[1] = q , rgb[2] = hsv[2];
			break;
		case 4:
			rgb[0] = t , rgb[1] = p , rgb[2] = hsv[2];
			break;
		default:
			rgb[0] = hsv[2] , rgb[1] = p , rgb[2] = q;
			break;
	}
	return rgb;
}

template< class Real >
int _main( int argc , char* argv[] )
{
	SurfaceVisualization& sv = SurfaceSharpeningViewer< Real >::sv;
	SurfaceSharpeningViewer< Real >::Init( argv[1] , Subdivide.value , NoColor.set , PreSmooth.value );
	SurfaceSharpeningViewer< Real >::cgIterations = CGIterations.value;
	SurfaceSharpeningViewer< Real >::threads = Threads.value;
	SurfaceSharpeningViewer< Real >::flowTime = FlowTime.value;
	SurfaceSharpeningViewer< Real >::screenWeight = ScreenWeight.value;
	SurfaceSharpeningViewer< Real >::lump = Lump.set;
	SurfaceSharpeningViewer< Real >::verbose = Verbose.set;
	SurfaceSharpeningViewer< Real >::smoothIters = Smooth.value;

	sprintf( sv.info.back() , "Mesh (original): %lld / %lld" , (unsigned long long)sv.vertices.size() , (unsigned long long)sv.triangles.size() );

	if( Out.set )
	{
		SurfaceSharpeningViewer< Real >::whichMesh = 1;

		double t = Time() , tt = Time();
		SurfaceSharpeningViewer< Real >::SetScreen();
		printf( "\t  Screen Time: %.2f (s)\n" , Time()-t ) , t = Time();
		SurfaceSharpeningViewer< Real >::SetFlow();
		printf( "\tSet Flow Time: %.2f (s)\n" , Time()-t ) , t = Time();
		SurfaceSharpeningViewer< Real >::AdvectSignal();
		printf( "\t  Advect Time: %.2f (s)\n" , Time()-t ) , t = Time();
		SurfaceSharpeningViewer< Real >::FitToNormals();
		printf( "\t     Fit Time: %.2f (s)\n" , Time()-t ) , t = Time();
		printf( "Total Time: %.2f (s)\n" , Time()-tt );
		SurfaceVisualization::OutputMeshCallBack( &SurfaceSharpeningViewer< Real >::sv , Out.value );
		return EXIT_SUCCESS;
	}


	SurfaceSharpeningViewer< Real >::SetScreen();
	SurfaceSharpeningViewer< Real >::SetFlow();
	SurfaceSharpeningViewer< Real >::AdvectSignal();
	SurfaceSharpeningViewer< Real >::FitToNormals();

	glutInitDisplayMode( GLUT_RGB | GLUT_DOUBLE | GLUT_DEPTH );
	glutInitWindowSize( sv.screenWidth , sv.screenHeight );
	glutInit( &argc , argv );
	char windowName[1024];
	sprintf( windowName , "Surface Sharpening Visualization" );
	glutCreateWindow( windowName );

	if( glewInit()!=GLEW_OK ) fprintf( stderr , "[ERROR] glewInit failed\n" ) , exit( 0 );
	glutIdleFunc    ( SurfaceSharpeningViewer< Real >::Idle );
	glutDisplayFunc ( SurfaceSharpeningViewer< Real >::Display );
	glutReshapeFunc ( SurfaceSharpeningViewer< Real >::Reshape );
	glutMouseFunc   ( SurfaceSharpeningViewer< Real >::MouseFunc );
	glutMotionFunc  ( SurfaceSharpeningViewer< Real >::MotionFunc );
	glutKeyboardFunc( SurfaceSharpeningViewer< Real >::KeyboardFunc );
	glutSpecialFunc ( SurfaceSharpeningViewer< Real >::SpecialFunc );

	glutMainLoop();
	return EXIT_SUCCESS;
}
int main( int argc , char* argv[] )
{
	if( argc<2 ){ ShowUsage( argv[0] ) ; return EXIT_FAILURE; }
	cmdLineParse( argc-2 , argv+2 , params );
	if( Single.set ) return _main< float  >( argc , argv );
	else             return _main< double >( argc , argv );
}