/**
 * UndoReader.java  1.00 99/03/11 Merlin Hughes
 *
 * Copyright (c) 1999 Merlin Hughes. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software
 * for non-commercial purposes and without fee is hereby granted
 * provided that this copyright notice appears in all copies.
 *
 * http://merlin.org/                          merlin@merlin.org
 */

package org.merlin.io;

import java.io.*;

public class UndoReader extends Reader {
  protected static final int INITIAL_UNDO_SIZE = 64;
  
  protected Reader reader;
  protected int[] buffer;
  protected int index, stored, capacity;
  protected boolean storing, restoring, closed;
  
  public UndoReader (Reader reader) {
    this (reader, INITIAL_UNDO_SIZE);
  }

  public UndoReader (Reader reader, int capacity) {
    super (reader);
    this.reader = reader;
    this.capacity = capacity;
    buffer = new int[capacity];
  }

  public int read () throws IOException {
    synchronized (lock) {
      int result;
      if (!storing && !restoring) {
        result = reader.read ();
      } else if (restoring) {
        result = buffer[index ++];
        if (index >= stored)
          restoring = false;
      } else {
        result = reader.read ();
	if (stored + 1 > capacity)
          ensureCapacity (1);
        buffer[stored ++] = result;
      }
      return result;
    }
  }

  protected void ensureCapacity (int amount) {
    capacity = capacity * 2 + amount;
    int[] copy = new int[capacity];
    System.arraycopy (buffer, 0, copy, 0, index);
    buffer = copy;
  }

  public int read (char[] dst, int offset, int length) throws IOException {
    synchronized (lock) {
      int result;
      if (!storing && !restoring) {
        result = reader.read (dst, offset, length);
      } else if (restoring) {
        if (buffer[index] < 0) {
          result = buffer[index ++];
        } else {
          result = (length < stored - index) ? length : stored - index;
          for (int i = 0; i < result; ++ i)
            dst[offset + i] = (char) buffer[index ++];
        }
        if (index >= stored)
          restoring = false;
      } else {
        result = reader.read (dst, offset, length);
        if (result < 0) {
          if (stored + 1 > capacity)
            ensureCapacity (1);
          buffer[stored ++] = result;
        } else {
          if (stored + result > capacity)
            ensureCapacity (result);
          for (int i = 0; i < result; ++ i)
            buffer[stored ++] = dst[offset + i];
        }
      }
      return result;
    }
  }

  public boolean ready () throws IOException {
    synchronized (lock) {
      return restoring || reader.ready ();
    }
  }

  public void close () throws IOException {
    try {
      reader.close ();
    } finally {
      synchronized (lock) {
        storing = restoring = false;
        closed = true;
      }
    }
  }

  public void checkpoint () throws IOException {
    synchronized (lock) {
      if (closed)
        throw new IOException ("Stream closed");
      else if (storing)
        throw new IOException ("Alreading checkpointed");
      if (restoring)
        System.arraycopy (buffer, index, buffer, 0, stored - index);
      stored -= index;
      index = 0;
      storing = true;
    }
  }

  public void commit () throws IOException {
    synchronized (lock) {
      if (closed)
        throw new IOException ("Stream closed");
      else if (!storing)
        throw new IOException ("Undo without checkpoint");
      storing = false;
    }
  }

  public void rollback () throws IOException {
    synchronized (lock) {
      if (closed)
        throw new IOException ("Stream closed");
      else if (!storing)
        throw new IOException ("Rollback without checkpoint");
      storing = false;
      index = 0;
      restoring = (index < stored);
    }
  }

  public void undo (int undo) throws IOException {
    synchronized (lock) {
      if (closed)
        throw new IOException ("Stream closed");
      else if (!storing)
        throw new IOException ("Undo without checkpoint");
      else if (undo < 0)
        throw new IOException ("Negative undo (" + undo + ")");
      if (restoring) {
        if (undo > index)
          throw new IOException ("Undo overflow");
        index -= undo;
      } else {
        if (undo > stored)
          throw new IOException ("Undo overflow");
        index = stored - undo;
        restoring = (index < stored);
      }
    }
  }

  public int peek () throws IOException {
    synchronized (lock) {
      if (closed)
        throw new IOException ("Stream closed");
      else if (!storing)
        throw new IOException ("Peek without checkpoint");
      int result = read ();
      undo (1);
      return result;
    }
  }
}
