/**
 * DataReader.java  1.00 99/03/11 Merlin Hughes
 *
 * Copyright (c) 1999 Merlin Hughes. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software
 * for non-commercial purposes and without fee is hereby granted
 * provided that this copyright notice appears in all copies.
 *
 * http://merlin.org/                          merlin@merlin.org
 *
 * This code was described in
 *   http://www.javaworld.com/javaworld/jw-04-1999/jw-04-step.html
 *
 * This version has been extended by Jason Eisner <jason@cs.jhu.edu>
 * to add the minMode field, the setMinMode method, and the version
 * of the constructor that can set setMinMode.  The original class
 * included commented-out code for handling this mode of reading, but
 * no way to get at that code short of uncommenting.
 */

package org.merlin.io;

import java.io.*;
import java.util.*;
import org.merlin.lang.*;

public class DataReader extends FilterReader {
  protected static final int EOF = -1, CR = '\r', LF = '\n';
  
  protected UndoReader undo;
  protected int whitespace;
  protected boolean lineMode;
  protected boolean minMode;  // added by JE; if set, then readInteger (for example)
                              // will read the initial 10 from 10abc or 10.3.
                              // This is like the behavior of scanf.  If not set
                              // (the default), then readInteger will read the
                              // entire word up to whitespace and then throw a
                              // NumberFormatException when it can't parse it as an
                              // integer.

  public DataReader (Reader reader) {
    this (reader, false, false);
  }

  public DataReader (Reader reader, boolean lineMode) {
    this (reader, lineMode, false);
  }

  public DataReader (Reader reader, boolean lineMode, boolean minMode) {
    super (new UndoReader (reader));
    lock = reader;
    undo = (UndoReader) in;
    setLineMode (lineMode);
    setMinMode (minMode);
  }

  public void setLineMode (boolean lineMode) {
    synchronized (lock) {
      this.lineMode = lineMode;
      if (lineMode) {
        whitespace = WHITESPACE_NOT_CRLF;
      } else {
        whitespace = WHITESPACE;
      }
    }
  }

  public void setMinMode (boolean minMode) {
    synchronized (lock) {
      this.minMode = minMode;
    }
  }

  public String readLine () throws IOException {
    synchronized (lock) {
      undo.checkpoint ();
      try {
        String line = readExp (~CRLF, ZERO_OR_MORE);
        int chr = undo.read ();
        if ((chr == CR) || (chr == LF)) {
          if ((chr == CR) && (undo.read () != LF))
            undo.undo (1);
        } else if ((chr == EOF) && "".equals (line)) {
          throw new EOFException ("EOF reading line");
        } else {
          undo.undo (1);
        }
        undo.commit ();
        return line;
      } catch (EOFException ex) {
        undo.commit ();
        throw ex;
      } catch (IOException ex) {
        undo.rollback ();
        throw ex;
      }
    }
  }

  public String readWord () throws IOException {
    synchronized (lock) {
      undo.checkpoint ();
      try {
        String value = readWordString ("word");
        undo.commit ();
        return value;
      } catch (EOFException ex) {
        undo.commit ();
        throw ex;
      } catch (IOException ex) {
        undo.rollback ();
        throw ex;
      }
    }
  }

  protected String readWordString (String what) throws IOException {
    prepare (what);
    return readExp (~WHITESPACE, ZERO_OR_MORE);
  }
  
  protected void prepare (String what) throws IOException {
    readExp (whitespace, ZERO_OR_MORE);
    int chr = undo.read ();
    if (chr == EOF) {
      throw new EOFException ("EOF reading " + what);
    } else if ((chr == CR) || (chr == LF)) {
      if ((chr == CR) && (undo.read () != LF))
        undo.undo (1);
      throw new EOLException ("EOL reading " + what);
    }
    undo.undo (1);
  }

  public long readLong () throws IOException, NumberFormatException {
    return readInteger ("long", Long.MIN_VALUE, Long.MAX_VALUE);
  }

  public int readInt () throws IOException, NumberFormatException {
    return (int) readInteger ("int", Integer.MIN_VALUE, Integer.MAX_VALUE);
  }

  public short readShort () throws IOException, NumberFormatException {
    return (short) readInteger ("short", Short.MIN_VALUE, Short.MAX_VALUE);
  }

  public byte readByte () throws IOException, NumberFormatException {
    return (byte) readInteger ("byte", Byte.MIN_VALUE, Byte.MAX_VALUE);
  }

  protected long readInteger (String what, long min, long max)
      throws IOException, NumberFormatException {
    synchronized (lock) {
      undo.checkpoint ();
      try {
        String value = readIntegerString (what);
        long result = Long.parseLong (value);
        if ((result < min) || (result > max))
          throw new NumberFormatException (value);
        undo.commit ();
        return result;
      } catch (EOFException ex) {
        undo.commit ();
        throw ex;
      } catch (IOException ex) {
        undo.rollback ();
        throw ex;
      } catch (NumberFormatException ex) {
        undo.rollback ();
        throw ex;
      }
    }
  }

  protected String readIntegerString (String what) throws IOException {
    if (minMode) {
      prepare (what);
      StringBuffer buffer = new StringBuffer ();
      buffer.append (readExp ('-', ZERO_OR_ONE));
      buffer.append (readExp (DIGITS, ZERO_OR_MORE));
      return buffer.toString ();
    } else
      return readWordString (what);
  }

  public long readUnsignedInt () throws IOException, NumberFormatException {
    return readCardinal ("unsigned int", 1L << 32);
  }
  
  public int readUnsignedShort () throws IOException, NumberFormatException {
    return (int) readCardinal ("unsigned short", 1 << 16);
  }

  public short readUnsignedByte () throws IOException, NumberFormatException {
    return (short) readCardinal ("unsigned byte", 1 << 8);
  }

  protected long readCardinal (String what, long max)
      throws IOException, NumberFormatException {
    synchronized (lock) {
      undo.checkpoint ();
      try {
        String value = readCardinalString (what);
        long result = Long.parseLong (value);
        if ((result < 0) || (result > max))
          throw new NumberFormatException (value);
        undo.commit ();
        return result;
      } catch (EOFException ex) {
        undo.commit ();
        throw ex;
      } catch (IOException ex) {
        undo.rollback ();
        throw ex;
      } catch (NumberFormatException ex) {
        undo.rollback ();
        throw ex;
      }
    }
  }

  protected String readCardinalString (String what) throws IOException {
    if (minMode) {
      prepare (what);
      StringBuffer buffer = new StringBuffer ();
      buffer.append (readExp (DIGITS, ZERO_OR_MORE));
      return buffer.toString ();
    } else 
      return readWordString(what);
  }

  public double readDouble () throws IOException, NumberFormatException {
    return readDecimal ("double");
  }

  public float readFloat () throws IOException, NumberFormatException {
    return (float) readDecimal ("float");
  }
  
  protected double readDecimal (String what) throws IOException, NumberFormatException {
    synchronized (lock) {
      undo.checkpoint ();
      try {
        String value = readDecimalString (what);
        double result = Double.valueOf (value).doubleValue ();
        undo.commit ();
        return result;
      } catch (EOFException ex) {
        undo.commit ();
        throw ex;
      } catch (IOException ex) {
        undo.rollback ();
        throw ex;
      } catch (NumberFormatException ex) {
        undo.rollback ();
        throw ex;
      }
    }
  }

  protected String readDecimalString (String what) throws IOException {
    if (minMode) {
      prepare (what);
      StringBuffer buffer = new StringBuffer ();
      buffer.append (readExp (SIGN, ZERO_OR_ONE));
      buffer.append (readExp (DIGITS, ZERO_OR_MORE));
      buffer.append (readExp ('.', ZERO_OR_ONE));
      buffer.append (readExp (DIGITS, ZERO_OR_MORE));
      if (buffer.length () > 0) {
	int chr = undo.read ();
	if ((chr == 'e') || (chr == 'E')) {
	  buffer.append ((char) chr);
	  buffer.append (readExp (SIGN, ZERO_OR_ONE));
	  buffer.append (readExp (DIGITS, ZERO_OR_MORE));
	} else {
	  undo.undo (1);
	}
      }
      return buffer.toString ();
    } else 
      return readWordString(what);
  }

  protected static final Hashtable booleans = new Hashtable ();
  
  static {
    booleans.put ("true", Boolean.TRUE);
    booleans.put ("false", Boolean.FALSE);
  }

  public boolean readBoolean () throws IOException, BooleanFormatException {
    synchronized (lock) {
      undo.checkpoint ();
      try {
        String value = readWordString ("boolean");
        Boolean result = (Boolean) booleans.get (value.toLowerCase ());
        if (result == null)
          throw new BooleanFormatException (value);
        undo.commit ();
        return result.booleanValue ();
      } catch (EOFException ex) {
        undo.commit ();
        throw ex;
      } catch (IOException ex) {
        undo.rollback ();
        throw ex;
      } catch (BooleanFormatException ex) {
        undo.rollback ();
        throw ex;
      }
    }
  }

  public char readChar () throws IOException {
    synchronized (lock) {
      undo.checkpoint ();
      try {
        prepare ("char");
        int chr = undo.read ();
        undo.commit ();
        return (char) chr;
      } catch (EOFException ex) {
        undo.commit ();
        throw ex;
      } catch (IOException ex) {
        undo.rollback ();
        throw ex;
      }
    }
  }

  public char readAnyChar () throws IOException {
    synchronized (lock) {
      undo.checkpoint ();
      try {
        int chr = undo.read ();
        if (chr == EOF) {
          throw new EOFException ("EOF reading any char");
        } else if (lineMode && ((chr == CR) || (chr == LF))) {
          if ((chr == CR) && (undo.read () != LF))
            undo.undo (1);
          throw new EOLException ("EOL reading any char");
        }
        undo.commit ();
        return (char) chr;
      } catch (EOFException ex) {
        undo.commit ();
        throw ex;
      } catch (IOException ex) {
        undo.rollback ();
        throw ex;
      }
    }
  }

  public int peek () throws IOException {
    synchronized (lock) {
      undo.checkpoint ();
      try {
        readExp (whitespace, ZERO_OR_MORE);
        int chr = undo.peek ();
        undo.rollback ();
        return chr;
      } catch (IOException ex) {
        undo.rollback ();
        throw ex;
      }
    }
  }

  public int peekAny () throws IOException {
    synchronized (lock) {
      undo.checkpoint ();
      try {
        int chr = undo.peek ();
        undo.rollback ();
        return chr;
      } catch (IOException ex) {
        undo.rollback ();
        throw ex;
      }
    }
  }

  public boolean peekEOL () throws IOException {
    int chr = peek ();
    return (chr == LF) || (chr == CR) || (chr == EOF);
  }

  public boolean peekEOF () throws IOException {
    int chr = peek ();
    return (chr == EOF);
  }

  public boolean skipWhitespace () throws IOException {
    synchronized (lock) {
      undo.checkpoint ();
      try {
        readExp (whitespace, ZERO_OR_MORE);
        int chr = undo.peek ();
        undo.commit ();
        return (chr != LF) && (chr != CR) && (chr != EOF);
      } catch (IOException ex) {
        undo.rollback ();
        throw ex;
      }
    }
  }
  
  public boolean skipLine () throws IOException {
    synchronized (lock) {
      try {
       readLine ();
      } catch (EOFException eof) {
        return false;
      }
      return true;
    }
  }

  public boolean skipEOL () throws IOException {
    synchronized (lock) {
      if (peekEOL ()) {
        skipLine ();
        return true;
      } else {
        return false;
      }
    }
  }

  protected static final int
    WHITESPACE = 0x10000,
    WHITESPACE_NOT_CRLF = 0x10001,
    CRLF = 0x10002,
    DIGITS = 0x10003,
    SIGN = 0x10004;
  
  protected static final int
    ZERO_OR_MORE = 0,
    ZERO_OR_ONE = 1;

  protected final char[] buffer = new char[16];
  
  protected String readExp (int type, int volume) throws IOException {
    StringBuffer result = new StringBuffer ();
    int index, amount, max = (volume == ZERO_OR_ONE) ? 1 : buffer.length;
    boolean invert = (type < 0);

    if (invert)
      type = ~type;
    do {
      index = 0;
      amount = undo.read (buffer, 0, max);
      while ((index < amount) && (invert ^ isMatch (buffer[index], type)))
        ++ index;
      result.append (buffer, 0, index);
    } while ((amount >= 0) && (index >= amount) && (volume == ZERO_OR_MORE));
    
    undo.undo ((amount < 0) ? 1 : amount - index);
    
    return result.toString ();
  }

  protected boolean isMatch (char chr, int type) {
    boolean result;
    switch (type) {
      case WHITESPACE:
        result = Character.isWhitespace (chr);
        break;
      case WHITESPACE_NOT_CRLF:
        result = (chr != CR) && (chr != LF) && Character.isWhitespace (chr);
        break;
      case CRLF:
        result = (chr == CR) || (chr == LF);
        break;
      case DIGITS:
        result = Character.isDigit (chr);
        break;
      case SIGN:
        result = (chr == '-') || (chr == '+');
        break;
      default:
        result = (type == chr);
        break;
    }
    return result;
  }
}
