package container;

/** In addition to storing its usual data, a class might allow users
 * to "decorate" it with some extra temporary data.  For example, we
 * might say that a vertex of a graph is colored red or black, or that
 * it is marked as visited, or that it stores a level number or a back
 * pointer.<p>
 *
 * A Decorable object can store any number of extra temporary fields.
 * Each is like a temporary instance variable, called an "attribute,"
 * that we can add or remove from the object.  For example, the
 * object's COLOR attribute could be "red" or "black," or the object
 * might not have a COLOR attribute right now.<p>
 * 
 * To make a new attribute, COLOR, use the constructor for the
 * Attribute class.  Now you can do p.put(COLOR,RED) and p.get(COLOR)
 * for a position p.  Here RED would be an object of class Color, or
 * perhaps just the string "red" if you are too lazy to make such a
 * class.<p>
 *
 * The textbook discusses decorations (section 12.3) and gives
 * examples (e.g., code fragment 12.4), but it does not use an
 * Attribute class as we do.  See Attribute.java for discussion.
 * Also, the textbook sticks to decorable positions, whereas any 
 * object could implement Decorable interface given here.<p>
 */

public interface Decorable {
  
  /** @return this object's value for attribute a, or null
   * if no such value is currently stored.
   */
  public Object get(Attribute a);
  
  /** Does this object currently have any value for this attribute?
   *
   * (Asking has(a) is clearer than asking get(a)!=null, and it
   * distinguishes between no value and a value that happens to be null.)
   */
  public boolean has(Attribute a);
  
  /** Makes this object store o as the value of attribute a.
   * @return previous value, or null if there was none
   */
  public Object put(Attribute a, Object o);
  
  /** Removes attribute a and its associated value from this object.
   * @return previous value, or null if there was none
   */
  public Object remove(Attribute a);
}
