/*
    BT848 driver - split apart to support both Linux & FreeBSD
    brad@parker.boston.ma.us (I don't care for copyrights)

    Portions Copyright (C) 1996,97 Ralph Metzler (rjkm@thp.uni-koeln.de)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <linux/bios32.h>
#include <linux/config.h>
#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/kernel.h>
#include <linux/major.h>
#include <linux/module.h>
#include <linux/malloc.h>
#include <linux/mm.h>
#include <linux/pci.h>
#include <linux/signal.h>
#include <asm/io.h>
#include <asm/pgtable.h>
#include <asm/page.h>
#include <linux/sched.h>
#include <asm/segment.h>
#include <linux/types.h>

/*
#define ioremap vremap 
#define iounmap vfree
*/

#include "bt848_linux.h"
#include "ioctl_meteor.h"
#include "ioctl_bt848.h"
#include "brktree_reg.h"
#include "bt848_common.h"

#define PCI_VENDOR_ID_BROOKTREE 0x109e
#define PCI_DEVICE_ID_BT848     0x350

struct mm_struct *initmm;

/* set by insmod (optionally) */
static unsigned int remap = 0;

static int major = CDEV_MAJOR;

void find_initmm(void)
{
  int i = 0;
  struct task_struct *cur;
  
  cur = current;
  while (cur != cur->p_opptr) {
    i++;
    if (i > 200) break;
    cur = cur->p_opptr;
  }
  initmm = cur->mm;
  /*printk("bt848: initmm %d, %08x\n", i, initmm);*/
}

#define  TRITON_PCON	           0x50 
#define  TRITON_BUS_CONCURRENCY   (1<<0)
#define  TRITON_STREAMING	  (1<<1)
#define  TRITON_WRITE_BURST	  (1<<2)
#define  TRITON_PEER_CONCURRENCY  (1<<3)
  
static void
handle_chipset(void)
{
  int index;
  
  for (index = 0; index < 8; index++) {
    unsigned char bus, devfn;
    unchar b, bo;
    
    /* nothing wrong with this one, just checking buffer control config */
    if (!pcibios_find_device(PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_82441,
			    index, &bus, &devfn))  {
      pcibios_read_config_byte(bus, devfn, 0x53, &b);
      printk("bt848: Host bridge: 82441FX Natoma, ");
      printk("bufcon=0x%02x\n",b);
    }

    if (!pcibios_find_device(PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_82437,
			    index, &bus, &devfn)) {
      printk("bt848: Host bridge 82437FX Triton PIIX\n");
      pcibios_read_config_byte(bus, devfn, TRITON_PCON, &b);

      bo=b;
      printk("bt848: 82437FX: PCON: 0x%x\n",b);

#if 1
      /* 430FX (Triton I) freezes with bus concurrency on -> switch it off */
      if(!(b & TRITON_BUS_CONCURRENCY)) {
	printk("bt848: 82437FX: disabling bus concurrency\n");
	b |= TRITON_BUS_CONCURRENCY;
      }

      /* still freezes on other boards -> switch off even more */
      if(b & TRITON_PEER_CONCURRENCY) {
	printk("bt848: 82437FX: disabling peer concurrency\n");
	b &= ~TRITON_PEER_CONCURRENCY;
      }

      if(!(b & TRITON_STREAMING)) {
	printk("bt848: 82437FX: disabling streaming\n");
	b |=  TRITON_STREAMING;
      }
#endif

      if (b!=bo) {
	pcibios_write_config_byte(bus, devfn, TRITON_PCON, b); 
	printk("bt848: 82437FX: PCON changed to: 0x%x\n",b);
      }
    }
  }
}

#define UNIT(x)		((x) & 0x0f)
#define DEV(x)		((x >> 4) & 0x0f)
#define VIDEO_DEV	0x00
#define TUNER_DEV	0x01

int bt848_capture_wait(bktr_ptr_t bktr)
{
  struct wait_queue wait = { current, NULL };
  int status = 0;

  /*printk("capture_wait\n");*/

  add_wait_queue(&bktr->sleep, &wait);
  current->state = TASK_INTERRUPTIBLE;

  schedule();

  /* I know it seems like we should return -ERESTARTSYS,
   * but we're in "freebsd emulation mode" here, so return errno > 0
   */
  if (current->signal & ~current->blocked) {
    status = /*-*/ERESTARTSYS;
  }

  remove_wait_queue(&bktr->sleep, &wait);
  current->state = TASK_RUNNING;

  /*printk("capture_wait done, status %d\n", status);*/
  return status;
}

int bt848_capture_wakeup(bktr_ptr_t bktr)
{
  wake_up_interruptible(&bktr->sleep);
  return 0;
}

static int
bt848_read(struct inode *ino, struct file *file, char *ubuf, int ucount)
{
  unsigned int minor = MINOR(ino->i_rdev);
  int unit = UNIT(minor);
  bktr_ptr_t bktr;
  bt848_ptr_t	bt848;
  int status, count;
	
  if (unit >= brooktree_cnt)
    return -ENODEV;
  bktr = &brooktree[unit];

  if (minor > 0)
    return -ENXIO;

  if (bktr->bigbuf == 0)	/* no frame buffer allocated (ioctl failed) */
    return -ENOMEM;

  if (bktr->flags & METEOR_CAP_MASK)
    return -EIO;		/* already capturing */

  count = bktr->rows * bktr->cols * 
    bt_pixfmt_table[ bktr->pixfmt ].public.Bpp;

  if ((int)ucount < count)
    return -EINVAL;

#ifdef READ_WAITS_FOR_CAPTURE
  bktr->flags &= ~(METEOR_CAP_MASK | METEOR_WANT_MASK);

  /* Start capture */
  bt848 = bktr->base;
  bt848->gpio_dma_ctl = FIFO_ENABLED;
  bt848->gpio_dma_ctl = FIFO_RISC_ENABLED;

  /* status = tsleep((caddr_t)bktr, METPRI, "captur", 0); */
  status = bt848_capture_wait(bktr);
#else
  status = 0;
#endif

  if (status == 0) {
    status = verify_area(VERIFY_WRITE, (int *)ubuf, count);
    if (status == 0) {
      memcpy_tofs((void *)ubuf, (void *)bktr->bigbuf, count);
      status = count;
    }
  }

#ifdef READ_WAITS_FOR_CAPTURE
  bktr->flags &= ~(METEOR_SINGLE | METEOR_WANT_MASK);
#endif

  return status;
}

static int
bt848_ioctl(struct inode *ino, struct file* file,
	    unsigned int cmd, unsigned long arg)
{
  unsigned int minor = MINOR(ino->i_rdev);
  int unit = UNIT(minor);
  bktr_ptr_t bktr;
  int ret;

  if (unit >= brooktree_cnt)
    return -ENODEV;
  bktr = &brooktree[unit];

  if (bktr->bigbuf == 0)	/* no frame buffer allocated (ioctl failed) */
    return -ENOMEM;

  switch ( DEV( minor ) ) {
  case VIDEO_DEV:
    ret = video_ioctl( bktr, unit, cmd, (caddr_t)arg, (struct proc *)current );
    break;

  case TUNER_DEV:
    ret = tuner_ioctl( bktr, unit, cmd, (caddr_t)arg, (struct proc *)current );
    break;

  default:
    return -ENODEV;
  }

  if (ret > 0)
    return -ret;

  return 0;
}

static int
bt848_open( struct inode* ino, struct file* filep)
{
  unsigned int minor = MINOR(ino->i_rdev);
  int unit = UNIT(minor);
  bktr_ptr_t bktr;
  int ret;

  if (unit >= brooktree_cnt)
    return -ENODEV;
  bktr = &brooktree[unit];

  if (!(bktr->flags & METEOR_INITALIZED)) /* device not found */
    return -ENXIO;

  switch ( DEV(minor) ) {
  case VIDEO_DEV:
    ret = video_open( bktr );
    break;

  case TUNER_DEV:
    ret = tuner_open( bktr );
    break;

  default:
    return -ENODEV;
  }

  if (ret > 0)
    return -ret;

  MOD_INC_USE_COUNT;
  return 0;
}

static void
bt848_close( struct inode* ino, struct file* filep)
{
  unsigned int minor = MINOR(ino->i_rdev);
  int unit = UNIT(minor);
  bktr_ptr_t bktr;

  if (unit >= brooktree_cnt)
    return;
  bktr = &brooktree[unit];

  switch ( DEV( minor ) ) {
  case VIDEO_DEV:
    video_close( bktr );
    break;

  case TUNER_DEV:
    tuner_close( bktr );
    break;
  }

  MOD_DEC_USE_COUNT;
}

static int
bt848_mmap(struct inode *inode, struct file *filep,
	   struct vm_area_struct *vma)
{
  unsigned int minor = MINOR(inode->i_rdev);
  int unit = UNIT(minor);
  bktr_ptr_t bktr;

  if (unit >= brooktree_cnt)
    return;
  bktr = &brooktree[unit];

#if 0
#if defined(__i386__)
	/*
	 * hmm.. This disables high-memory caching, as the XFree86 team
	 * wondered about that at one time.
	 * The surround logic should disable caching for the high device
	 * addresses anyway, but right now this seems still needed.
	 */
	if (/*x86 > 3 && */vma->vm_offset >= high_memory)
		pgprot_val(vma->vm_page_prot) |= _PAGE_PCD;
#endif
#endif

  vma->vm_offset += (u_long)bktr->bigbuf;
  if (vma->vm_offset & ~PAGE_MASK)
    return -ENXIO;

  if (remap_page_range(vma->vm_start, vma->vm_offset,
		       vma->vm_end - vma->vm_start, vma->vm_page_prot))
    return -EAGAIN;
  vma->vm_inode = inode;
  inode->i_count++;
  return 0;
}

static struct file_operations bt848_fops = {
	NULL,
	bt848_read,
	NULL,
	NULL,
	NULL,		/* bt848_select */
	bt848_ioctl,
	bt848_mmap,
	bt848_open,
	bt848_close,
	NULL		/* fsync */
};

static void 
bt848_irq(int irq, void *dev_id, struct pt_regs * regs)
{
  bktr_ptr_t bktr = (bktr_ptr_t)dev_id;
  bktr_intr(bktr);
}

/* scan for a Bt848 card, request the irq and map the io memory */
static int
find_bt848(void)
{
  short pci_index;    
  unchar command, latency;
  int result, i;
  unchar bus, devfn;
  bktr_ptr_t bktr;

  brooktree_cnt = 0;

  if (!pcibios_present()) {
    printk("bt848: PCI-BIOS not present or not accessable!\n");
    return 0;
  }

  for (pci_index = 0;
       !pcibios_find_device(PCI_VENDOR_ID_BROOKTREE, PCI_DEVICE_ID_BT848,
			    pci_index, &bus, &devfn);
       ++pci_index)
    {
    bktr = &brooktree[brooktree_cnt];
    bktr->bus = bus;
    bktr->devfn = devfn;
    bktr->base = NULL;
    
    pcibios_read_config_byte(bktr->bus, bktr->devfn,
			     PCI_INTERRUPT_LINE, &bktr->irq);
    pcibios_read_config_dword(bktr->bus, bktr->devfn, PCI_BASE_ADDRESS_0,
			      &bktr->phys_base);

    if (remap && (!brooktree_cnt)) { 
      remap <<= 20;
      remap &= PCI_BASE_ADDRESS_MEM_MASK;
      printk("bt848: remapping to : 0x%08x\n", remap);
      remap |= bktr->phys_base & (~PCI_BASE_ADDRESS_MEM_MASK);
      pcibios_write_config_dword(bktr->bus, bktr->devfn, PCI_BASE_ADDRESS_0,
				 remap);
      pcibios_read_config_dword(bktr->bus, bktr->devfn, PCI_BASE_ADDRESS_0,
				&bktr->phys_base);
    }
    
    bktr->phys_base &= PCI_BASE_ADDRESS_MEM_MASK;
    pcibios_read_config_byte(bktr->bus, bktr->devfn, PCI_CLASS_REVISION,
			     &bktr->revision);
    printk("bt848: Brooktree Bt848 (revision %d)\n",bktr->revision);
    printk("bt848: bus: %d, devfn: %d, ", bktr->bus, bktr->devfn);
    printk("irq: %d, ",bktr->irq);
    printk("memory: 0x%08x\n", bktr->phys_base);
    
    bktr->base = vremap(bktr->phys_base, 0x1000);

    result = request_irq(bktr->irq, bt848_irq,
			 SA_SHIRQ | SA_INTERRUPT,"bt848",(void *)bktr);
    if (result == -EINVAL) {
      printk("bt848: Bad irq number or handler\n");
      return -EINVAL;
    }
    if (result == -EBUSY) {
      printk("bt848: IRQ %d busy, change your PnP config in BIOS\n",bktr->irq);
      return result;
    }
    if (result < 0) 
      return result;

#if 0
    if (result == -EBUSY) {
      for (i = 0; i < brooktree_cnt; i++)
	if (brooktree[i].irq == bktr->irq)
	  goto shared;
      printk("IRQ %d busy\n",b tv->irq);
      return -EBUSY;
    shared:
      printk("bt848: Sharing IRQ %d with other Bt848 card(s)\n", bktr->irq);
    }      
#endif

    /* enable bus-mastering */
    pcibios_read_config_byte(bktr->bus, bktr->devfn, PCI_COMMAND, &command);
    command |= PCI_COMMAND_MASTER;
    pcibios_write_config_byte(bktr->bus, bktr->devfn, PCI_COMMAND, command);
    pcibios_read_config_byte(bktr->bus, bktr->devfn, PCI_COMMAND, &command);
    if (!(command & PCI_COMMAND_MASTER)) {
      printk("bt848: PCI bus-mastering could not be enabled\n");
      return -1;
    }

    /*
     * PCI latency timer.  32 is a good value for 4 bus mastering slots, if
     * you have more than four, then 16 would probably be a better value.
     */
#ifndef BROOKTREE_DEF_LATENCY_VALUE
#define BROOKTREE_DEF_LATENCY_VALUE	10 /* 32 ? */
#endif

    pcibios_read_config_byte(bktr->bus, bktr->devfn, PCI_LATENCY_TIMER,
			     &latency);
    if (!latency) {
      latency = BROOKTREE_DEF_LATENCY_VALUE;
      pcibios_write_config_byte(bktr->bus, bktr->devfn, PCI_LATENCY_TIMER,
				latency);
    }
    printk("bt848: latency: %02x\n", latency);
    brooktree_cnt++;
  }
  printk("bt848: %d Bt848 card(s) found.\n", brooktree_cnt);
  return brooktree_cnt;
}

vm_offset_t
get_bktr_mem( int unit, unsigned size )
{
  vm_offset_t	addr = 0;

  addr = kmalloc(size, GFP_KERNEL);
  if (addr == 0) {
    printk("bt848%d: Unable to allocate %d bytes of memory.\n",
	   unit, size);
  }

  return( addr );
}



static int
init_bt848(int unit)
{
  bktr_ptr_t	bktr;
  bt848_ptr_t	bt848;
  u_char	*test;
  vm_offset_t	buf;
  u_long	latency;

  bktr = &brooktree[unit];

  /* allocate space for dma program */
  bktr->dma_prog = get_bktr_mem(unit, DMA_PROG_ALLOC);
  bktr->odd_dma_prog = get_bktr_mem(unit, DMA_PROG_ALLOC);

  printk("bt848%d: dma_prog %p, odd_dma_prog %p\n",
	 unit, bktr->dma_prog, bktr->odd_dma_prog);

  if (bktr->dma_prog == 0 || bktr->odd_dma_prog == 0)
    return -1;

  /* allocate space for pixel buffer */
  if ( BROOKTREE_ALLOC ) {
    buf = bigphysarea_alloc(BROOKTREE_ALLOC);
    if (buf == 0) {
      printk("bt848%d: Unable to allocate %d bytes (bigphysarea_alloc)\n",
	     unit, BROOKTREE_ALLOC);
    }
  } else
    buf = 0;

  printk("bt848%d: buffer size %d, addr 0x%x\n",
	 unit, BROOKTREE_ALLOC, vtophys(buf));

  bktr->bigbuf = buf;
  bktr->alloc_pages = BROOKTREE_ALLOC_PAGES;
  if (buf != 0) {
    memset((caddr_t) buf, 0, BROOKTREE_ALLOC);
    buf = (char *)vtophys(buf);

    bktr->flags = METEOR_INITALIZED | METEOR_AUTOMODE |
      METEOR_DEV0 | METEOR_RGB16;
    bktr->dma_prog_loaded = FALSE;
    bktr->cols = 640;
    bktr->rows = 480;
    bktr->frames = 1;		/* one frame */
    bktr->format = METEOR_GEO_RGB16;
    bktr->pixfmt = oformat_meteor_to_bt( bktr->format );
    bktr->pixfmt_compat = TRUE;
    bt848 = bktr->base;
    bt848->int_mask = ALL_INTS_DISABLED;
    bt848->gpio_dma_ctl = FIFO_RISC_DISABLED;
  }

  /* defaults for the tuner section of the card */
  bktr->tflags = TUNER_INITALIZED;
  bktr->tuner.frequency = 0;
  bktr->tuner.channel = 0;
  bktr->tuner.chnlset = DEFAULT_CHNLSET;
  bktr->audio_mux_select = 0;
  bktr->audio_mute_state = FALSE;

  bt_probeCard( bktr, TRUE );

#if 0
  u_long fun;
  fun = pcibios_read_config(tag, PCI_COMMAND_STATUS_REG);
  pcibios_write_config(tag, PCI_COMMAND_STATUS_REG, fun | 4);
#endif

  return 0;
}

static void
release_bt848()
{
  unchar command;
  int i;
  bktr_ptr_t bktr;
  bt848_ptr_t bt848;

  for (i = 0; i < brooktree_cnt; i++) {
    bktr = &brooktree[i];
    bt848 = bktr->base;

    /* first disable interrupts before unmapping the memory! */
    bt848->int_mask = ALL_INTS_DISABLED;
    bt848->int_stat = ALL_INTS_CLEARED;

    /* disable drivers on the GPIO port that control the MUXes */
    bt848->gpio_out_en = 0;
    
    /* disable PCI bus-mastering */
    pcibios_read_config_byte(bktr->bus, bktr->devfn, PCI_COMMAND, &command);
    command |= PCI_COMMAND_MASTER;
    pcibios_write_config_byte(bktr->bus, bktr->devfn, PCI_COMMAND, command);
    
    /* unmap and free memory */
    printk("bigbuf: 0x%08x\n", bktr->bigbuf);
    if (bktr->bigbuf)
      bigphysarea_free((char *)bktr->bigbuf, bktr->alloc_pages);

    printk("dma_prog: 0x%08x\n", bktr->dma_prog);
    if (bktr->dma_prog)
      kfree((char *)bktr->dma_prog);

    printk("odd_dma_prog: 0x%08x\n", bktr->odd_dma_prog);
    if (bktr->odd_dma_prog)
      kfree((char *)bktr->odd_dma_prog);

    free_irq(bktr->irq, bktr);

    printk("base: 0x%08x\n", bktr->base);
    if (bktr->base)
      vfree((char *)bktr->base);
  }
}

#ifdef __cplusplus
extern "C" {
#endif

extern int printk(const char* fmt, ...);

int
init_module( void)
{
  int i;

  find_initmm();
  handle_chipset();

  if (find_bt848() < 0)
    return -EIO;

  for (i = 0; i < brooktree_cnt; i++) 
    if (init_bt848(i) < 0) {
      release_bt848();
      return -EIO;
    } 
  
  if (register_chrdev(major, "bt848", &bt848_fops)) {
    printk("register_chrdev failed\n");
    release_bt848();
    return -EIO;
  }

  return 0;
}

void
cleanup_module(void)
{
  release_bt848();
  
  if (unregister_chrdev(major, "bt848") != 0)
    printk("bt848: cleanup_module failed\n");
  else
    printk("bt848: cleanup done\n");
}

#ifdef __cplusplus
}
#endif

