/*
 *   (c) 1997 Gerd Knorr <kraxel@cs.tu-berlin.de>
 *
 */
#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <sys/shm.h>
#include <sys/ipc.h>
#include <sys/wait.h>


#include <asm/types.h>		/* XXX glibc */
#include "videodev.h"

#include "colorspace.h"
#include "writefile.h"
#include "writeavi.h"
#include "sound.h"

#define DEVNAME "/dev/video"

#define POST_NOTHING     0
#define POST_RGB_SWAP    1
#define POST_UNPACK422   2
#define POST_UNPACK411   3
#define POST_RAW         4

/* ---------------------------------------------------------------------- */

static struct video_capability  capability;
static struct video_picture     pict;

static unsigned char            *map = NULL;
static struct video_mmap        gb1,gb2;
static struct video_mbuf        gb_buffers = { 2*0x151000, 0, {0,0x151000 }};

static int                      bufcount = 2;
static int                      wtalk,rtalk;

static char *device   = DEVNAME;
static char *filename = NULL;
int input_channel = 0;
long input_format = VIDEO_PALETTE_RGB32;

static int  single = 1, format = -1, absframes = 1;
static int  width = 640, height = 480, quiet = 0, fps = 30;

static int  signaled = 0;

/* ---------------------------------------------------------------------- */

void
usage(char *prog)
{
    char *h;

    if (NULL != (h = strrchr(prog,'/')))
	prog = h+1;
    
    fprintf(stderr,
	    "%s grabs image(s) from a bt848 card\n"
	    "\n"
	    "usage: %s [ options ]\n"
	    "\n"
	    "options:\n"
	    "  -q          quiet operation\n"
	    "  -i input    input channel\n"
	    "  -c device   specify device             [%s]\n"
	    "  -f format   specify output format      [%s]\n"
	    "  -s size     specify size               [%dx%d]\n"
	    "  -b buffers  specify # of buffers       [%d]\n"
	    "  -t times    number of frames           [%d]\n"
	    "  -r fps      frame rate                 [%d]\n"
	    "  -o file     output file name           [%s]\n"
	    "\n"
	    "If the filename has some digits, %s will write multiple files,\n"
	    "otherwise one huge file.  Writing to one file works with raw\n"
	    "data only.  %s will switch the default format depending on the\n"
	    "filename extention (ppm, pgm, jpg, jpeg, avi)\n"
	    "\n"
	    "funny chars:\n"
	    "  +      grabbed picture queued to fifo\n"
	    "  o      grabbed picture not queued (fifo full)\n"
	    "  -      picture written to disk and dequeued from fifo\n"
	    "  s      sync\n"
	    "  xx/yy  (at end of line) xx frames queued, yy frames grabbed\n" 
	    "\n"
	    "formats:\n"
	    "  raw data:      rgb, gray, 411, 422, raw\n"
	    "  file formats:  ppm, pgm, jpeg\n"
	    "  movie formats: avi15, avi24\n"
	    "\n"
	    "examples:\n"
	    "  %s -o image.ppm                           write single ppm file\n"
	    "  %s -f 411 | display -size 320x240 yuv:-   yuv raw data to stdout\n"
	    "  %s -s 320x240 -t 5 -o frame000.jpeg       write 5 jpeg files\n"
	    "\n"
	    "--\n"
	    "(c) 1998 Gerd Knorr <kraxel@cs.tu-berlin.de>\n",
	    prog, prog, device,
	    "none",
	    width, height, bufcount, absframes, fps,
	    filename ? filename : "stdout",
	    prog, prog, prog, prog, prog);
}

/* ---------------------------------------------------------------------- */

void
writer()
{
    char buffer;

    signal(SIGINT,SIG_IGN);

    for (;;) {

	/* wait for frame */
	switch (read(rtalk,&buffer,1)) {
	case -1:
	    perror("writer: read socket");
	    exit(1);
	case 0:
	    if (!quiet)
		fprintf(stderr,"writer: done\n");
	    exit(0);
	    return;
	}

	/* free buffer */
	if (1 != write(wtalk,&buffer,1)) {
	    perror("writer: write socket");
	    exit(1);
	}
	/*	if (!quiet)
		fprintf(stderr,"-");*/
    }
}


/* ---------------------------------------------------------------------- */

/* ---------------------------------------------------------------------- */

void
ctrlc(int signal)
{
    static char text[] = "^C - one moment please\n";
    write(2,text,strlen(text));
    signaled=1;
}

static struct video_channel     *channels;

static int
grab_input(int fd, int input, int norm)
{

  if ((input < 0) || (input >= capability.channels)){
    perror("Input out of range");
    return 0;
  }

  channels[input].norm=norm;
    if (-1 == ioctl(fd, VIDIOCSCHAN, &channels[input]))
	perror("ioctl VIDIOCSCHAN");
    return 0;
}


int
grab_picture(int fd, int color, int bright, int hue, int contrast)
{
    if (color != -1)
	pict.colour = color;
    if (contrast != -1)
	pict.contrast = contrast;
    if (bright != -1)
	pict.brightness = bright;
    if (hue != -1)
	pict.hue = hue;

    if (-1 == ioctl(fd,VIDIOCSPICT,&pict))
	perror("ioctl VIDIOCSPICT");

    return 0;
}

int
open_bttv(char *device)

{

  int fd;
  int i;

    /* open */
    if (-1 == (fd = open(device,O_RDWR))) {
	fprintf(stderr,"open %s: %s\n",device,strerror(errno));
	return -1;
    }

    /* get settings */
    if (-1 == ioctl(fd,VIDIOCGCAP,&capability)) {
	perror("ioctl VIDIOCGCAP");
	return -1;
    }

    /* Tell us about this device */
    printf("%s %x %dx%d to %dx%d\n",
	   capability.name,
	   capability.type,
	   capability.minwidth,	   capability.minheight,
	   capability.maxwidth,	   capability.maxheight);


    /* Figure out the channels */
    channels = malloc(sizeof(struct video_channel)*capability.channels);
    memset(channels,0,sizeof(struct video_channel)*capability.channels);

    for (i = 0; i < capability.channels; i++) {
	channels[i].channel = i;
	if (-1 == ioctl(fd,VIDIOCGCHAN,&channels[i])) {
	  perror("ioctl VIDIOCGCHAN"); return -1;}
    }

    return fd;
}


int
main(int argc, char **argv)
{
    int  c,s,count=0,i,n,p1[2],p2[2];
    int filefd;
    char outbuf[640*480*3];
    int fd;

    /* parse options */
    for (;;) {
	if (-1 == (c = getopt(argc, argv, "hqf:i:s:c:o:b:t:r:")))
	    break;
	switch (c) {
	case 'q':
	    quiet = 1;
	    break;
	case 'i':
	  input_channel = atoi(optarg);
	  break;
	  
	case 's':
	    if (2 != sscanf(optarg,"%dx%d",&width,&height))
		width = height = 0;
	    break;
	case 'c':
	    device = optarg;
	    break;
	case 'o':
	    filename = optarg;
	    for (i = 0, n = strlen(filename); i < n; i++) {
		if (isdigit(filename[i]))
		    single = 0;
	    }
	    if (format == -1) {
		if (strstr(filename,"ppm"))
		    format = 0;
		if (strstr(filename,"pgm"))
		    format = 1;
		if (strstr(filename,"jpeg"))
		    format = 2;
		if (strstr(filename,"jpg"))
		    format = 2;
		if (strstr(filename,"avi"))
		    format = 3;
	    }
	    break;
	case 't':
	    absframes = atoi(optarg);
	    break;
	case 'b':
	    bufcount = atoi(optarg);
	    break;
	case 'r':
	    fps = atoi(optarg);
	    break;
	case 'h':
	default:
	    usage(argv[0]);
	    exit(1);
	}
    }

    fd = open_bttv(device);

    /* Set channel */

    grab_input(fd, input_channel,VIDEO_MODE_NTSC);

    /* Map the video memory */

    map = mmap(0,gb_buffers.size,PROT_READ|PROT_WRITE,MAP_SHARED,fd,0);
    if ((unsigned char*)-1 == map) {
	perror("mmap");
    } else {
	fprintf(stderr,"v4l: mmap()'ed buffer size = 0x%x\n",
		gb_buffers.size);
    }

    /* start up writer */
    if (-1 == pipe(p1) || -1 == pipe(p2)) {
	perror("pipe");
	exit(1);
    }
    switch(fork()) {
    case -1:
	perror("fork");
	exit(1);
    case 0:
	close(p1[0]);
	close(p2[1]);
	wtalk = p1[1];
	rtalk = p2[0];
	fcntl(rtalk,F_SETFL,0);
	nice(5);
	writer();
	exit(0);
    default:
	close(p2[0]);
	close(p1[1]);
	wtalk = p2[1];
	rtalk = p1[0];
	fcntl(rtalk,F_SETFL,O_NONBLOCK);
	break;
    }

    /* catch ^C */
    signal(SIGINT,ctrlc);

    /* prepare for grabbing */
    gb1.format = input_format;
    gb1.frame  = 0;
    gb1.width  = width;
    gb1.height = height;

    gb2.format = input_format;
    gb2.frame  = 1;
    gb2.width  = width;
    gb2.height = height;

    if (-1 == ioctl(fd,VIDIOCMCAPTURE,&gb1)) {
	if (errno == EAGAIN)
	    fprintf(stderr,"grabber chip can't sync (no station tuned in?)\n");
	else
	    perror("ioctl VIDIOCMCAPTURE");
	exit(1);
    }
    count++;

    if (-1 == ioctl(fd,VIDIOCSYNC, &gb1.frame)) {
      perror("ioctl VIDIOCSYNC");
      exit(1);
    }

    /* For right now, write out a raw file */

    filefd = open("test.ppm",O_CREAT | O_RDWR | O_TRUNC, 0666);
    if (filefd <= 0) {
      perror("file open failed\n");
      exit(1);
    }
    printf(" -- %d -- \n",gb1.width*gb1.height*4);

    sprintf(outbuf,"P6\n640 460 \n255 \n");
    write(filefd, outbuf, strlen(outbuf));
    i = 0;
    for (count = 0;count < 40;count+=4) {
      printf("%d, %d %d %d %d \n",count,map[count],
      map[count+1],
      map[count+2],map[count+3]);
    }

    for (count = 0;count < gb1.width*gb1.height*4;count+=4) {
      outbuf[i++] = map[count+1];
      outbuf[i++] = map[count+0];
      outbuf[i++] = map[count+2];
    }

    write(filefd, outbuf, gb1.width*gb1.height*3);
    close(filefd);
      
#ifdef 0

    /* main loop */
    gettimeofday(&start,NULL);
    for (;queued < absframes && !signaled; count++) {

	if (-1 == ioctl(fd,VIDIOCMCAPTURE,(count%2) ? &gb2 : &gb1)) {
	    if (errno == EAGAIN)
		fprintf(stderr,"grabber chip can't sync (no station tuned in?)\n");
	    else
		perror("ioctl VIDIOCMCAPTURE");
	    exit(1);
	}

	if (-1 == ioctl(fd,VIDIOCSYNC,(count%2) ? &gb1.frame : &gb2.frame)) {
	    perror("ioctl VIDIOCSYNC");
	    exit(1);
	}

	/*	queued = putbuffer(map + gb_buffers.offsets[(count%2) ? 0 : 1]);*/
    }

    if (-1 == ioctl(fd,VIDIOCSYNC,(count%2) ? &gb1.frame : &gb2.frame)) {
	perror("ioctl VIDIOCSYNC");
	exit(1);
    }
#endif
    /* done */
    if (!quiet)
	fprintf(stderr,"\n");
    close(fd);
    close(wtalk);
    wait(&s);
    return 0;
}



