 /*                                                                -*-c++-*-
    Copyright (C) 1996 Gregory D. Hager
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//-----------------------------------------------------------------------------
//*  Declaration of XVImage class.  XVImage provides a storage area for XVImages,
//*  allowing them to be refreshed and manipulated.
//*
//*  01/06/96  Gregory D. Hager
//*  Pretty complete rewrite and extension of the whole thing using 
//*  templates
//*
//*  For explanations of the individual member functions -- please
//*  refer to the Image class, which is the non-template version
//*  of XVImage.
//-----------------------------------------------------------------------------

#define MANUAL_INSTANTIATE
// manual instantiation is always defined
#ifndef XVImage_hh
#define XVImage_hh


#ifndef NO_INCLUDES
#include "Tools.hh"
#include "XVColorSchemes.hh"
// #include "Image.hh"
#endif // #ifndef NO_INCLUDES

//-----------------------------------------------------------------------------
//  XVImage class declaration
//-----------------------------------------------------------------------------

#define EPS 0.001


/* this stuff gets included only once........ */
#ifndef TEMPLATELINE1
#define TEMPLATELINE1 template <class XV_TP>
#define TEMPLATELINE2 template <class XV_TP1,class XV_TP2>
#define _ORDINAL_

void XVImage_panic(char* mess);
template<class T> class XVImage;

  template <class T> XVImage<T>& operator &= (XVImage<T>& x, const T y);
  template <class T> XVImage<T>& operator |= (XVImage<T>& x, const T y);
  template <class T> XVImage<T>& operator ^= (XVImage<T>& x, const XVImage<T>& y);
  template <class T> XVImage<T>& operator <<=(XVImage<T>& x, const int shift);
  template <class T> XVImage<T>& operator >>=(XVImage<T>& x, const int shift);
  template <class T> XVImage<T> operator ^ (const XVImage<T>& x, const XVImage<T>& y);//xor
  template <class T> XVImage<T> operator ~ (const XVImage<T>& x);      // one's complement


  template <class T> XVImage<T> square(XVImage<T>& x);

template <class T>
class XVImage {


  friend XVImage<T>& operator &=<T> (XVImage<T>& x, const T y);
  friend XVImage<T>& operator |=<T> (XVImage<T>& x, const T y);
  friend XVImage<T>& operator ^=<T> (XVImage<T>& x, const XVImage<T>& y);
  friend XVImage<T>& operator <<=<T>(XVImage<T>& x, const int shift);
  friend XVImage<T>& operator >>=<T>(XVImage<T>& x, const int shift);
  friend XVImage<T> operator ^ <T>(const XVImage<T>& x, const XVImage<T>& y);//xor
  friend XVImage<T> operator ~ <T>(const XVImage<T>& x);      // one's complement
  
protected:  
  int ncols, nrows;
  int storesize;
  T *image;
  void init (int rows, int columns);

public:
  src_color_scheme color_scheme;
  
  XVImage (int nrows_in, int ncols_in, const T* data);
  XVImage (int nrows_in = 0, int ncols_in = 0);
  XVImage (const XVImage<T> &im);
  // XVImage (const Image &im);
  ~XVImage() { delete image; }

  // Basic XVImage information

  int width()  const {return ncols;}
  int height() const {return nrows;}
  T *data()    const {return image;}
  T *operator[](int r) {return image + r*ncols;}
  T *operator[](int r) const {return image + r*ncols;}

  void resize(int rows, int cols);
  void reshape(int rows, int cols);

  // XVImage unary scalar operators

  XVImage<T>& operator =  (const T x);
  
  XVImage<T>  operator +  (const T x) const;
  XVImage<T>  operator -  (const T x) const;
  XVImage<T>  operator *  (const T x) const;
  XVImage<T>  operator /  (const T x) const;
  XVImage<T>& operator += (const T x);
  XVImage<T>& operator -= (const T x);
  XVImage<T>& operator *= (const T x);
  XVImage<T>& operator /= (const T x);

  // XVImage unary XVImage operators; these all operate elementwise
  XVImage<T>  operator -  () const;

  XVImage<T>& operator  = (const XVImage<T>& im);
  XVImage<T>& operator += (const XVImage<T>& x);
  XVImage<T>& operator -= (const XVImage<T>& x);
  XVImage<T>& operator *= (const XVImage<T>& x);
  XVImage<T>& operator /= (const XVImage<T>& x);

  // Other XVImage operations

  XVImage<T> Map(T (*)(T)) const;
  double ip(const XVImage<T> &x) const;            // XVImage inner product
  double sum() const;                            // Sum of values
  XVImage<T> &subimage (XVImage<T> &target, int ux, int uy) const;
  XVImage<T> subimage(int lx, int ly, int ux, int uy) const;
  XVImage<T> abs() const;

#if (XV_OS != XV_IRIX) 
  XVImage<T> sqr(XVImage<T>& x);  // works fine in all but irix
#endif

  friend XVImage<T> square<T>(XVImage<T>& x);
  
  // Operations for changing resolution.

  // These don't work for XVImage<int> - ADR
  XVImage<T> &scale(XVImage<T> &target, double sx, double sy) const;

  XVImage<T> &scale(XVImage<T> &target, double sc) const {
    return scale(target, sc, sc);
  }

  XVImage<T> &scale(XVImage<T> &target) const {
    return scale(target,
		 ((double)target.width())/((double)width())+EPS,
		 ((double)target.height())/((double)height())+EPS);
  }

  XVImage<T> &Shear(XVImage<T> &target, double sx) const;
  XVImage<T> &Shear_quad_v(XVImage<T> &target, double a, double b=0) const; 
  XVImage<T> &Shear_quad_h(XVImage<T> &target, double a, double b=0) const; 

  XVImage<T> &resample(XVImage<T> &target) const {
    return scale(target,
	  ((double)target.width())/width() + EPS,
	  ((double)target.height())/height() + EPS);
  }

  // These don't work for XVImage<int> - ADR
  XVImage<T> &reduce_resolution(int factorx, int factory, XVImage<T>& target) const;
  XVImage<T> &reduce_resolution(int factor, XVImage<T>& target) const {
    return reduce_resolution(factor,factor,target);
  }
  XVImage<T> reduce_resolution(int factorx, int factory) const {
    XVImage<T> temp;
    return reduce_resolution(factorx,factory,temp);
  }
  XVImage<T> reduce_resolution(int factor) const {
    return reduce_resolution(factor,factor);};
  XVImage<T> &magnify(int factorx, int factory, XVImage<T>& target) const;
  XVImage<T> &magnify(int factor, XVImage<T>& target) const {
    return magnify(factor,factor,target);
  }
  XVImage<T> magnify(int factorx, int factory) const {
    XVImage<T> temp(width()*factorx,height()*factory);
    return magnify(factorx,factory,temp);
  }
  XVImage<T> magnify(int factor) const {return magnify(factor,factor);}

  // Note: this function scales individual colours to lie in a certain
  // range.  This is bound to be confused with the other scale which
  // resizes an image.
  XVImage<T> scale(int minv=0, int maxv=255) const;
  T max() const;

  int write_XVImage(char *name);
  int write_byte_XVImage(char *name);
  int read_XVImage(char *name);
  int write_pgm(char* name, int doscale = 0) const;
  int write_ppm(char* name) const;
  int read_pgm(char* name);

  // for backward compatibility with old Image class:
  int write_image(char* name) { return write_XVImage(name); }
  int read_image(char* name) { return read_XVImage(name); }

  // FILTERING OPERATIONS
  // a few image processing routines (tab 7/21/97)
  XVImage<T> &convolve(XVImage<T> &targ, const XVImage<T> &mask, 
                       bool transpose = false) const;

  // Separable convolution
  XVImage<T> &convolve(XVImage<T> &targ, 
		       const XVImage<T> &maskx, 
		       const XVImage<T> &masky, 
                       bool transpose = false) const;

  // Box filtering in 1d

  XVImage<T> &erode(XVImage<T> &targ, const XVImage<T> &mask, 
                       bool transpose = false) const;
  void compute_gaussian_mask(double sigma_squared);
  XVImage<T> &gaussian_pyramid_1d(XVImage<T> &targ) const;
  XVImage<T> &gaussian_pyramid(XVImage<T> &targ, int iterations) const;

};

// XVImage binary operations



TEMPLATELINE2
inline int operator == (const XVImage<XV_TP1>& x, const XVImage<XV_TP2>& y)
{
  return ((x.height() == y.height()) && (x.width() == y.width()));
}

TEMPLATELINE2
inline int operator != (const XVImage<XV_TP1>& x, const XVImage<XV_TP2>& y)
{
  return (!(x == y));
}

#include "XVImage.icc"

// Filtering

TEMPLATELINE1 extern XVImage<XV_TP> &box_filter(XVImage<XV_TP> &targ,const XVImage<XV_TP> &src, int height, int width);
TEMPLATELINE1 extern XVImage<XV_TP> &double_box_filter_x(XVImage<XV_TP> &targ,const XVImage<XV_TP> &src, int width);
TEMPLATELINE1 extern XVImage<XV_TP> &prewitt_filter_dx(XVImage<XV_TP> &targ, const XVImage<XV_TP> &src);
TEMPLATELINE1 extern XVImage<XV_TP> &prewitt_filter_dx(XVImage<XV_TP> &targ, const XVImage<XV_TP> &src,int avgwidth);
TEMPLATELINE1 extern XVImage<XV_TP> &prewitt_filter_dy(XVImage<XV_TP> &targ, const XVImage<XV_TP> &src);
TEMPLATELINE1 extern XVImage<XV_TP> &prewitt_filter_dy(XVImage<XV_TP> &targ, const XVImage<XV_TP> &src,int avgwidth);


// Other operations

TEMPLATELINE1 inline XVImage<XV_TP> square(const XVImage<XV_TP>& x) {return x.Map(sqr);};
TEMPLATELINE1 inline XVImage<XV_TP> sqr(const XVImage<XV_TP>& x) {return square(x);};
TEMPLATELINE1 inline XVImage<XV_TP> abs(const XVImage<XV_TP>& x) {return x.abs();};
TEMPLATELINE1 inline XVImage<XV_TP> log(const XVImage<XV_TP>& x) {return x.Map(log);};
TEMPLATELINE1 inline XVImage<XV_TP> XVImage<XV_TP>::abs() const {return Map(::abs);};

#endif   // #ifndef TEMPLATELINE1
/* this ends the stuff that gets included only once........ */


TEMPLATELINE1 XVImage<XV_TP>  operator + (const XVImage<XV_TP>& x,const XVImage<XV_TP>& y);
TEMPLATELINE1 XVImage<XV_TP>  operator - (const XVImage<XV_TP>& x,const XVImage<XV_TP>& y);
TEMPLATELINE1 XVImage<XV_TP>  operator * (const XVImage<XV_TP>& x,const XVImage<XV_TP>& y);
TEMPLATELINE1 XVImage<XV_TP>  operator / (const XVImage<XV_TP>& x,const XVImage<XV_TP>& y);
TEMPLATELINE1 ostream&  operator << (ostream& s,const XVImage<XV_TP>& x);

TEMPLATELINE1 XVImage<XV_TP> transpose(const XVImage<XV_TP>&);
TEMPLATELINE1 XVImage<XV_TP> Transpose(const XVImage<XV_TP>&);


// dont try to integrate these into the library for now...
#if 0
TEMPLATELINE1 XVImage<XV_TP>& operator ^= (XVImage<XV_TP>& x, 
					   const XVImage<XV_TP>& y);
TEMPLATELINE1 XVImage<XV_TP>& operator <<=(XVImage<XV_TP>& x, const int shift);
TEMPLATELINE1 XVImage<XV_TP>& operator >>=(XVImage<XV_TP>& x, const int shift);
TEMPLATELINE1 XVImage<XV_TP> operator ^ (const XVImage<XV_TP>& x, 
					 const XVImage<XV_TP>& y); //xor
TEMPLATELINE1 XVImage<XV_TP> operator ~ (const XVImage<XV_TP>& x); // 1's complement
#endif


//-----------------------------------------------------------------------------
#endif // #ifndef XVImage_hh

