/*
    Copyright (C) 1996 Gregory D. Hager  (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//-----------------------------------------------------------------------------
//
//   XVImage.cc
//
//   Definition of XVImage member functions.
//
//   1-10-96
//   Complete rewrite with templates
//   Greg Hager     
//
//-----------------------------------------------------------------------------

#include <fstream.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include "XVImage.hh"
#include "Tools.hh"
#include "RGB2int.hh"

// This is how you can easily get the "instantiated" versions of the
// templates which are often useful for customization; Just redefine
// TEMPLATELINE to nothing and Txx to the desired type.  This
// Is also useful for compilers the don't support templates properly

#define TEMPLATELINE template <class Txx>

void XVImage_panic(char *mess)
{
  cerr << "XVImage class error: " << mess << endl;
  exit(1);
}

TEMPLATELINE
void XVImage<Txx>::init (int rows, int cols) {
  ncols = cols; nrows = rows;
  image = new Txx[storesize = (ncols * nrows)];
}

TEMPLATELINE
XVImage<Txx>::XVImage (int nrows_in, int ncols_in, const Txx* data) {
  init( nrows_in, ncols_in );
  if (data)
    memcpy(image, data, sizeof(Txx)*storesize);
}

TEMPLATELINE
XVImage<Txx>::XVImage (int nrows_in, int ncols_in) {
  init( nrows_in, ncols_in );
}

TEMPLATELINE
XVImage<Txx>::XVImage (const XVImage<Txx> &im) {

    ncols = im.ncols; nrows = im.nrows;
    image = new Txx[storesize = (ncols * nrows)];
    *this = im;
  }

/*
TEMPLATELINE
XVImage<Txx>::XVImage (const Image &im) {
  int i;
  int *x = im.data();
  
  storesize = 0;
  image = NULL;
  resize(im.height(),im.width());
  
  for (i=0;i<storesize;i++)
    image[i] = (Txx)x[i];
}
*/

TEMPLATELINE
void XVImage<Txx>::resize(int rows, int cols)
{
  if (rows*cols > 0)
    if (rows*cols > storesize) {
      if (image) delete image;
      image = new Txx[storesize = rows * cols];
    }

  nrows = rows;
  ncols = cols;
}

TEMPLATELINE
void XVImage<Txx>::reshape(int rows, int cols)
{
  assert ((nrows*ncols) == (rows*cols));

  nrows = rows;
  ncols = cols;
}


// XVImage scalar operations

TEMPLATELINE
XVImage<Txx>& XVImage<Txx>::operator= (const Txx x)
{
  Txx *counter = image,
    *endimage = counter + nrows*width();

  for (; counter < endimage; )
    *counter++ = x;

  return *this;
}

TEMPLATELINE
XVImage<Txx> XVImage<Txx>::operator + (const Txx y) const
{
  XVImage<Txx> temp(*this);

  temp += y;

  return temp;
}

TEMPLATELINE
XVImage<Txx> XVImage<Txx>::operator - (const Txx y) const
{
  XVImage<Txx> temp(*this);

  temp -= y;

  return temp;
}

TEMPLATELINE
XVImage<Txx> XVImage<Txx>::operator * (const Txx y) const
{
  XVImage<Txx> temp(*this);

  temp *= y;

  return temp;
}

TEMPLATELINE
XVImage<Txx> XVImage<Txx>::operator / (const Txx y) const
{
  XVImage<Txx> temp(*this);

  temp /= y;

  return temp;
}

TEMPLATELINE
XVImage<Txx> XVImage<Txx>::operator - () const
{
  XVImage<Txx> temp(height(),width());
  Txx *counter  = image, 
    *tcounter = temp.image,
    *endimage = image + nrows*width();

  for (; counter < endimage; ) *tcounter++ = - *counter++;

  return temp;
}

TEMPLATELINE
XVImage<Txx>& XVImage<Txx>::operator += (const Txx y)
{
  Txx *counter = image,
    *endimage = image + nrows*width();

  for (; counter < endimage; ) *counter++ += y;

  return *this;
}

TEMPLATELINE
XVImage<Txx>& XVImage<Txx>::operator -= (const Txx y)
{
  Txx *counter = image,
    *endimage = image + nrows*width();

  for (; counter < endimage; ) *counter++ -= y;

  return *this;
}

TEMPLATELINE
XVImage<Txx>& XVImage<Txx>::operator *= (const Txx y)
{
  Txx *counter = image,
    *endimage = image + nrows*width();

  for (; counter < endimage; ) *counter++ *= y;

  return *this;
}

TEMPLATELINE
XVImage<Txx>& XVImage<Txx>::operator /= (const Txx y)
{
  Txx *counter = image,
    *endimage = image + nrows*width();

  for (; counter < endimage; ) *counter++ /= y;

  return *this;
}



// Special cases to ensure backward compatibility w/Image class
/*
  TEMPLATELINE
  XVImage<Txx>& XVImage<Txx>::operator *= (double x)
  {
  Txx *counter = image;
  Txx *endimage = image + nrows*width();
  
  for (; counter < endimage; ) {*counter++ *= x;}
  return *this;
  }
  
  TEMPLATELINE
  XVImage<Txx> XVImage<Txx>::operator* (double y) 
  {
  XVImage<Txx>& temp = XVImage<Txx>(this.width(), this.height());
  Txx *counter = temp.image,
  *xcounter = this.image,
  *endimage = counter + temp.height()*temp.width();
  
  for (; counter < endimage; ) {*counter++ = *xcounter++ *y;}
  return temp;
  }
  
  TEMPLATELINE
  XVImage<Txx> XVImage<Txx>::operator/ (double x)
  {
  XVImage<Txx> output(this.width(),this.height());
  int *pf=this.data();
  int *enddata=pf+this.width()*this.height();
  int *pr=output.data();
  
  for (;pf < enddata;pf++) { *pr++=(int)((double)(*pf)/x); }
  return output;
  }
*/

// XVImage unary XVImage operators

TEMPLATELINE
XVImage<Txx>& XVImage<Txx>::operator= (const XVImage<Txx>& source)
{
  resize(source.height(), source.width());
  memcpy(image,source.image,sizeof(Txx)*height()*width());
  return *this;
}

TEMPLATELINE
XVImage<Txx>& XVImage<Txx>::operator += (const XVImage<Txx>& x)
{
  if (x != *this)
  {
    XVImage_panic("Wrong sized XVImages in XVImage self addition (+=) \n");
    exit(1);
  }

  Txx *counter = image, *endimage = image + nrows*width(),
    *counter1 = x.image;

  for (; counter < endimage; )
    *counter++ += *counter1++;

  return *this;
}

TEMPLATELINE
XVImage<Txx>& XVImage<Txx>::operator -= (const XVImage<Txx>& x)
{
  if (x != *this)
  {
    XVImage_panic("Wrong sized XVImages in XVImage self subtraction (-=) \n");
    exit(1);
  }

  Txx *counter = image, *endimage = image + nrows*width(),
    *counter1 = x.image;

  for (; counter < endimage; )
    *counter++ -= *counter1++;

  return *this;
}


TEMPLATELINE
XVImage<Txx>& XVImage<Txx>::operator *= (const XVImage<Txx>& x)
{
  if (x != *this)
  {
    XVImage_panic("Wrong sized XVImages in XVImage self multiplication (*=) \n");
    exit(1);
  }

  Txx *counter = image, *endimage = image + nrows*width(),
    *counter1 = x.image;

  for (; counter < endimage; )
    *counter++ *= *counter1++;

  return *this;
}

TEMPLATELINE
XVImage<Txx>& XVImage<Txx>::operator /= (const XVImage<Txx>& x)
{
  if (x != *this)
  {
    XVImage_panic("Wrong sized XVImages in XVImage self division (/=) \n");
    exit(1);
  }

  Txx *counter = image, *endimage = image + nrows*width(),
    *counter1 = x.image;

  for (; counter < endimage; )
    *counter++ /= *counter1++;

  return *this;
}

// XVImage/XVImage binary operators


TEMPLATELINE
XVImage<Txx> operator+ (const XVImage<Txx> &x,const XVImage<Txx> &y)
{
  if (x != y)
    XVImage_panic( "Wrong sized XVImages in XVImage binary addition");

  XVImage<Txx> temp(x);;

  return temp += y;
}


TEMPLATELINE
XVImage<Txx> operator- (const XVImage<Txx> &x,const XVImage<Txx> &y)
{
  if (x != y)
    XVImage_panic( "Wrong sized XVImages in XVImage binary subtraction");

  XVImage<Txx> temp(x);

  return temp -= y;
}

TEMPLATELINE
XVImage<Txx> operator* (const XVImage<Txx> &x,const XVImage<Txx> &y)
{
  if (x != y)
    XVImage_panic( "Wrong sized XVImages in XVImage binary multiplication");

  XVImage<Txx> temp(x);

  return temp *= y;
}

TEMPLATELINE
XVImage<Txx> operator/ (const XVImage<Txx> &x,const XVImage<Txx> &y)
{
  if (x != y)
    XVImage_panic( "Wrong sized XVImages in XVImage binary division");

  XVImage<Txx> temp(x);

  return temp /= y;
}

// Other XVImage operations


TEMPLATELINE
ostream& operator << (ostream& s, const XVImage<Txx>& x)
{
  int i,j;

  for (i = 0; i < x.height(); i++)
  {
    for (j = 0; j < x.width(); j++)
      cout << x[i][j] << " ";
    cout << "\n";
  }
  return s;
}

// Function to transpose an image;

TEMPLATELINE
XVImage<Txx> transpose(const XVImage<Txx>& f)
{
  int sd,td;
  int h=f.height();
  int w=f.width();
  XVImage<Txx> temp(w,h);
  
  for(sd=0;sd<h;sd++)
    for(td=0;td<w;td++)
      temp[td][sd]=f[sd][td];

  return temp;
}

TEMPLATELINE
XVImage<Txx> Transpose(const XVImage<Txx>& f)
{
  return transpose(f);
}

// General unary mapping function

TEMPLATELINE
XVImage<Txx> XVImage<Txx>::Map(Txx (*fn)(Txx)) const
{
  XVImage<Txx> temp(height(),width());
  Txx *pf=data(),
    *enddata=pf+width()*height(),
    *counter=temp.data();
  
  for (;pf < enddata;)
     *counter++=fn(*pf++);
  
  return temp;
}

TEMPLATELINE
double XVImage<Txx>::ip(const XVImage<Txx> &x) const
{

  if (!(x == *this))
    XVImage_panic ("Different size XVImages in operator ip\n");

  Txx  *counter = image;
  Txx *counter1 = x.image;
  Txx  *endimage = image + nrows*width();
  
  double sum = 0;

  for(; counter < endimage; ) sum += (*counter++ * *counter1++);

  return sum;
}


TEMPLATELINE
double XVImage<Txx>::sum() const
{

  Txx *counter = image,
      *endimage = image + nrows*width();
  double sum = 0;

  for(; counter < endimage; ) sum += *counter++;

  return sum;
}

TEMPLATELINE
XVImage<Txx> &XVImage<Txx>::subimage(XVImage<Txx> &target, int ux, int uy) const
{
  int i;

  for (i=0;i<target.height();i++)
    memcpy(target.image + i*target.width(),
	   image + (uy + i) * width() + ux,
	   sizeof(Txx)*(target.width()));

  return target;
}

TEMPLATELINE
XVImage<Txx> XVImage<Txx>::subimage(int lx, int ly, int ux, int uy) const
{
  if ((lx < 0) || (ly < 0) || (ux > width()) || (uy > height()))
    XVImage_panic("Attempt to acquire subimage outside the range of parent image.");
  XVImage<Txx> temp(uy-ly, ux-lx);
  int i;

  for (i=ly;i<uy;i++) {memcpy(temp[i-ly],(*this)[i]+lx,sizeof(Txx)*(ux-lx));}
  return temp;
}


// Changing XVImage size and resolution

TEMPLATELINE
inline double Inter1(Txx y1, Txx y2, double frac)
{ return y1+(y2-y1)*frac;
}


// An approximate shearing based on shifting without interpolation.

TEMPLATELINE
XVImage<Txx> &XVImage<Txx>::Shear(XVImage<Txx> &target, double sx) 
const
{
  if ((target.height() != height()) ||
      (target.width()+ round(height()*sx/2.0)) > width())
    XVImage_panic("Original too small in Shear \n");

  int i;
  double mid = (width()+1)/2.0;

  double startj = mid - target.width()/2.0 -
		   (target.height()/2.0)*sx +0.5;

  for (i=0;i<target.height();i++) {
    memcpy(target[i],((*this)[i])+((int)startj),
	   target.width()*sizeof(Txx)); 
    startj += sx;
  }

  return target;
}

TEMPLATELINE
XVImage<Txx> &XVImage<Txx>::Shear_quad_v(XVImage<Txx> &target, double a, double b) 
const
{
  if ((target.height() != height()) ||
      (target.width()+ fabs(a) + fabs(b)*(round(height())/2.0)) > width())
    XVImage_panic("Original too small in Shear_quad \n");

  int i;
  double mid = (width()+1)/2.0;
  double midh = (height()+1)/2.0;
  double maxd = ::sqr((double)(height()/2.0));
  int coord;

  for (i=0;i<target.height();i++) {

    coord = round(mid - (::sqr(i-midh)*a/maxd) + a/2.0 + (i-midh)*b - target.width()/2.0);
    memcpy(target[i],
	   ((*this)[i])+coord,
	   target.width()*sizeof(Txx)); 
  }

  return target;
}
TEMPLATELINE
XVImage<Txx> &XVImage<Txx>::Shear_quad_h(XVImage<Txx> &target, double a, double b) 
const
{
  XVImage<Txx> temp(target.height(),target.width());
  target = Transpose(Transpose(*this).Shear_quad_v(temp,a,b));

  return target;
}

TEMPLATELINE
XVImage<Txx> &XVImage<Txx>::scale(XVImage<Txx> &target, double sx, double sy) 
const
{
  double x,y;
  int i,j;

  if (((target.width()/sx) > width()) ||
      ((target.height()/sy) > height()))
    XVImage_panic("Original too small in scale \n");

  // Now, resample it back up.

  double midxo = (width()-1)/2.0;
  double midyo = (height()-1)/2.0;

  double midx = (target.width()-1)/2.0;
  double midy = (target.height()-1)/2.0;
  double skipx =1/sx;
  double skipy = 1/sy;
  double diff;
  Txx *r1,*r2,*tdata = target.data();
  int tw1 = target.width()-1;
  double *diffs = new double[target.width()];
  int *Js = new int[target.width()];

  x = -midx/sx + midxo;
  for(j=0; j<target.width(); j++)
    { 
      Js[j]= (int)x;
      diffs[j] = x-Js[j];
      x += skipx;
    }

// This is how you can easily get the "instantiated" versions of the
// templates which are often useful for customization; Just redefine
// TEMPLATELINE to nothing and Txx to the desired type.  This
// Is also useful for compilers the don't support templates properly
  y = -midy/sy + midyo;
  int iy;
  for(i=0; i<target.height(); i++) {

    r1 = (*this)[iy = (int)y];
    diff = y-iy;

    if (iy < (height() -1))
      r2 = (*this)[iy+1];
    else
      r2 = r1;

    for(j=0; j<tw1; j++)
      { 
	*tdata++=round(Inter1(Inter1(r1[Js[j]],r2[Js[j]],diff),
			      Inter1(r1[Js[j]+1],r2[Js[j]+1],diff),
			      diffs[j]));
      }

    if (Js[tw1]  >= (width()-1))
      *tdata++=round(Inter1(r1[width()-1],r2[width()-1],diff));
    else
      *tdata++=round(Inter1(Inter1(r1[Js[tw1]],r2[Js[tw1]],diff),
			    Inter1(r1[Js[tw1]+1],r2[Js[tw1]+1],diff),
			    diffs[tw1]));
    
    y += skipy;
  }

  delete Js;
  delete diffs;

  return target;
}

/*
TEMPLATELINE
XVImage<Txx> &XVImage<Txx>::Scalet(XVImage<Txx> &target, double sx, double sy)
{
  double x,y;
  int i,j;


  if (((target.width()/sx) > width()) ||
      ((target.height()/sy) > height()))
    XVImage_panic("Original too small in Scale \n");

  // Now, resample it back up.

  double midxo = (width()-1)/2.0;
  double midyo = (height()-1)/2.0;

  double midx = (target.width()-1)/2.0;
  double midy = (target.height()-1)/2.0;
  double skipx =1/sx;
  double skipy = 1/sy;
  double diff;
  Txx *r1,*r2;
  double *fr1,*fr2;
  int tw1 = target.width()-1;
  double *diffs = new double[target.width()];
  Txx *Js = new Txx[target.width()];
  double temp[target.width()*(*this).height()];
  double *tdata = temp;
  Txx *tdatai = target.data();

  x = -midx/sx + midxo;
  for(j=0; j<target.width(); j++)
    { 
      Js[j]= (int)x;
      diffs[j] = x-Js[j];
      x += skipx;
    }

  for(i=0; i<Original.height(); i++) {
    r1 = (*this)[i];
    for(j=0; j<tw1; j++)
      { 
	*tdata++=Inter1(r1[Js[j]],r1[Js[j]+1],diffs[j]);
      }

    if (Js[tw1]  >= (width()-1))
      *tdata++=r1[width()-1];
    else
      *tdata++=round(Inter1(r1[Js[tw1]],r1[Js[tw1]+1],diffs[tw1]);
  }

  y = -midy/sy + midyo;
  int iy;
  fr1 = temp;
  fr2 = temp+target.width();
  for(i=0; i<target.height(); i++, fr1 += target.width()) {
    diff = y-iy;
    if (iy < (height() -1))
      fr2 = fr1 + target.width();
    else
      fr2 = fr1;

    for(j=0; j<target.width; j++)
      { 
	*tdatai++=round(Inter1(r1[Js[j]],r2[Js[j]],diff));
      }
    y += skipy;
  }

  delete Js;
  delete diffs;
  delete temp;

  return target;
}
*/

TEMPLATELINE
XVImage<Txx> &XVImage<Txx>::reduce_resolution(int factorx, int factory, XVImage<Txx>& target) const
{
  target.resize(height()/factory,width()/factorx);

  int i;
  Txx dfactor = factorx * factory;
  Txx *tcount = target.data(), *counter = data(), *colc, *rowc;
  Txx *endcounter = counter + width() * height();
  memset(tcount,0,sizeof(Txx)*target.height()*target.width());

  while (counter < endcounter) {
    for (i=0;i<factory;i++) {
      for (rowc = counter + width(); counter < rowc;tcount++)
	for (colc = counter + factorx;counter < colc;)
	  *tcount += *counter++;
      tcount -= target.width();
    }
    tcount += target.width();
  }

  target /= dfactor;

  return target;
}
TEMPLATELINE
XVImage<Txx> &XVImage<Txx>::magnify(int factorx, int factory, XVImage<Txx>& target) const
{

  target.resize(height()*factory,width()*factorx);

  int i;
  Txx *target_im = target.data(), *counter = data(), *colc, *rowc;
  Txx *endtarget_im = target_im + target.width() * target.height();

  while (target_im < endtarget_im) {
    for (i=0;i<factory;i++) {
      for (rowc = counter + width(); counter < rowc;counter++)
	for (colc = target_im + factorx;target_im < colc;)
	  *target_im++ = *counter;
      counter -= width();
    }
    counter += width();
  }

  return target;
}


TEMPLATELINE
XVImage<Txx> XVImage<Txx>::scale(int minv,int maxv) const
{
  
  if (minv > maxv) 
    XVImage_panic("Incorrect arguments in XVImage::scale \n");

  Txx *counter = image,
      *endimage = image + nrows*width();
  Txx tmin, tmax;
  XVImage<Txx> tmp(*this);
  Txx *ptr = tmp.data();
  Txx *endptr = ptr + height()*width();

  tmin = tmax = *counter++;

  for(; counter < endimage; counter++) {
    if (*counter < tmin) tmin = *counter;
    if (*counter > tmax) tmax = *counter;
  }

  double range;

  if (tmin == tmax)

    range = 255;
  else
    range = tmax - tmin;

  double boost = minv - tmin;
  double factor = ((double)(maxv-minv))/range;

  for (;ptr<endptr;ptr++) {
    *ptr = (Txx)((*ptr + boost) *factor);
  }

  return tmp;
}
 
TEMPLATELINE
Txx XVImage<Txx>::max() const
{
  
  Txx *counter = image,
      *endimage = image + nrows*width();
  Txx tmax = *counter++;

  for(; counter < endimage; counter++) {
    if (*counter > tmax) tmax = *counter;
  }
  return tmax;
}
 



TEMPLATELINE
int XVImage<Txx>::write_XVImage(char *name)
    {
      ofstream XVImageout(name);
      if (!XVImageout) {
	cerr << "Cannot open file \"" << name << "\"\n";
	return 0;
      }
      else {
	XVImageout.write((char *)image,width()*height()*sizeof(Txx));
	XVImageout.close();
      }
      return 1;
    }
TEMPLATELINE
int XVImage<Txx>::write_byte_XVImage(char *name)
    {
      ofstream XVImageout(name);
      if (!XVImageout) {
	cerr << "Cannot open file \"" << name << "\"\n";
	return 0;
      }
      else {
 	XVImage<Txx> temp = scale();
	unsigned char *tempb = new unsigned char[width()*height()];

	for (int i=0;i<width()*height();i++) {
	  tempb[i] = (unsigned char)(((unsigned)temp.image[i]) & 0xff);
	}

	XVImageout.write((char *)tempb,width()*height());
	XVImageout.close();
	delete tempb;
      }
      return 1;
    }

// Read in an image;
TEMPLATELINE
int XVImage<Txx>::read_XVImage(char *name)
{ 
  ifstream XVImage_in(name,ios::in|ios::nocreate|ios::binary);
  if (!XVImage_in) {
    cerr << "Cannot open XVImage \"" << name << "\"\n";
    return 0;
  }
  else {
    XVImage_in.read((char*)data(),width()*height()*sizeof(Txx));
    XVImage_in.close();
  }
  return 1;
}

TEMPLATELINE
int XVImage<Txx>::write_pgm(char *name, int doscale) const
    {
      ofstream imageout(name);
      if (!imageout) {
	cerr << "Cannot open file \"" << name << "\"\n";
	return 0;}

      imageout << "P5" << endl <<  width() <<  " " << height() << endl <<
	"255" << endl;

      unsigned char *tempb = new unsigned char[width()*height()];

      if (doscale) {
	XVImage<Txx> temp = scale();
	for (int i=0;i<width()*height();i++) {
	  tempb[i] = (unsigned char)(((unsigned)temp.image[i]) & 0xff);
	}
      }
      else
	for (int i=0;i<width()*height();i++) {
	  tempb[i] = (unsigned char)(((unsigned)image[i]) & 0xff);
	}

      imageout.write((char *)tempb,width()*height());
      imageout.close();
      delete tempb;

      return 1;
    }

TEMPLATELINE
int XVImage<Txx>::write_ppm(char *name) const
    {
      ofstream imageout(name,ios::out | ios::binary);
      if (!imageout) {
	cerr << "Cannot open file \"" << name << "\"\n";
	return 0;}

      imageout << "P6" << endl <<  width() <<  " " << height() << endl <<
	"255" << endl;

      //      XVImage<Txx> temp = scale();
      unsigned char *tempb = new unsigned char[width()*height()*3];
      for (int i=0,j=0;i<width()*height();i++) {
	tempb[j++] = compute_red((int)image[i]);
	tempb[j++] = compute_green((int)image[i]);
	tempb[j++] = compute_blue((int)image[i]);
      }

      imageout.write((char *)tempb,width()*height()*3);
      imageout.flush();
      imageout.close();
      delete tempb;

      return 1;
    }

TEMPLATELINE
int
XVImage<Txx>::read_pgm(char *name)
{
  ifstream imagein(name, ios::in | ios::binary);
  int height, width, npix, bits, nfound;
  unsigned char *imbuff;
  int found = 0;
  char buff[80];

  if ( !imagein) {
    cerr << "Cannot open file " << name << endl;
    return 0;}

  // Leaf through the comments looking for "P5"
  // which is the only thing this brain-damaged implementation
  // can deal with.

  while ((!found) && (imagein.getline(buff,80) != NULL))
    found = (strstr(buff,"P5") != NULL);

  if (!found) {
    cerr << "No P5 marker found in " << name << endl;
    return 0;
  }

    // Skip over comments
  while ((imagein.getline(buff,80) != NULL) && (buff[0] == '#'));

    // Assume the next line is the width/height thing
  nfound = sscanf(buff,"%d %d %d",&width,&height,&bits);
  if (nfound < 2) {
    cerr << "Error opening pgm image" <<  name << endl;
    panic(""); }
  else {
    if (nfound < 3)              // assume 255 on next line
      imagein.getline(buff,80); }

    // Make the image big enough;
  resize(height,width);
    // Allocate local buffer
  imbuff = new unsigned char[width*height];
    // Get the data
  npix = imagein.tellg();
    // Change -- 8/8/96 (ZD) added cast to appease angry C++ spirits
  imagein.read((char*)imbuff,width*height);
  npix = imagein.tellg()-npix;
  if (npix != (width*height)) {
    cerr << "Warning: file read truncated before image was finished!" << endl;
    cerr << npix << " of " << width*height << " pixels read." << endl;
  }

  Txx *p=data();
  for (int i=0; i<npix;i++) {*p++ = (Txx) imbuff[i];}
  delete imbuff;
  imagein.close();
  return 1;
}



#if 0
/* Derivative in x direction */

TEMPLATELINE
XVImage<Txx> &Dx(const XVImage<Txx>& imag,XVImage<Txx>& targ)
{

  if (((imag.width() -1) > (targ.width())) ||
      (imag.height() > (targ.height())))
    XVImage_panic ("Target too small in Dx \n");
 
  Txx *counter, *icounter, *endrow;
  int i;

  icounter = imag.data();
  counter = targ.data();
  for (i = 0; i < imag.height(); i++)
  {
    endrow = icounter + imag.width() - 1;
    for (; icounter < endrow; icounter++)
      *counter++ = *(icounter + 1) - *icounter;
    icounter++;
  }

  return targ;
}


/* Derivative in y direction */

TEMPLATELINE
XVImage<Txx> &Dy(const XVImage<Txx>& imag,XVImage<Txx> &targ)
{
  Txx *counter, *icounter, *endimage;

  if ((imag.width() > (targ.width())) ||
      ((imag.height()-1) > (targ.height())))
    XVImage_panic ("Target too small in compressy \n");

  
  icounter = imag.data();
  counter  = targ.data();
  endimage = counter + targ.width() * targ.height();
  
  for (; counter < endimage; icounter++)
    *counter++ = *(icounter + targ.width()) - *icounter;
 
  return targ;
}


/* Compress for one position in x direction */

TEMPLATELINE
XVImage<Txx> &compressx(const XVImage<Txx> &imag,XVImage<Txx> &targ)
{ 
  Txx *counter, *icounter, *endrow;
  int i;

  if (((imag.width()-1) > targ.width()) ||
      (imag.height() > (targ.height())))
    XVImage_panic ("TargeTxx too small in compressx \n");

  
  icounter = imag.data();
  counter = targ.data();
  for (i = 0; i < targ.height(); i++)
    { 
      endrow = icounter + targ.width();
      for (; icounter < endrow; icounter++)
	*counter++ = (*(icounter + 1) + *icounter) / 2;
      icounter++;
    }
  return targ;
}

/* Same thing in y direction */

TEMPLATELINE
XVImage<Txx> &compressy(const XVImage<Txx> &imag, XVImage<Txx> &targ)
{ 
  Txx *counter, *icounter, *endimage;
  
  if ((imag.width() > (targ.width())) ||
      ((imag.height()-1) > targ.height()))
    XVImage_panic ("Target too small in compressy \n");

  icounter = imag.data();   
  counter  = targ.data();
  endimage = counter + targ.height() * targ.width();
  
  for (; counter < endimage; icounter++)
    *counter++ = (*(icounter + targ.width()) + *icounter) / 2;
  
  return targ;
}

/* Derivative in x direction */

TEMPLATELINE
XVImage<Txx> Dx(const XVImage<Txx>& imag)
{
  XVImage<Txx> temp(imag.height(), imag.width()-1);

  return Dx(imag,temp);
}


/* Derivative in y direction */

TEMPLATELINE
XVImage<Txx> Dy(const XVImage<Txx>& imag)
{
  XVImage<Txx> temp(imag.height()- 1, imag.width());
  return Dy(imag,temp);
}


/* Compress for one position in x direction */

TEMPLATELINE
XVImage<Txx> compressx(const XVImage<Txx> &imag)
{ 
  XVImage<Txx> temp(imag.height(), imag.width() -1);
  return compressx(imag,temp);
}

/* Same thing in y direction */

TEMPLATELINE
XVImage<Txx> compressy(const XVImage<Txx> &imag)
{
  XVImage<Txx> temp(imag.height() -1 , imag.width());
  return compressy(imag,temp);
}

            
/* These 2 just ensure that the XVImage<Txx>is again square afterwards... */

TEMPLATELINE
XVImage<Txx> smoothDx(const XVImage<Txx>& image)
{
  return compressy(Dx(image));
}

TEMPLATELINE
XVImage<Txx> smoothDy(const XVImage<Txx>& image)
{
  return compressx(Dy(image));
}

TEMPLATELINE
XVImage<Txx> &smoothDx(const XVImage<Txx>& image,XVImage<Txx> &targ)
{
  return compressy(Dx(image),targ);
}

TEMPLATELINE
XVImage<Txx> &smoothDy(const XVImage<Txx>& image,XVImage<Txx> &targ)
{
  return compressx(Dy(image),targ);
}


#endif


#define MASK_SIZE 5
#define MASK { 0.05, 0.25, 0.4, 0.25, 0.05 }
#define REDUCTION_FACTOR 2

TEMPLATELINE
XVImage<Txx> &
XVImage<Txx>::gaussian_pyramid_1d(XVImage<Txx> &targ) const
{
  static double mask[MASK_SIZE] = MASK;
  Txx *ip = data();
  int window_mid = (MASK_SIZE - 1) / 2;
  int wid = width() - (width() % REDUCTION_FACTOR),
      ht = height();
  register double temp;

  if ((targ.height() != width() / REDUCTION_FACTOR) ||
      (targ.width() != height())) {

    cout << width() << "x" << height() << " -> "
         << targ.width() << "x" << targ.height() << endl;

    XVImage_panic("Wrong dimensions in gaussian_pyramid");
  }

  for (int i = 0; i < ht; i++) {
    ip = data() + i * width();

    for (int j = 0; j <wid; 
         j += REDUCTION_FACTOR, ip += REDUCTION_FACTOR) {
 

      //      targ[j/REDUCTION_FACTOR][i] = 0;
      temp = 0;

      for (int k = max2(window_mid-j, 0); 
           k < MASK_SIZE; k++) {

	//     targ[j/REDUCTION_FACTOR][i] += mask[k] * ip[k - window_mid];
	temp += mask[k] * ip[k - window_mid];

      }
      targ[j/REDUCTION_FACTOR][i] = (Txx) temp;
    }
  }

  return targ;
}

TEMPLATELINE
XVImage<Txx> &XVImage<Txx>::gaussian_pyramid(XVImage<Txx> &targ, int iterations) const
{
  int wid = width(), ht = height();

  targ = *this;

  for (int i = 0; i < iterations; i++) {
    XVImage<Txx> temp(wid/2, ht);
    
    targ.gaussian_pyramid_1d(temp);
    targ.resize(ht/2, wid/2);
    temp.gaussian_pyramid_1d(targ);
    wid /= 2;
    ht /= 2;
  }
  
  return targ;
}

//* Does a zero-padded convolution --- preserves position for odd sized
//* masks, but offsets by half a pixel for even sized masks.

TEMPLATELINE
XVImage<Txx> &
XVImage<Txx>::convolve(XVImage<Txx> &targ, const XVImage<Txx> &mask, bool transpose)
const
{
  int mask_width = mask.width(),
      mask_height = mask.height();
  Txx   *op = targ.data();
  register double temp;
  int dl = mask_width / 2;
  int dk = mask_height / 2;
  int ul = mask_width - dl;
  int uk = mask_height - dk;
  int i,j,k,l;

  double norm = (mask.abs()).sum();
  assert(norm > 0);

  if (!transpose) {
    targ.resize(height(),width());
  } else
    targ.resize(width(),height());

  if (!transpose) {

    for ( i = 0;i < dk;i++) 
      for ( j = 0; j < width(); j++) *op++ = 0;  // zero pad top

    for (i=dk;i<= height()-uk;i++) {
      for ( j = 0; j < dl; j++) *op++ = 0;  // zero pad left
    
      for ( j = dl; j <= width()-ul; j++) {   //  real convolution
	//        *op = 0;
	temp = 0;
        for ( k = i - dk; k < i + uk; k++)
          for ( l = j - dl; l < j + ul; l++) 
            temp += (*this)[k][l] * mask[k+dk-i][l+dl-j];
	*op++ = (Txx)(temp/norm);
      }

      for ( j = width()-ul+1; j < width(); j++) *op++ = 0;  // zero pad right
    }

    for ( i = height()-uk+1;i < height();i++) 
      for ( j = 0; j < width(); j++) *op++ = 0;  // zero pad bottom
      
  }

  else {
    /*
    for ( j = 0;j < dk;i++) 
      for ( j = 0; j < width(); j++) *op++ = 0;  // zero pad top

    for (i=dk;i<= height()-uk;i++) {
      for ( j = 0; j < dl; j++) *op++ = 0;  // zero pad left
    
      for ( j = dl; j <= width()-ul; j++) {   //  real convolution
        *op = 0;
        for ( k = i - dk; k < i + uk; k++)
          for ( l = j - dl; l < j + ul; l++) 
            *op += (*this)[k][l] * mask[k+dk-i][l+dl-j];
	*op++ = *op/norm;
      }

      for ( j = width()-ul+1; j < width(); j++) *op++ = 0;  // zero pad right
    }

    for ( i = height()-uk+1;i < height();i++) 
      for ( j = 0; j < width(); j++) *op++ = 0;  // zero pad bottom
      */
    cout << "Transpose on convolution still needs to be fixed!" << endl;
#warning "Transpose on convolution still needs to be fixed!" 

    /*    for (int j = 0; j < width(); j++)
      for (int i = 0; i < height(); i++) {
        *op = 0;
        for (int k = max2(i - dk,0); k <= min2(height()-1,i + dk); k++)
          for (int l = max2(j - dl,0); l <= min2(j+dl,width()-1) ; l++)
            *op += (*this)[k][l] * mask[i-k+dk][j-l+dl];
	*op++ = (int)rint(*op/norm);
      }*/

  }

  return targ;

}

TEMPLATELINE
XVImage<Txx> &
XVImage<Txx>::convolve(XVImage<Txx> &targ, 
		       const XVImage<Txx> &maskx, 
		       const XVImage<Txx> &masky, 
		       bool transpose)
const
{
  XVImage<Txx> tmp(targ.height(),targ.width());

  convolve(tmp,maskx,transpose);
  return tmp.convolve(targ,masky,transpose);
}


TEMPLATELINE
XVImage<Txx> &
XVImage<Txx>::erode(XVImage<Txx> &targ, const XVImage<Txx> &mask, bool transpose)
const
{
  int mask_width = mask.width(),
      mask_height = mask.height();
  Txx   *op = targ.data();
  int temp = 0;
  int dl = (mask_width - 1) / 2;
  int dk = (mask_height - 1) / 2;

  if (!transpose) {
    targ.resize(height(),width());
  } else
    targ.resize(width(),height());

  if (!transpose)

    for (int i = 0; i < height(); i++)
      for (int j = 0; j < width(); j++) {
        temp = 1;
        for (int k = max2(i - dk,0); k <= min2(height()-1,i + dk); k++)
          for (int l = max2(j - dl,0); l <= min2(j+dl,width()-1) ; l++)
	    temp &= (((*this)[k][l] != 0) | (mask[i-k+dk][j-l+dl] == 0));
        *op++ = temp;
      }

  else

    for (int j = 0; j < width(); j++)
      for (int i = 0; i < height(); i++) {
        temp = 1;
        for (int k = max2(i - dk,0); k <= min2(height()-1,i + dk); k++)
          for (int l = max2(j - dl,0); l <= min2(j+dl,width()-1) ; l++)
	    temp &= (((*this)[k][l] != 0) | (mask[i-k+dk][j-l+dl] == 0));
        *op++ = temp;
      }

  return targ;

}

// THis is wrong!! should probably do double precision???

TEMPLATELINE
void XVImage<Txx>::compute_gaussian_mask(double sigma_squared)
{
  int xmid = (width() - 1) / 2,
      ymid = (height() - 1) / 2;

  for (int i = 0; i <= ymid; i++)
    for (int j = 0; j <= xmid; j++)
      (*this)[i][width() - 1 - j] =
        (*this)[height() - 1 - i][j] =
        (*this)[height() - 1 - i][width() - 1 - j] =
        (*this)[i][j] = (Txx)
        exp(-((i-ymid)*(i-ymid) + (j-xmid)*(j-xmid))/(2*sigma_squared));
}

#undef  TEMPLATELINE1
#undef TEMPLATELINE2
#define TEMPLATELINE1 template 
#define TEMPLATELINE2 template 
#define XV_TP int
#define XV_TP1 int
#define XV_TP2 int
#undef XVImage_hh
template class XVImage<XV_TP>;
#include "XVImage.hh"
#define XV_TP short
#define XV_TP1 short
#define XV_TP2 short
#undef XVImage_hh
template class XVImage<XV_TP>;
#include "XVImage.hh"
#define XV_TP  char
#define XV_TP1 char
#define XV_TP2 char
#undef XVImage_hh
template class XVImage<XV_TP>;
#include "XVImage.hh"
#define XV_TP  float
#define XV_TP1 float
#define XV_TP2 float
#undef XVImage_hh
template class XVImage<XV_TP>;
#include "XVImage.hh"
