/*                                                                -*-c++-*-
    Copyright (C) 1993 Gregory D. Hager and Sidd Puri (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.
*/
//---------------------------------------------------------------------
//*
//* Matrix.hh
//*
//* 02/20/95 Gregory D. Hager
//* 
//* Header file for a Matrix package for small systems
//* A simple matrix package for small applications.  Matrices are stored in
//* row-major order using a ragged-array data structure.  Submatrices are
//* also supported as are row and column vectors.
//*
//* Bounds checking and size checking are also included by default.
//* Defining the NO_BOUNDS_CHECK turns off bounds checking.
//* Defining check_size turns off size checking on matrix operations.
//* Check in Matrix.cc at the start of the file to see whether
//* NO_BOUNDS_CHECK or CHECKSIZE are defined as you would like them to be.
//*
//* The SUBSCRIPT_START_WITH_1 preprocessor variable, if defined,
//* allows you to index these structures from 1 to n, rather than from
//* 0 to n-1.
//*
//* Matrices should be templates, but for some reason templates aren't
//* working so I'll just hardcode the type for now.
//*
//* This header file contains the definitions for the classes
//* _rowvec, RowVector, ColumnVector, Matrix, and DiagonalMatrix
//*
//---------------------------------------------------------------------

//#define SUBSCRIPT_START_WITH_1

#ifndef FR_Matrix_hh
#define FR_Matrix_hh

#include <stdio.h>
#include <stdlib.h>
#include <iostream.h>
#include <string.h>

#include "Image.hh"      // Comment these out for a stand-alone matrix package
#include "XVImage.hh"

// Next two #defines used to be in Matrix.cc, but should be here.
#define NO_BOUNDS_CHECK
#define CHECKSIZE(m,msg) /* do nothing */

#define FR_EPSFCN 1.0E-16       // FrReal is the type of the matrices
#define FrReal double           // FR_EPSFCN is the minimum available value

extern void _panic(char *);     // For simple exception handling.

typedef struct {                // Data structure for reference counts
  FrReal *data;
  int refCount;
} RefCounter;

/// Redefine the macro CHECKSIZE to turn off size checking
#ifndef CHECKSIZE
#define CHECKSIZE(m,mess) \
{if ((m.rowNum != rowNum) || (m.colNum != colNum)) _panic(mess);}
#endif

/// the class _rowvec
/// is a data structure used to implement checking on 2nd matrix index.
/// A matrix is essentially a column vector of _rowvecs
class _rowvec
{
private:
  FrReal *data;
  int colNum;

public:

  _rowvec(FrReal *datain,int ncols) {data = datain; colNum = ncols;}

  FrReal&
  operator[] (int n) {
    if ((n < 0) || (n >= colNum))
      _panic("Matrix second argument out of range");
    return data[n];
  }

  const FrReal&
  operator[] (int n) const {
    if ((n < 0) || (n >= colNum))
      _panic("Matrix second argument out of range");
    return data[n];
  }

};

class RowVector;
class ColVector;
class Matrix;
class DiagonalMatrix;


// The Matrix class definition starts here
//---------------------------------------------------------------------
//*
//* Matrix.hh
//*
//* 02/20/95 Gregory D. Hager
//* 
//* Header file for a Matrix package for small systems
//* A simple matrix package for small applications.  Matrices are stored in
//* row-major order using a ragged-array data structure.  Submatrices are
//* also supported as are row and column vectors.
//*
//* Bounds checking and size checking are also included by default.
//* Defining the NO_BOUNDS_CHECK turns off bounds checking.
//* Defining check_size turns off size checking on matrix operations.
//* Check in Matrix.cc at the start of the file to see whether
//* NO_BOUNDS_CHECK or CHECKSIZE are defined as you would like them to be.
//*
//* The SUBSCRIPT_START_WITH_1 preprocessor variable, if defined,
//* allows you to index these structures from 1 to n, rather than from
//* 0 to n-1.
//*
//* Matrices should be templates, but for some reason templates aren't
//* working so I'll just hardcode the type for now.
//*
//* This header file contains the definitions for the classes
//* _rowvec, RowVector, ColumnVector, Matrix, and DiagonalMatrix
//*
//---------------------------------------------------------------------

class Matrix
{
  friend class RowVector;
  friend class ColVector;
  friend class DiagonalMatrix;
  friend Matrix add_column(Matrix &x, const ColVector &c);

private:
    /// this creates a numerical approximation to the derivative at x
    /// using the function values yi and the horizontal step dstep
  static FrReal deriv_from_pts(FrReal y1, FrReal y2, FrReal y3,
			       FrReal dstep, FrReal x);

    /// this takes the (numerical approximation to the) derivative of
    /// the function *fn and puts it into "deriv"
  static void deriv_n(ColVector (*fn)(const ColVector&, const ColVector&),
	 const ColVector& val, int n, const ColVector& extra, ColVector& deriv);

    /// Used for submatrix operations
  int init(Matrix &m,int startr, int startc, int nrows, int ncols);

  // initialise all fields to 0
  void init_empty(void);

protected:
    /// Reference count data structure.  This structure is shared
    /// with all other matrices using this storage.
  RefCounter *refPtr;
  int rowNum;
  int colNum;

    /// These are the data pointers
  FrReal *data;
  FrReal **rowPtrs;

    /// Total storage size --  by default, we allocate double the
    /// needed storage to allow room for resizing.
  int csize;
    /// Size of the matrix now (rowNum * colNum)
  int dsize;
    /// Total Row Space
  int trsize;

    /// These are functions to reference and unreference shared data.
  RefCounter *ref();
  void unref();

    /// Return value == is my data shared with someone?
  int dataShared();

    /// submatrix constructor  
  Matrix(Matrix &m, int startr, int startc, int nrows, int ncols);


public:
    /// All of the public matrix constructors and destructors
  Matrix();
  Matrix(const Matrix& m);
  Matrix(int rr,int cc);
  Matrix(int rr,int cc,FrReal* x);
  virtual ~Matrix ();
    /// Accessor functions for the row and column number
  int n_of_rows() const;
  int n_of_cols() const;

    /// Now resize () is used only in constructors. Maybe it should be used in
    /// in destructive assignment (or <<, which is not implemented yet).
  int resize(int nrows, int ncols);

    /// All of the next 10  matrix operators return references to the calling
    /// Matrix, which is altered as documented here. The following ones return
    /// copies of newly constructed matrices. The operators which take 
    /// This operator= sets all of the elements to x
  Matrix& operator=(FrReal x);
  Matrix& operator=(const Matrix &m);
  Matrix& operator<<(const Matrix &m);
    /// This assigns the matrix elements, row-dominant, from an array
  Matrix& operator<<(FrReal*);
  Matrix& operator+=(FrReal x);
  Matrix& operator+=(const Matrix &mat);
  Matrix& operator-=(FrReal x);
  Matrix& operator-=(const Matrix &mat);
  Matrix& operator*=(FrReal x);
  Matrix& operator/=(FrReal x);
  Matrix operator*(FrReal x) const;
  Matrix operator*(const Matrix &mat) const;
  ColVector operator*(ColVector &mat) const;
  RowVector operator*(RowVector &mat) const;
  Matrix operator/(FrReal x) const;
  Matrix operator+(const Matrix &mat) const;
  Matrix operator-(const Matrix &mat) const;
  Matrix operator-() const;
    /// returns the Sum of the Squares of the elements
  FrReal SumSquare() const;
    /// returns the Sum of the elements
  FrReal Sum() const;

    /// These are destructive solution operations.
#ifndef NO_BOUNDS_CHECK
  _rowvec operator[](int n);
#else
  FrReal* operator[](int n);
#endif

    /// These functions return copies of specified rows & cols
  FrReal* operator[](int n) const;
  RowVector row(int i);
  RowVector Row(int i);
  ColVector col(int j);
  ColVector Column(int j);

#ifdef SUBSCRIPT_START_WITH_1  
    /// If these functions appear, then SUBSCRIPT_START_WITH_1 is defined
  Matrix operator()(int sr, int lr, int sc, int lc);
  Matrix Rows(int first_row, int last_row);
  Matrix Columns(int first_col, int last_col);
        
#else
    /// If these functions appear, then SUBSCRIPT_START_WITH_1 is not defined
  Matrix operator()(int sr, int lr, int sc, int lc);
  Matrix Rows(int first_row, int last_row);
  Matrix Columns(int first_col, int last_col);

#endif // SUBSCRIPT_START_WITH_1
 
    /// Inverse, transpose, inner product and outer product.
  Matrix i() const;
  Matrix t() const;
  Matrix ip() const;
  Matrix op() const;

    /// The following functions are both desctuctive and nondestructive
    /// Destructive LU decomposition
  void LUDcmp(int* perm, int& d);
    /// Destructive backsubstitution
  void LUBksb(int* perm, ColVector& b);
    /// Nondestructive solving Ax = B
  void solveByLUD(const ColVector& b, ColVector& x);
    /// Nondestructive x = A.LUDsolve(B), equivalent to solveByLUD
  ColVector LUDsolve(const ColVector& b);
    /// Does L D Ltranspose decomposition
  void LDLtDcmp();
    /// Matrix square root
  Matrix sqrt();
    /// Destructive singular value decomposition
  void SVDcmp(ColVector& w, Matrix& v);
    /// Destructive singular value backsubstitution
  void SVBksb(const ColVector& w,const Matrix& v,
	      const ColVector& b, ColVector& x);
    /// Nondestructive solving of Ax = B
  void solveBySVD(const ColVector& b, ColVector& x);
    /// Nondestructive equivalent to above
  ColVector SVDsolve(const ColVector& b);
    /// creates a jacobian matrix of the first argument at "where"
    /// The jacobian has odim (output dimension) rows and
    /// as many columns as "where"
  static Matrix jacobian(ColVector (*fn)(const ColVector&, const ColVector&),
			 int odim, const ColVector& where, const ColVector& extra);
    /// prints out the elements of the matrix "m"
  friend ostream &operator << (ostream &s,const Matrix &m);
    /// This takes the function "*fn" and performs it to each element
    /// of the invoking matrix and puts the resulting elements into the
    /// returned matrix.
  Matrix Map(FrReal (*fn)(FrReal)) const;

  Matrix add_column(const ColVector &c) const;

};



/// RowVector class
/// For specifics on the member functions here, see the Matrix class
/// from which this is derived. Basically, RowVector is a horizontal
/// matrix, while ColVector is a vertical matrix.
class RowVector : public Matrix
{
  friend class Matrix;

protected:
  RowVector(Matrix &m, int i);

public:
  RowVector();
  RowVector(int nn);
  RowVector(const RowVector &v);

  void resize(int i);
  FrReal &operator [](int n);
  const FrReal &operator [](int n) const;
  RowVector &operator=(const RowVector &v);
  RowVector &operator<<(const RowVector &v);
  RowVector &operator=(const Matrix &m);
  RowVector &operator=(FrReal x);
  ColVector t() const;
};



/// ColVector class
/// For specifics on the member functions here, see the Matrix class
/// from which this is derived. Basically, RowVector is a horizontal
/// matrix, while ColVector is a vertical matrix.
class ColVector : public Matrix
{
  friend class Matrix;

protected:
  ColVector (Matrix &m, int j);
  ColVector (ColVector &m, int startr, int nrows);

public:

  ColVector();
  ColVector(int nn);
  ColVector (const ColVector &v);
  ~ColVector();

  void resize(int i);
  FrReal &operator [](int n);
  const FrReal &operator [](int n) const;
  ColVector &operator=(const ColVector &v);
  ColVector &operator<<(const ColVector &v);
  ColVector &operator<<(FrReal *);
  ColVector &operator=(const Matrix &m);
  ColVector &operator=(FrReal x);

  // J.W. added the below Aug 15th
  
  ColVector operator+(const ColVector &mat) const;
  ColVector operator-(const ColVector &mat) const;
  ColVector operator*(FrReal x) const;
  ColVector operator-();

  ColVector Rows(int first_row, int last_row);
  RowVector t() const;
  FrReal ip() const;
  FrReal ip(const ColVector &) const;
};

/// SVD(M, Md, U, V), where M=U.t * Md * V
extern void SVD(Matrix& , Matrix&, Matrix&, Matrix& );

/// For matrix operations used directly on images
#ifdef XVImage_hh
ColVector &operator << (ColVector &x,const Image& y);
RowVector &operator << (RowVector &x,const Image& y);
Image     &operator >>(const ColVector &x, Image &y);
Image     &operator >>(const RowVector &x, Image &y);

template <class T> ColVector  &operator <<(ColVector &x,const XVImage<T> &y);
template <class T> RowVector  &operator <<(RowVector &x,const XVImage<T> &y);
template <class T> XVImage<T> &operator >>(const ColVector &x,XVImage<T> &y);
template <class T> XVImage<T> &operator >>(const RowVector &x,XVImage<T> &y);

#endif





/// Diagonal Matrix Class
/// A diagonal matrix is basically a ColVector 
/// with support to multiply by ordinary matrices in the appropriate
/// fashion. All of the member functions are explained in the Matrix
/// class (most are just adapted versions of those from the Matrix
/// class anyway).
class DiagonalMatrix
{
private:  
  ColVector t;

public:
  DiagonalMatrix(int n= 0);
  DiagonalMatrix(const ColVector &tin);

  int n_of_cols() const;
  int n_of_rows() const;
  ColVector& diagonal();
  void resize(int n);

  FrReal &operator() (int n);
  FrReal operator() (int n) const;
  DiagonalMatrix &operator = (const DiagonalMatrix &m);
  DiagonalMatrix &operator = (FrReal x);

  friend Matrix operator*(const DiagonalMatrix &, const Matrix &);
  friend RowVector operator*(const DiagonalMatrix &, const RowVector &);
  friend Matrix operator*(const Matrix&, const DiagonalMatrix &);
  friend DiagonalMatrix operator*(const DiagonalMatrix &x,const DiagonalMatrix &y);
  friend ostream &operator << (ostream &,const DiagonalMatrix &);
};


//----------------------------------------------------------------
// utilities
//----------------------------------------------------------------

FrReal invs(FrReal x);

Matrix add_column(Matrix &x, const Image &I);
template <class T>
Matrix add_column(Matrix &x, const XVImage<T> &I);


#include "Matrix.icc"

#endif




