/*                                                                -*-c++-*-
    Copyright (C) 1993 Gregory D. Hager and Sidd Puri (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.
*/
//---------------------------------------------------------------------
//*
//* Matrix.cc
//*
//*  originally by Chien-Ping Lu
//*  Bug fix by Jonathan Wang (09/04/95)
//*
//---------------------------------------------------------------------

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <assert.h>

#include "Matrix.hh"

void _panic(char *mess)
{
  cerr << mess << endl;
  exit(1);
}

Matrix::Matrix (const Matrix& m) 
{
  init_empty();
  resize(m.rowNum,m.colNum);
  for (int i=0; i<rowNum; i++) {
    for (int j=0; j<colNum; j++) {
      rowPtrs[i][j] = m.rowPtrs[i][j];
    }
  }
}

int
Matrix::resize(int nrows, int ncols) 
{
  // If things are the same, do nothing and return
  
  if ((nrows == rowNum) && (ncols == colNum)) {return 1;}
  rowNum = nrows; colNum = ncols;
  
  if (dataShared())
    _panic("Cannot resize a matrix which has an outstanding submatrix");
  
  // Check data room
  if ( (nrows*ncols > csize) || (csize == 0) ) {
    dsize = nrows*ncols; csize = dsize;
    if ( data ) delete data;
    data = new FrReal[csize];
  }
  // Check column room
  if ( (nrows > trsize) || (trsize == 0) ) {
    trsize = nrows;
    if ( rowPtrs ) delete rowPtrs;
    rowPtrs = new FrReal*[trsize];
  }
  FrReal **t;
  t = rowPtrs;
  for (int i=0, j=0; j<trsize; i+=ncols, ++j) {
    rowPtrs[j] = data + i;
  }
  return 1;
}


Matrix&
Matrix::operator=(const Matrix &m) 
{
  CHECKSIZE(m,"Incompatible size in =");
  resize(m.rowNum, m.colNum); // now it's same with <<
  for (int i=0; i<rowNum; i++) {
    for (int j=0; j<colNum; j++) {
      rowPtrs[i][j] = m.rowPtrs[i][j];
    }
  }
  return *this;
}

Matrix&
Matrix::operator<<(const Matrix &m) 
{
  resize(m.rowNum, m.colNum);

  for (int i=0; i<rowNum; i++) {
    for (int j=0; j<colNum; j++) {
      rowPtrs[i][j] = m.rowPtrs[i][j];
    }
  }
  return *this;
}

Matrix& Matrix::operator<<( FrReal *x )
{
  for (int i=0; i<rowNum; i++) {
    for (int j=0; j<colNum; j++) {
      rowPtrs[i][j] = *x++;
    }
  }
  return *this;
}


Matrix&
Matrix::operator=(FrReal x) 
{
  for (int i=0; i<rowNum; i++) {
    for (int j=0; j<colNum; j++) {
      rowPtrs[i][j] = x;
    }
  }
  return *this;
}

Matrix
Matrix::operator*(const Matrix &mat) const
{
  Matrix p(rowNum,mat.colNum);
  p = 0;
  
  if (colNum != mat.rowNum)
    _panic("Matrix mismatch in matrix matrix multiply");
  
  for (int i=0;i<rowNum;i++)
    for (int j=0;j<mat.colNum;j++)
      for (int k=0;k<mat.rowNum;k++)
	p[i][j]+=rowPtrs[i][k] * mat.rowPtrs[k][j];
  return p;
}

ColVector
Matrix::operator*(ColVector &mat) const
{
  ColVector p(rowNum);
  if (colNum != mat.rowNum)
    _panic("Matrix mismatch in matrix/vector multiply\n");

  p = 0.0;
  for (int j=0;j<colNum;j++) {
    for (int i=0;i<rowNum;i++) {
      p[i]+=rowPtrs[i][j] * mat[j];
    }
  }

  return p;
}

RowVector
Matrix::operator*(RowVector &mat) const
{
  RowVector p(colNum);
  FrReal *ptr = mat.rowPtrs[0],*ptr1;
  float pp;

  if (rowNum != mat.colNum)
    _panic("Matrix mismatch in matrix/vector multiply\n");

  p = 0.0;
  for (int i=0;i<rowNum;i++) {
    ptr1 = rowPtrs[i];
    pp = ptr[i];
    for (int j=0;j<colNum;j++) {
      p[j] += ptr1[j] * pp;
    }
  }
  return p;
}

Matrix
Matrix::operator/(FrReal x) const
{
  Matrix v(rowNum,colNum);
  int i;int j;
  for (i=0;i<rowNum;i++)
    for(j=0;j<colNum;j++)
      v.rowPtrs[i][j] = rowPtrs[i][j]/x;
  return v;
}

Matrix
Matrix::operator+(const Matrix &mat) const
{
  CHECKSIZE(mat,"Incompatible size in +");
  Matrix v(rowNum,colNum);
  int i;int j;
  for (i=0;i<rowNum;i++)
    for(j=0;j<colNum;j++)
      v.rowPtrs[i][j] = mat.rowPtrs[i][j]+rowPtrs[i][j];
  return v;
}


ColVector
ColVector::operator+(const ColVector &mat) const
{
  CHECKSIZE(mat,"Incompatible size in +");
  ColVector v(rowNum);
  for (int i=0;i<rowNum;i++)
      v[i] = (*this)[i] + mat[i];
  return v;
}

Matrix
Matrix::operator-(const Matrix &mat) const 
{
  CHECKSIZE(mat,"Incompatible size in -");
  Matrix v(rowNum,colNum);
  int i;int j;
  for (i=0;i<rowNum;i++)
    for(j=0;j<colNum;j++)
      v.rowPtrs[i][j] = rowPtrs[i][j]-mat.rowPtrs[i][j];
  return v;
}

ColVector
ColVector::operator-(const ColVector &mat) const
{
  CHECKSIZE(mat,"Incompatible size in +");
  ColVector v(rowNum);
  for (int i=0;i<rowNum;i++)
      v[i] = (*this)[i] - mat[i];
  return v;
}

Matrix
Matrix::operator-() const //negate
{
  Matrix A=(*this);

  for (int i=0; i<rowNum; i++)
    for (int j=0; j<colNum; j++)
    	A[i][j] = - rowPtrs[i][j];
  return A;
}

ColVector
ColVector::operator-() 
{
  ColVector A=(*this);

  for (int i=0; i<rowNum; i++)
    	A[i]= - (*this)[i];
  return A;
}

Matrix&
Matrix::operator+=(FrReal x){
  int i;int j;
  for (i=0;i<rowNum;i++)
    for(j=0;j<colNum;j++)
      rowPtrs[i][j] += x;
  return *this;
}

Matrix&
Matrix::operator+=(const Matrix &mat)
{
  CHECKSIZE(mat,"Incompatible size in +=");
  int i;int j;
  for (i=0;i<rowNum;i++)
    for(j=0;j<colNum;j++)
      rowPtrs[i][j] += mat.rowPtrs[i][j];
  return *this;
}

Matrix&
Matrix::operator-=(FrReal x){
  int i;int j;
  for (i=0;i<rowNum;i++)
    for(j=0;j<colNum;j++)
      rowPtrs[i][j] -= x;
  return *this;
}

Matrix&
Matrix::operator-=(const Matrix &mat) {
  CHECKSIZE(mat,"Incompatible size in -=");
  int i;int j;
  for (i=0;i<rowNum;i++)
    for(j=0;j<colNum;j++)
      rowPtrs[i][j] -= mat.rowPtrs[i][j];
  return *this;
}

Matrix&
Matrix::operator*=(FrReal x)
{
  int i;int j;
  for (i=0;i<rowNum;i++)
    for(j=0;j<colNum;j++)  rowPtrs[i][j] *= x;
  return *this;
}

Matrix&
Matrix::operator/=(FrReal x)
{
  int i;int j;
  for (i=0;i<rowNum;i++)
    for(j=0;j<colNum;j++)  rowPtrs[i][j] /= x;
  return *this;
}

Matrix
Matrix::operator*(FrReal x) const
{
  Matrix v(rowNum,colNum);
  int i;int j;
  for (i=0;i<rowNum;i++)
    for(j=0;j<colNum;j++)  v.rowPtrs[i][j] = rowPtrs[i][j]*x;
  return v;
}

ColVector
ColVector::operator*(FrReal x) const
{
  ColVector v(rowNum);
  for (int i=0;i<rowNum;i++)
    v[i] = (*this)[i] * x;
  return v;
}

ColVector&
ColVector::operator=(const Matrix &m) 
{
  CHECKSIZE(m,"Incompatible size in =");
  
  resize(m.rowNum);
  for (int i=0; i<rowNum; i++) {
    for (int j=0; j<colNum; j++) {
      rowPtrs[i][j] = m.rowPtrs[i][j];
    }
  }
  return *this;
}

ColVector&
ColVector::operator=(const ColVector &v) 
{
  CHECKSIZE(v,"Incompatible size in =");

  resize(v.rowNum);
  for (int i=0; i<rowNum; i++) {
    for (int j=0; j<colNum; j++) {
      rowPtrs[i][j] = v.rowPtrs[i][j];
    }
  }
  return *this;
}

ColVector&
ColVector::operator<<(const ColVector &v) 
{
  resize(v.rowNum);
  for (int i=0; i<rowNum; i++) {
    for (int j=0; j<colNum; j++) {
      rowPtrs[i][j] = v.rowPtrs[i][j];
    }
  }
  return *this;
}

ColVector&
ColVector::operator<<( FrReal *x )
{
  for (int i=0; i<rowNum; i++) {
    for (int j=0; j<colNum; j++) {
      rowPtrs[i][j] = *x++;
    }
  }
  return *this;
}

ColVector&
ColVector::operator=(FrReal x) 
{
  for (int i=0; i<rowNum; i++) {
    for (int j=0; j<colNum; j++) {
      rowPtrs[i][j] = x;
    }
  }
  return *this;
}

RowVector&
RowVector::operator=(const RowVector &v) 
{
  CHECKSIZE(v,"Incompatible size in =");

  if (colNum==0) {resize(v.colNum);}
  for (int i=0; i<rowNum; i++) {
    for (int j=0; j<colNum; j++) {
      rowPtrs[i][j] = v.rowPtrs[i][j];
    }
  }
  return *this;
}

RowVector&
RowVector::operator=(const Matrix &m) 
{
  CHECKSIZE(m,"Incompatible size in =");

  if (colNum==0) {resize(m.colNum);}
  for (int i=0; i<rowNum; i++) {
    for (int j=0; j<colNum; j++) {
      rowPtrs[i][j] = m.rowPtrs[i][j];
    }
  }
  return *this;
}

RowVector&
RowVector::operator<<(const RowVector &v) 
{
  resize(v.colNum);
  for (int i=0; i<rowNum; i++) {
    for (int j=0; j<colNum; j++) {
      rowPtrs[i][j] = v.rowPtrs[i][j];
    }
  }
  return *this;
}

RowVector&
RowVector::operator=(FrReal x) 
{
  for (int i=0; i<rowNum; i++) {
    for (int j=0; j<colNum; j++) {
      rowPtrs[i][j] = x;
    }
  }
  return *this;
}

FrReal
Matrix::SumSquare() const
{
  FrReal sum=0.0;
  for (int i=0; i<rowNum; i++)
    for (int j=0; j<colNum; j++)
      sum +=rowPtrs[i][j]*rowPtrs[i][j];

  return sum;
}

FrReal
Matrix::Sum() const
{
  FrReal sum=0.0;
  for (int i=0; i<rowNum; i++)
    for (int j=0; j<colNum; j++)
      sum +=rowPtrs[i][j];

  return sum;
}


void
Matrix::init_empty(void)
{
  refPtr  = 0;
  rowNum  = 0;
  colNum  = 0;
  data    = 0;
  rowPtrs = 0;
  csize   = 0;
  dsize   = 0;
  trsize  = 0;
}

int
Matrix::init(Matrix &m,int startr, int startc, int nrows, int ncols)
{

#ifndef NO_BOUNDS_CHECK
  if ((startr < 0) || (startc < 0) || (nrows < 0) || (ncols < 0) ||
      (nrows > m.rowNum) || (ncols > m.colNum))
    _panic("Submatrix requested out of bounds");
#endif
  
    // Set up the Matrix parameters
  rowNum = nrows;
  colNum = ncols;
  rowPtrs = new FrReal*[nrows];
 
    // Set up the pointers
  int i;
  for (i=0;i<nrows;i++) {rowPtrs[i] = m.rowPtrs[i+startr]+startc;}
  
    // This is a shared structure, so reference it.
  refPtr = m.ref();
  data = m.data;
  return 1;
}

Matrix::~Matrix()
{
  delete rowPtrs; 
  if (!dataShared()) {
    if (refPtr != NULL) delete refPtr;
    delete data; 
  }
  else
    unref();
}

Matrix
Matrix::t() const 
{
  Matrix tmp(colNum,rowNum);
  int i,j;
  for (i=0;i<rowNum;i++)
    for (j=0;j<colNum;j++)
      tmp[j][i] = (*this)[i][j];
  return tmp;
}

ColVector
RowVector::t() const 
{
  ColVector tmp(colNum);
  for (int i=0;i<colNum;i++)
      tmp[i] = (*this)[i];
  return tmp;
}

RowVector
ColVector::t() const 
{
  RowVector tmp(rowNum);
  for (int i=0;i<rowNum;i++)
      tmp[i] = (*this)[i];
  return tmp;
}

// computes M^t M
Matrix
Matrix::ip() const
{
  Matrix tmp(colNum,colNum);
  FrReal sum = 0;

  int i,j,k;
  for (i=0;i<colNum;i++) {
    for (j=i;j<colNum;j++) {
      sum = 0;
      for (k=0;k<rowNum;k++)
	sum += (*this)[k][i]* (*this)[k][j];
      tmp[j][i] = tmp[i][j] = sum;
    }
  }
  return tmp;
}

FrReal
ColVector::ip() const 
{
  FrReal sum = 0;
  for (int i=0;i<rowNum;i++)
      sum += sqr((*this)[i]);

  return sum;
}

FrReal
ColVector::ip(const ColVector &x) const 
{
  assert(n_of_rows() == x.n_of_rows());

  FrReal sum = 0;

  for (int i=0;i<rowNum;i++)
      sum += ((*this)[i])*x[i];

  return sum;
}

// computes M M^T
Matrix Matrix::op() const
{
  Matrix tmp(rowNum,rowNum);
  FrReal sum = 0;
  int i,j,k;
  for (i=0;i<rowNum;i++) {
    for (j=0;j<rowNum;j++) {
      sum = 0;
      for (k=0;k<colNum;k++) {
	sum += rowPtrs[i][k] * rowPtrs[j][k];
      }
      tmp[j][i] = tmp[i][j] = sum;
    }
  }
  return tmp;
}

#define TINY 1.0e-20;

/*****************************************************
 *                                                   *
 * LUD related functions                             *
 *                                                   *
 *****************************************************/

void
Matrix::LUDcmp(int *perm, int& d)
{
  int n = rowNum;

  int i,imax,j,k;
  FrReal big,dum,sum,temp;
  ColVector vv(n);
  
  d=1;
  for (i=0;i<n;i++) {
    big=0.0;
    for (j=0;j<n;j++)
      if ((temp=fabs(rowPtrs[i][j])) > big) big=temp;
    if (big == 0.0) _panic("Singular matrix in  LUDcmp");
    vv[i]=1.0/big;
  }
  for (j=0;j<n;j++) {
    for (i=0;i<j;i++) {
      sum=rowPtrs[i][j];
      for (k=0;k<i;k++) sum -= rowPtrs[i][k]*rowPtrs[k][j];
      rowPtrs[i][j]=sum;
    }
    big=0.0;
    for (i=j;i<n;i++) {
      sum=rowPtrs[i][j];
      for (k=0;k<j;k++)
	sum -= rowPtrs[i][k]*rowPtrs[k][j];
      rowPtrs[i][j]=sum;
      if ( (dum=vv[i]*fabs(sum)) >= big) {
	big=dum;
	imax=i;
      }
    }
    if (j != imax) {
      for (k=0;k<n;k++) {
	dum=rowPtrs[imax][k];
	rowPtrs[imax][k]=rowPtrs[j][k];
	rowPtrs[j][k]=dum;
      }
      d *= -1;
      vv[imax]=vv[j];
    }
    perm[j]=imax;
    if (rowPtrs[j][j] == 0.0) rowPtrs[j][j]=TINY;
    if (j != n) {
      dum=1.0/(rowPtrs[j][j]);
      for (i=j+1;i<n;i++) rowPtrs[i][j] *= dum;
    }
  }
}

#undef TINY

void
Matrix::LUBksb(int *perm, ColVector& b)
{
  int n = rowNum;

  int i,ii=-1,ip,j;
  FrReal sum;
  
  for (i=0;i<n;i++) {
    ip=perm[i];
    sum=b[ip];
    b[ip]=b[i];
    if (ii != -1)
      for (j=ii;j<=i-1;j++) sum -= rowPtrs[i][j]*b[j];
    else if (sum) ii=i;
    b[i]=sum;
  }
  for (i=n-1;i>=0;i--) {
    sum=b[i];
    for (j=i+1;j<n;j++) sum -= rowPtrs[i][j]*b[j];
    b[i]=sum/rowPtrs[i][i];
  }
}

void
Matrix::solveByLUD(const ColVector &B, ColVector& X)
{
  if (colNum != rowNum)
    _panic("Solution for nonsquare matrix");

  Matrix A(rowNum, rowNum);
  A = *this;

  X = B;

  int *perm = new int[rowNum];
  int p;

  A.LUDcmp(perm, p);

  A.LUBksb(perm, X);

  delete perm;
}

ColVector Matrix::LUDsolve(const ColVector& B) {
  ColVector X(rowNum);
  solveByLUD(B, X);
  return X;
}

void
Matrix::LDLtDcmp()
{
  Matrix* A = this;
  int n = rowNum;
  Matrix L(n, n);
  int i, j;

  ColVector v(A->rowNum);
  for (j=0; j<n; j++) {
    // Compute v
    for (i=0; i<=j-1; i++) {
      v[i] = (*A)[j][i]*(*A)[i][i];
    }

    if (j==0) {
      v[j] = (*A)[j][j];
      // Store D[j] only 
      (*A)[j][j] = v[j];
      (*A)(j+1,n-1,j,j) = (*A)(j+1,n-1,j,j)/v[j];
    } else {
      v[j] = (*A)[j][j]-((*A)(j,j,0,j-1)*v(0,j-1,0,0))[0][0];
      // Store D[j] and compute L(j+1:n,j)
      (*A)[j][j] = v[j];
      (*A)(j+1,n-1,j,j) = ((*A)(j+1,n-1,j,j)-
			   (*A)(j+1,n-1,0,j-1)*v(0,j-1,0,0))/
			   v[j];
    }
  }
}
  
Matrix
Matrix::sqrt()
{
    // the matrix has to be symmetric and positive definite
  Matrix A(rowNum, colNum);

  A = *this;
  A.LDLtDcmp();
  for (int j=0; j<colNum; j++) {
    A[j][j] = ::sqrt(A[j][j]);
    for (int i=j+1; i<rowNum; i++) {
      A[i][j] *= A[j][j];
      A[j][i] = 0.0;
    }
  }
  return A;
}

Matrix
Matrix::Map(FrReal (*fn)(FrReal)) const
{
  int i, j;
  Matrix temp(rowNum,colNum);
  
  for (i=0;i<rowNum;i++)
    for (j=0;j<colNum;j++)
      temp[i][j] = fn((*this)[i][j]);

  return temp;
}

/*****************************************************
 *                                                   *
 * SVD related functions                             *
 *                                                   *
 *****************************************************/

static FrReal at,bt,ct;
#define PYTHAG(a,b) ((at=fabs(a)) > (bt=fabs(b)) ? \
(ct=bt/at,at*::sqrt(1.0+ct*ct)) : (bt ? (ct=at/bt,bt*::sqrt(1.0+ct*ct)): 0.0))

static FrReal maxarg1,maxarg2;

#ifdef MAX
#undef MAX
#endif

#define MAX(a,b) (maxarg1=(a),maxarg2=(b),(maxarg1)>(maxarg2)?(maxarg1):(maxarg2))

#ifdef SIGN
#undef SIGN
#endif

#define SIGN(a,b) ((b) >= 0.0 ? fabs(a) : -fabs(a))

// singular value decomposition
//
// M = U.t * Md * V
//        
// if I understand it correctly, W is the diagonal matrix stored in 
// a column vector, V becomes V.t, and A becomes U.t. 
// try not to use SVDcmp directly. See SVD(M, Md, U, V) -- JW. 

void Matrix::SVDcmp(ColVector& W, Matrix& V)
{
  int m = this->rowNum;
  int n = this->colNum;

  int flag,i,its,j,jj,k,l,nm;
  FrReal c,f,h,s,x,y,z;
  FrReal anorm=0.0,g=0.0,scale=0.0;

  // So that the original NRC code (using 1..n indexing) can be used
  // This should be considered as a temporary fix.
  FrReal **a = new FrReal*[m+1];
  FrReal **v = new FrReal*[n+1];
  FrReal **w = W.rowPtrs;

  w--;
  for (i=1;i<=m;i++) {
    a[i] = this->rowPtrs[i-1]-1; 
  }
  for (i=1;i<=n;i++) {
    v[i] = V.rowPtrs[i-1]-1;
  }
  
  if (m < n) _panic("SVDcmp: You must augment A with extra zero rows");
  FrReal* rv1=new FrReal[n+1]; 

  for (i=1;i<=n;i++) {
    l=i+1;
    rv1[i]=scale*g;
    g=s=scale=0.0;
    if (i <= m) {
      for (k=i;k<=m;k++) scale += fabs(a[k][i]);
      if (scale) {
	for (k=i;k<=m;k++) {
	  a[k][i] /= scale;
	  s += a[k][i]*a[k][i];
	}
	f=a[i][i];
	g = -SIGN(::sqrt(s),f);
	h=f*g-s;
	a[i][i]=f-g;
	if (i != n) {
	  for (j=l;j<=n;j++) {
	    for (s=0.0,k=i;k<=m;k++) s += a[k][i]*a[k][j];
	    f=s/h;
	    for (k=i;k<=m;k++) a[k][j] += f*a[k][i];
	  }
	}
	for (k=i;k<=m;k++) a[k][i] *= scale;
      }
    }
    w[i][0]=scale*g;
    g=s=scale=0.0;
    if (i <= m && i != n) {
      for (k=l;k<=n;k++) scale += fabs(a[i][k]);
      if (scale) {
	for (k=l;k<=n;k++) {
	  a[i][k] /= scale;
	  s += a[i][k]*a[i][k];
	}
	f=a[i][l];
	g = -SIGN(::sqrt(s),f);
	h=f*g-s;
	a[i][l]=f-g;
	for (k=l;k<=n;k++) rv1[k]=a[i][k]/h;
	if (i != m) {
	  for (j=l;j<=m;j++) {
	    for (s=0.0,k=l;k<=n;k++) s += a[j][k]*a[i][k];
	    for (k=l;k<=n;k++) a[j][k] += s*rv1[k];
	  }
	}
	for (k=l;k<=n;k++) a[i][k] *= scale;
      }
    }
    anorm=MAX(anorm,(fabs(w[i][0])+fabs(rv1[i])));
  }
  for (i=n;i>=1;i--) {
    if (i < n) {
      if (g) {
	for (j=l;j<=n;j++)
	  v[j][i]=(a[i][j]/a[i][l])/g;
	for (j=l;j<=n;j++) {
	  for (s=0.0,k=l;k<=n;k++) s += a[i][k]*v[k][j];
	  for (k=l;k<=n;k++) v[k][j] += s*v[k][i];
	}
      }
      for (j=l;j<=n;j++) v[i][j]=v[j][i]=0.0;
    }
    v[i][i]=1.0;
    g=rv1[i];
    l=i;
  }
  for (i=n;i>=1;i--) {
    l=i+1;
    g=w[i][0];
    if (i < n)
      for (j=l;j<=n;j++) a[i][j]=0.0;
    if (g) {
      g=1.0/g;
      if (i != n) {
	for (j=l;j<=n;j++) {
	  for (s=0.0,k=l;k<=m;k++) s += a[k][i]*a[k][j];
	  f=(s/a[i][i])*g;
	  for (k=i;k<=m;k++) a[k][j] += f*a[k][i];
	}
      }
      for (j=i;j<=m;j++) a[j][i] *= g;
    } else {
      for (j=i;j<=m;j++) a[j][i]=0.0;
    }
    ++a[i][i];
  }
  for (k=n;k>=1;k--) {
    for (its=1;its<=30;its++) {
      flag=1;
      for (l=k;l>=1;l--) {
	nm=l-1;
	if (fabs(rv1[l])+anorm == anorm) {
	  flag=0;
	  break;
	}
	if (fabs(w[nm][0])+anorm == anorm) break;
      }
      if (flag) {
	c=0.0;
	s=1.0;
	for (i=l;i<=k;i++) {
	  f=s*rv1[i];
	  if (fabs(f)+anorm != anorm) {
	    g=w[i][0];
	    h=PYTHAG(f,g);
	    w[i][0]=h;
	    h=1.0/h;
	    c=g*h;
	    s=(-f*h);
	    for (j=1;j<=m;j++) {
	      y=a[j][nm];
	      z=a[j][i];
	      a[j][nm]=y*c+z*s;
	      a[j][i]=z*c-y*s;
	    }
	  }
	}
      }
      z=w[k][0];
      if (l == k) {
	if (z < 0.0) {
	  w[k][0] = -z;
	  for (j=1;j<=n;j++) v[j][k]=(-v[j][k]);
	}
	break;
      }
      if (its == 30) _panic("SVDcmp:no convergence in 30 iterations");
      x=w[l][0];
      nm=k-1;
      y=w[nm][0];
      g=rv1[nm];
      h=rv1[k];
      f=((y-z)*(y+z)+(g-h)*(g+h))/(2.0*h*y);
      g=PYTHAG(f,1.0);
      f=((x-z)*(x+z)+h*((y/(f+SIGN(g,f)))-h))/x;
      c=s=1.0;
      for (j=l;j<=nm;j++) {
	i=j+1;
	g=rv1[i];
	y=w[i][0];
	h=s*g;
	g=c*g;
	z=PYTHAG(f,h);
	rv1[j]=z;
	c=f/z;
	s=h/z;
	f=x*c+g*s;
	g=g*c-x*s;
	h=y*s;
	y=y*c;
	for (jj=1;jj<=n;jj++) {
	  x=v[jj][j];
	  z=v[jj][i];
	  v[jj][j]=x*c+z*s;
	  v[jj][i]=z*c-x*s;
	}
	z=PYTHAG(f,h);
	w[j][0]=z;
	if (z) {
	  z=1.0/z;
	  c=f*z;
	  s=h*z;
	}
	f=(c*g)+(s*y);
	x=(c*y)-(s*g);
	for (jj=1;jj<=m;jj++) {
	  y=a[jj][j];
	  z=a[jj][i];
	  a[jj][j]=y*c+z*s;
	  a[jj][i]=z*c-y*s;
	}
      }
      rv1[l]=0.0;
      rv1[k]=f;
      w[k][0]=x;
    }
  }
  delete[] rv1;
  delete[] a;
  delete[] v;
}

#undef SIGN
#undef MAX
#undef PYTHAG

void Matrix::SVBksb(const ColVector& w, const Matrix& v,  
		    const ColVector& b, ColVector& x)
{
  int m = this->rowNum;
  int n = this->colNum;
  FrReal** u = rowPtrs;

  int jj,j,i;
  FrReal s,*tmp;
  
  tmp=new FrReal[n];
  for (j=0;j<n;j++) {
    s=0.0;
    if (w[j]) {
      for (i=0;i<m;i++) s += u[i][j]*b[i];
      s /= w[j];
    }
    tmp[j]=s;
  }
  for (j=0;j<n;j++) {
    s=0.0;
    for (jj=0;jj<n;jj++) s += v[j][jj]*tmp[jj];
    x[j]=s;
  }
  delete tmp;
}



#define TOL 1.0e-5

void 
Matrix::solveBySVD(const ColVector& b, ColVector& x)
{
  int j;
  FrReal wmax,thresh;

  int ma = this->colNum;

  ColVector w(ma);
  Matrix v(ma,ma);

  Matrix A(rowNum, colNum);
  A= *this;
  A.SVDcmp(w,v);

  wmax=0.0;
  for (j=0;j<ma;j++)
    if (w[j] > wmax) wmax=w[j];
  thresh=TOL*wmax;
  for (j=0;j<ma;j++)
    if (w[j] < thresh) w[j]=0.0;

  A.SVBksb(w,v,b,x);
}

ColVector Matrix::SVDsolve(const ColVector& B) {
  ColVector X(colNum);
  solveBySVD(B, X);
  return X;
}

#undef TOL
#define TOL 1e-5

Matrix
Matrix::i() const
{
  int i,j;

  if ( rowNum != colNum)
    _panic("Cannot invert a non-square matrix");

  Matrix B(rowNum, rowNum), X(rowNum, rowNum);
  Matrix V(rowNum, rowNum);
  ColVector W(rowNum);

  for (i=0; i<rowNum; i++) {
    for (j=0; j<rowNum; j++) {
      B[i][j] = (i == j) ? 1 : 0;
    }
  }

  Matrix A(rowNum, rowNum), C(rowNum, rowNum);
  C = A = *this;


  A.SVDcmp(W, V);

  // Zero out small W's
  FrReal maxW=0.0;
  for (j=0;j<rowNum;j++)
    if (W[j] > maxW) maxW=W[j];
  FrReal thresh=TOL*maxW;
  for (j=0;j<rowNum;j++)
    if (W[j] < thresh) W[j]=0.0;

#ifdef SUBSCRIPT_START_WITH_1
  for (j=1; j<=rowNum; j++) { // col() starts with 1
#else
  for (j=0; j<rowNum; j++) { // col() starts with 0
#endif

    // This generates a compiler warning, but it is
    // right === the .col generates an rvalue which
    // is a pointer into X which then gets modified by
    // the routine --- any ideas how to do it better
    // without copying?

    ColVector Xcol = X.col(j);
    A.SVBksb(W, V, B.col(j), Xcol);
  }

//  cerr << "A=\n"<< A << endl
//       << "W=\n"<< W << endl
//       << "V=\n"<< V << endl;

  return X;
}

#undef TOL


ostream &operator <<(ostream &s,const Matrix &m)
{
  int i,j;

  for (i=0;i<m.rowNum;i++) {
    for (j=0;j<m.colNum;j++)
      s << m[i][j] << "  ";
    s << endl;
  }

  return s;
}

/************************************************************************
 *
 * For computing jacobian matrix using finite difference approximation
 *
 ************************************************************************/

/* Try to take derivatives of a function. */
FrReal
Matrix::deriv_from_pts(FrReal y1, FrReal y2, FrReal y3,
		       FrReal dstep, FrReal x)
{
  FrReal a,b;
  /* printf("y1 = %f, y2 = %f, y3 = %f ",y1,y2,y3); */
  y1 -= y2;
  y3 -= y2;
  a = (y1 + y3)/(2*dstep*dstep);
  b = a * (dstep - 2*x) - y1/dstep;
  /* printf("a = %f, b = %f \n",a,b);*/
  return 2*a*x + b;
}

/* Approximate dfn(i)/dval(n) for each i using some method or another. */
void  
Matrix::deriv_n(ColVector (*fn)(const ColVector&, const ColVector&),
	const ColVector& val, int n, const ColVector& extra, ColVector& deriv)
{
  int i;
  ColVector temp[3];
  ColVector new_val;
  new_val << val;

  FrReal dstep = ::sqrt(FR_EPSFCN)*fabs(val[n]);
  new_val[n] -= dstep;

  for (i=0;i<3;i++) {
    temp[i] << fn(new_val,extra);
    new_val[n] += dstep;
  }
  int dim = temp[0].rowNum;

  for (i=0;i<dim;i++) {
    deriv[i] = deriv_from_pts(temp[0][i],temp[1][i],temp[2][i],dstep,val[n]); }
}


/* computes the jacobian of a function evaluated with parameters
   where and other inputs extra.  Form is:
   
   dy1/da1  dy2/da1 ....
   dy1/da2  dy2/da2 ....
   .         .
   .         .
   .         .
   
   where y = f(a).
   
   */
Matrix
Matrix::jacobian(ColVector (*fn)(const ColVector&, const ColVector&),
		 int odim, const ColVector& where, const ColVector& extra)
{
  int idim = where.rowNum;
  Matrix jac(odim,idim);

#ifdef SUBSCRIPT_START_WITH_1
  for (int i=1;i<=idim;i++)     // col() starts with 1
#else
  for (int i=0;i<idim;i++) {    // col() starts with 0
#endif  
    ColVector jac_col = jac.col(i);
    deriv_n(fn,where,i,extra,jac_col);
  }
  return jac;
}


// M = U.t * Md * V
// a more intuitive version (calls  SVDcmp)  -- JW.

void SVD(Matrix& M, Matrix& Md, Matrix& U, Matrix& V) {

  ColVector D(M.n_of_rows());
  U = M;
  U.SVDcmp(D, V); // see explanation on SVDcmp

  U = U.t();
  V = V.t();

  for (int i=0; i<M.n_of_rows(); i++)
    for (int j=0; j<M.n_of_cols(); j++)
        if (i==j)
          Md[i][i]=D[i];
        else //i!=j
          Md[i][j]=0;
}

#ifdef WITH_MAIN

ColVector
test_fcn(const ColVector& where, const ColVector& extra)
{
  FrReal x, y;
  ColVector res(3);

  x = where[0]; y = where[1];
  res[0] = x*x+y*y; res[1] = x-y;
  res[2] = x+y;
  return res;
}

  
void
main()
{

  Matrix A(3, 3), D(3,3);
  ColVector B(3); //double B[2];
  Matrix AA(6, 3);
  ColVector C(3), X(3);

  A[0][0] = 10; A[0][1] = 20; A[0][2] = 30;
  A[1][0] = 20; A[1][1] = 45; A[1][2] = 80;
  A[2][0] = 20; A[2][1] = 80; A[2][2] = 171;
  B[0] = 50; B[1] = 120; B[2] = 231;

  cout<<"A="<<A;
  cout<<"D="<<D;
  Matrix &tmp=(A.col(1)-D.col(1)).Rows(1,2);
  C.Rows(1,2)=tmp;
  cout<<"tmp="<<tmp;
  cout<<"C="<<C;

  cout << (A.ip())[0][0] << endl;
  cout << (C.ip())[0][0] << endl;
  cout << A.i() << endl;
  cout << A.i()*A << endl;

}
  
#endif /* WITH_MAIN */


#ifdef Image_hh

template <class T>
ColVector&
operator <<(ColVector &t,const XVImage<T> &x)
{
  int i;
  T *ptr = x.data();

  t.resize(x.width()*x.height());
  for (i=0;i<t.n_of_rows();i++)
    t[i] = (FrReal)(*ptr++);

  return t;
}


template <class T>
RowVector&
operator << (RowVector &t,const XVImage<T> &x)
{
  int i;
  T *ptr = x.data();

  t.resize(x.width()*x.height());
  for (i=0;i<t.n_of_cols();i++)
    t[i] = (FrReal)(*ptr++);

  return t;
}



ColVector&
operator <<(ColVector &t,const Image &x)
{
  int i;
  int *ptr = x.data();

  t.resize(x.width()*x.height());
  for (i=0;i<t.n_of_rows();i++)
    t[i] = (FrReal)(*ptr++);

  return t;
}

RowVector&
operator <<(RowVector &t,const Image &x)
{
  int i;
  int *ptr = x.data();

  t.resize(x.width()*x.height());
  for (i=0;i<t.n_of_cols();i++)
    t[i] = (FrReal)(*ptr++);

  return t;
}



template <class T>
XVImage<T>&
operator >>(const ColVector &t, XVImage<T> &x)
{
  int i,j,k;

  if (x.width()*x.height() != t.n_of_rows())
    _panic("Image/matrix injection size mismatch");

  k=0;
  for (i=0;i<x.height();i++)
    for (j=0;j<x.width();j++)
      x[i][j] = T(t[k++]);

  return x;
}

template <class T>
XVImage<T>&
operator >>(const RowVector &t, XVImage<T> &x)
{
  int i,j,k;

  if (x.width()*x.height() != t.n_of_cols())
    _panic("Image/matrix injection size mismatch");

  k=0;
  for (i=0;i<x.height();i++)
    for (j=0;j<x.width();j++)
      x[i][j] = T(t[k++]);

  return x;
}


Image&
operator >>(const ColVector &t, Image &x)
{
  int i,j,k;

  if (x.width()*x.height() != t.n_of_rows())
    _panic("Image/matrix injection size mismatch");

  k=0;
  for (i=0;i<x.height();i++)
    for (j=0;j<x.width();j++)
      x[i][j] = (int)t[k++];

  return x;
}

Image&
operator >>(const RowVector &t, Image &x)
{
  int i,j,k;

  if (x.width()*x.height() != t.n_of_cols())
    _panic("Image/matrix injection size mismatch");

  k=0;
  for (i=0;i<x.height();i++)
    for (j=0;j<x.width();j++)
      x[i][j] = (int)t[k++];

  return x;
}


#endif /* Image_hh */

Matrix
operator*(const DiagonalMatrix &x, const Matrix &y)
{
  if (x.n_of_cols() != y.n_of_rows())
    _panic("Matrix mismatch in matrix/vector multiply\n");

  Matrix temp(y);

  for (int i=0;i<y.n_of_rows();i++) {
    for (int j=0;j<y.n_of_cols();j++) {
      temp[i][j] *= x(i);
    }
  }
  return temp;
}


RowVector
operator*(const DiagonalMatrix &x, const RowVector &y)
{
  if (x.n_of_cols() != y.n_of_cols())
    _panic("Matrix mismatch in matrix/vector multiply\n");

  RowVector temp(y);

  for (int j=0;j<y.n_of_cols();j++) {
      temp[j] *= x(j);
    }
  return temp;
}


Matrix
operator*(const Matrix &x, const DiagonalMatrix &y)
{
  if (x.n_of_cols() != y.n_of_rows())
    _panic("Matrix mismatch in matrix/vector multiply\n");

  Matrix temp(x);

  for (int j=0;j<x.n_of_cols();j++) {
    for (int i=0;i<x.n_of_rows();i++) {
      temp[i][j] *= y(j);
    }
  }
  return temp;
}

  
DiagonalMatrix
operator*(const DiagonalMatrix &x, const DiagonalMatrix &y)
{
  if (x.n_of_rows() != y.n_of_rows())
    _panic("Matrix mismatch in diagonal matrix multiply\n");

  ColVector temp(x.t);

  for (int j=0;j<y.n_of_cols();j++) 
    temp[j] *= y(j);
  return DiagonalMatrix(temp);
}

  
ostream &operator << (ostream &s,const DiagonalMatrix &x)
{
  s << (x.t).t();
  return s;
}

//----------------------------------------------------------------
// Utilities
//----------------------------------------------------------------

FrReal invs(FrReal x) {return 1/x;}

// Add a column to x
Matrix add_column(Matrix &x, const ColVector &c)
{
  Matrix temp(x.rowNum,x.colNum+1);
  for (int i=0; i<x.rowNum; ++i) {
    for (int j=0; j<x.colNum; ++j) {
      temp.rowPtrs[i][j] = x.rowPtrs[i][j];
    }
  }
  temp.Column(temp.n_of_cols()-1) = c;
  return temp;
}

Matrix add_column(Matrix &x, const Image &I)
{
  ColVector temp;
  temp << I;
  return add_column(x,temp);
}

template <class T>
Matrix add_column(Matrix &x, const XVImage<T> &I)
{
  ColVector temp;
  temp << I;
  return add_column(x,temp);
}

//----------------------------------------------------------------
// Instantiations
//----------------------------------------------------------------

#ifdef MANUAL_INSTANTIATE
#ifdef XVImage_hh

#if (XV_OS == XV_IRIX)

#pragma instantiate  ColVector &operator <<(ColVector &x,const XVImage<int> &y)
#pragma instantiate  ColVector &operator <<(ColVector &x,const XVImage<float> &y)
#pragma instantiate  XVImage<int> &operator >>(const ColVector &x,XVImage<int> &y)
#pragma instantiate  XVImage<float> &operator >>(const ColVector &x,XVImage<float> &y)

#pragma instantiate  RowVector &operator <<(RowVector &x,const XVImage<int> &y)
#pragma instantiate  RowVector &operator <<(RowVector &x,const XVImage<float> &y)
#pragma instantiate  XVImage<int> &operator >>(const RowVector &x,XVImage<int> &y)
#pragma instantiate  XVImage<float> &operator >>(const RowVector &x,XVImage<float> &y)

#pragma instantiate  Matrix add_column(Matrix &x,const XVImage<float> &y)

#endif /* (XV_OS == XV_IRIX) */
#if __GNUC__

template  ColVector &operator <<(ColVector &x,const XVImage<int> &y);
template  ColVector &operator <<(ColVector &x,const XVImage<float> &y);
template  XVImage<int> &operator >>(const ColVector &x,XVImage<int> &y);
template  XVImage<float> &operator >>(const ColVector &x,XVImage<float> &y);

template  RowVector &operator <<(RowVector &x,const XVImage<int> &y);
template  RowVector &operator <<(RowVector &x,const XVImage<float> &y);
template  XVImage<int> &operator >>(const RowVector &x,XVImage<int> &y);
template  XVImage<float> &operator >>(const RowVector &x,XVImage<float> &y);

template  Matrix add_column(Matrix &x,const XVImage<int> &y);
template  Matrix add_column(Matrix &x,const XVImage<float> &y);

#endif /* __GNUC__ */
#if (XV_OS == XV_SOLARIS)

//special  ColVector &operator <<(ColVector &x,const XVImage<int> &y);
//special  XVImage<int> &operator >>(const ColVector &x,XVImage<int> &y);

//special  ColVector &operator <<(ColVector &x,const XVImage<float> &y);
//special  XVImage<float> &operator >>(const ColVector &x,XVImage<float> &y);

#endif /* (XV_OS == XV_SOLARIS) */

#endif /* XVImage_hh */
#endif /* MANUAL_INSTANTIATE */

//----------------------------------------------------------------
// End
//----------------------------------------------------------------
