/*
    Copyright (C) 1996 Gregory D. Hager
    (Yale Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//--------------------------------------------------------------------------
//
// Image.cc
//
//-------------------------------------------------------------------------

#include "Image.hh"
#include <Tools.hh>
#include <RGB2int.hh>
#include <math.h>

#include "Acquire.cc"
//#include "Drawing.cc"


//-----------------------------------------------------------------------------
//  Some useful functions.
//-----------------------------------------------------------------------------


// Draw one box on the image;

void Box(Image &im,int center_x,int center_y,int size_x,int size_y)
{ int j,temp;
  int shift;
  
  shift=center_y-size_y/2;
  for(j=center_y-size_y/2;j<=center_y+size_y/2;j++)
    { temp=(j-shift)/4;
      if(temp/2*2==temp)
	{ im[j][center_x+size_x/2]=255;
	  im[j][center_x-size_x/2]=255;
	}
      else
	{ im[j][center_x+size_x/2]=0;
	  im[j][center_x-size_x/2]=0;
	}
    }
  
  shift=center_x-size_x/2;
  for(j=center_x-size_x/2;j<=center_x+size_x/2;j++)
    { temp=(j-shift)/4;
      if(temp/2*2==temp)
	{ im[center_y+size_y/2][j]=255;
	  im[center_y-size_y/2][j]=255;
	}
      else
	{ im[center_y+size_y/2][j]=0;
	  im[center_y-size_y/2][j]=0;
	}
    }
}


/* Derivative in x direction */

Image &Dx(const Image& imag,Image& targ)
{

  if ((imag.width() < (targ.width() -1)) ||
      (imag.height() < (targ.height())))
    panic ("Target too small in Dx \n");
 
 int *counter, *icounter, *endrow, i;

  icounter = imag.data();
  counter = targ.data();
  for (i = 0; i < imag.height(); i++)
  {
    endrow = icounter + imag.width() - 1;
    for (; icounter < endrow; icounter++)
      *counter++ = *(icounter + 1) - *icounter;
    icounter++;
  }

  return targ;
}


/* Derivative in y direction */

Image &Dy(const Image& imag,Image &targ)
{
  int *counter, *icounter, *endimage;

  if ((imag.width() < (targ.width())) ||
      (imag.height() < (targ.height()-1)))
    panic ("Target too small in compressy \n");

  
  icounter = imag.data();
  counter  = targ.data();
  endimage = counter + targ.width() * targ.height();
  
  for (; counter < endimage; icounter++)
    *counter++ = *(icounter + targ.width()) - *icounter;
 
  return targ;
}


/* Compress for one position in x direction */

Image &compressx(const Image &imag,Image &targ)
{ 
  int *counter, *icounter, *endrow, i;

  if ((imag.width() < (targ.width() -1)) ||
      (imag.height() < (targ.height())))
    panic ("Target too small in compressx \n");

  
  icounter = imag.data();
  counter = targ.data();
  for (i = 0; i < targ.height(); i++)
    { 
      endrow = icounter + targ.width();
      for (; icounter < endrow; icounter++)
	*counter++ = (*(icounter + 1) + *icounter) / 2;
      icounter++;
    }
  return targ;
}

/* Same thing in y direction */

Image &compressy(const Image &imag, Image &targ)
{ 
  int *counter, *icounter, *endimage;
  
  if ((imag.width() < (targ.width())) ||
      (imag.height() < (targ.height()-1)))
    panic ("Target too small in compressy \n");

  icounter = imag.data();   
  counter = targ.data();
  endimage = counter + targ.height() * targ.width();
  
  for (; counter < endimage; icounter++)
    *counter++ = (*(icounter + targ.width()) + *icounter) / 2;
  
  return targ;
}

/* Derivative in x direction */

Image Dx(const Image& imag)
{
  Image temp = Image(imag.width() - 1, imag.height());

  return Dx(imag,temp);
}


/* Derivative in y direction */

Image Dy(const Image& imag)
{
  Image temp = Image(imag.width(), imag.height() - 1);
  return Dy(imag,temp);
}


/* Compress for one position in x direction */

Image compressx(const Image &imag)
{ 
  Image temp(imag.width() - 1, imag.height());
  return compressx(imag,temp);
}

/* Same thing in y direction */

Image compressy(const Image &imag)
{ 
  Image temp(imag.width(), imag.height() - 1);
  return compressy(imag,temp);
}

            
/* These 2 just ensure that the Image is again square afterwards... */

Image smoothDx(const Image& image)
{
  return compressy(Dx(image));
}


Image smoothDy(const Image& image)
{
  return compressx(Dy(image));
}

Image &smoothDx(const Image& image,Image &targ)
{
  return compressy(Dx(image),targ);
}


Image &smoothDy(const Image& image,Image &targ)
{
  return compressx(Dy(image),targ);
}

Image make_showable(const XVImage<float> &W)
{
  Image temp(W.width(),W.height());

  int i, j;

  for (i=0;i<W.height();i++)
    for (j=0;j<W.width();j++)
      temp[i][j] = (int)W[i][j];

  return temp;
}

// Acquire a (w,h)-sized rectangle of pixels from (x,y) with orientation
// atan2 dy dx.  quadrant should be in range [0..7] and match (dx,dy).
Image Image::acquire(int x, int y, 
		     short int dx, short int dy, short int quadrant,
		     int w, int h,
		     int sampw, int samph) const
{
  Image r(w,h);
  int*  s_image = image;
  int*  r_image = r.data();
  short int mask = 0xff;

  // ToDo: detect and optimise aligned grabs!

  if (sampw == 1) {
    if (samph == 1) {
      bham_rectangle(s_image,r_image,x,y,dx,dy,quadrant,h,w,mask,ncols);
    } else {
      bham_rectangle_msh(s_image,r_image,x,y,dx,dy,quadrant,h,w,samph,mask,ncols);
    }
  } else {
    if (samph == 1) {
      bham_rectangle_msw(s_image,r_image,x,y,dx,dy,quadrant,h,w,sampw,mask,ncols);
    } else {
      bham_rectangle_ms(s_image,r_image,x,y,dx,dy,quadrant,h,w,samph,sampw,mask,ncols);
    }
  }

  return r;
}

#include <Pipe_op.hh>


#ifndef DBL_MAX
#include <float.h>
#endif


// need this in addition to Image::subimage for const Image objects
Image subimage(const Image &im, int lx, int ly, int ux, int uy) {
  int *im_data;
  int im_width;
  int i,j;

  if ((lx < 0) || (ly < 0) ||
      (ux >= im.width()) || (uy >= im.height()))
    panic("subimage: attempt to acquire subimage outside the range of parent image.");

  Image processed(ux-lx+1,uy-ly+1);

  im_width = im.width();
  im_data = im.data();

  uy++;

  for (i=ly;i<uy;i++) {
    memcpy(processed[i-ly],im_data+(i*im_width)+lx,(sizeof(int)*(ux-lx+1)));
  }

  return processed;
}

// pixel square root
Image sqroot(const Image& im)
{
  int im_size;
  int *processed_data;
  int i, j;
  
  Image processed = im;

  im_size = processed.width() * processed.height();

  processed_data = processed.data();

  for (i=0; i<im_size; i++) {
    *processed_data = (int) rint(sqrt((double) *processed_data));
    processed_data++;
  }
  return processed;
}

// change RGB color image to grayscale

Image grayscale(const Image& im) {
  double mono_val;
  int *im_data, *gray_image_data;
  int i, im_size;

  Image gray_image(im.width(),im.height());

  im_data = im.data();
  gray_image_data = gray_image.data();

  im_size = im.width()*im.height();

  // average RGB values to get grayscale value
  for (i=0; i<im_size; i++) {
    mono_val = compute_red(*im_data);
    mono_val += compute_green(*im_data);
    mono_val += compute_blue(*im_data);
    mono_val /= 3;

    *gray_image_data = (int) rint(mono_val);
    im_data++;
    gray_image_data++;
  }
  return gray_image;
}

// Perform an image projection --- last one determines
// whether to normalize the result --- since we
// are using image, right now this normalizes to
// a fixed value of magnitude <= 255

Image project(const Image& im,float rc, float gc, float bc, int normalize) {
  double mono_val;
  int *im_data, *gray_image_data;
  int i, im_size;
  double r,b,g;
  double norm = sqrt(rc*rc+bc*bc+gc*gc);
  rc/=norm;
  bc/=norm;
  gc/=norm;

  Image gray_image(im.width(),im.height());

  im_data = im.data();
  gray_image_data = gray_image.data();

  im_size = im.width()*im.height();

  // average RGB values to get grayscale value
  for (i=0; i<im_size; i++,im_data++) {
    mono_val = (r = compute_red(*im_data)) *rc;
    mono_val += (g = compute_green(*im_data)) *gc;
    mono_val += (b = compute_blue(*im_data)) *bc;

    //    cout << r << " " << rc << " " << b << " " << bc << " " << g << " " << gc << endl;
    // Hope we never see an image with zeros in it..
    if (normalize) {
      mono_val *= 255.0/sqrt(r*r+g*g+b*b);
    }
    //    cout << mono_val << " ";
    *gray_image_data++ = (int) mono_val;
  }
  return gray_image;
}

// Binary Threshold
Image threshold(const Image& im, int thresh, int val)
{
  int im_size;
  int *im_data, *processed_data;
  int i, j;
  
  Image processed(im.width(),im.height());

  im_size = im.width() * im.height();

  im_data = im.data();
  processed_data = processed.data();

  for (i=0; i<im_size; i++) {
    // below threshold, black background
    if (*im_data < thresh)
      *processed_data = 0;
    else
      *processed_data = val;

    im_data++;
    processed_data++;
  }
  return processed;
}

// convolve im with mask
Image convolve(const Image &im, const Image& mask)
{
  Image newim(im);
  return im.convolve(newim,mask);
}
Image erode(const Image &im, const Image& mask)
{
  Image newim(im); 
  return im.erode(newim,mask);
}

// This is a slightly different definition of dx ...

// x derivative
Image dx(const Image& im)
{
  static Image mask(3,1);
  static int make_mask = 1;

  // make masks the first time around
  if (make_mask) { 
    int *mask_data = mask.data();

    *mask_data++ = 1;
    *mask_data++ = 0;
    *mask_data++ = -1;

    make_mask = 0;  // only make the mask once
  }

  // convolve im with mask
  Image processed = convolve(im,mask);

  return processed;
}

// y derivative
Image dy(const Image& im)
{
  static Image mask(1,3);
  static int make_mask = 1;

  // make masks the first time around
  if (make_mask) { 
    int *mask_data = mask.data();

    *mask_data++ = 1;
    *mask_data++ = 0;
    *mask_data++ = -1;

    make_mask = 0;  // only make the mask once
  }

  // convolve im with mask
  Image processed = convolve(im,mask);

  return processed;
}

// Sobel edge detector
Image sobel(const Image& im)
{
  static Image mask_x(3,3);
  static Image mask_y(3,3);
  static int make_mask = 1;
  int im_size;
  int *processed_data, *processed_y_data;
  int i;

  // make masks the first time around
  if (make_mask) { 
    int *mask_x_data = mask_x.data();
    int *mask_y_data = mask_y.data();

    // make x mask
    *mask_x_data++ = -1;
    *mask_x_data++ = -2;
    *mask_x_data++ = -1;
    *mask_x_data++ = 0;
    *mask_x_data++ = 0;
    *mask_x_data++ = 0;
    *mask_x_data++ = 1;
    *mask_x_data++ = 2;
    *mask_x_data++ = 1;

    // make y mask
    *mask_y_data++ = -1;
    *mask_y_data++ = 0;
    *mask_y_data++ = 1;
    *mask_y_data++ = -2;
    *mask_y_data++ = 0;
    *mask_y_data++ = 2;
    *mask_y_data++ = -1;
    *mask_y_data++ = 0;
    *mask_y_data++ = 1;

    make_mask = 0;  // only make the mask once
  }

  // look for edges in x direction first by convolving
  Image processed = convolve(im,mask_x);

  // check y direction by convolving
  Image processed_y = convolve(im,mask_y);

  // take absolute values for magnitude of edge values
  processed.abs();
  processed_y.abs();

  processed_y_data = processed_y.data();
  processed_data = processed.data();

  im_size = im.width() * im.height();

  // take max of x and y edge values
  for (i=0; i<im_size; i++) {
    if (*processed_y_data > *processed_data)
      *processed_data = *processed_y_data;
    
    processed_data++;
    processed_y_data++;
  }
  return processed;
}
