/*
    Copyright (C) 1996 Gregory D. Hager
    (Yale Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//--------------------------------------------------------------------------
//
// SSD.hh
//
//-------------------------------------------------------------------------

#ifndef SSD_H
#define SSD_H 1

#include "Tracker.hh"
#include "Video.hh"
//#include "XVImage.hh"
#include "Image.hh"
#include "CompFeature.hh"
#include "FTypes.hh"
#include "Matrix.hh"

extern Image make_showable(const XVImage<float>&);

//----------------------------------------------------------------------------
//  SSD base class declaration
//----------------------------------------------------------------------------

// SSD is now also a PointFeature
// Chien-Ping Lu
// Tue Feb 28 22:50:26 EST 1995


/* This class implements a modified version of the Lucas and Kanade
    style SSD matching algorithm for Image regions.  The computations
    include brightness and contrast compensation, translation,
    rotation, and two scaling components.  Various portions of these
    updates can be turned on and off via a variety of software
    switches as documented below.

    By default, SSD performs brightness and contrast plus translation.

*/


/*

   The state vector parameter definitions are:

   0: residual or correlation
   1: x
   2: y
   3: rotation
   4: scale
   5: aspect ratio
   6: shear
   7...15: other motions
   15...25: illumination

*/


#define N_SSD_PARMS 15
#define FIRST_ILLUM (B_QUAD+1)


// These definitions are for the processing bitmap and for parameter locations

#define B_RES 0
#define B_TRANS  1   // both x and y are a single bit
#define B_TRANSX  1
#define B_TRANSY  2
#define B_ROT    3
#define B_SCALE 4
#define B_ASPECT_RATIO 5
#define B_SHEAR  6
#define B_QUAD   7

#define B_BRIGHT FIRST_ILLUM
#define B_CONTRAST (FIRST_ILLUM + 1)

// This is just a "scaling" factor used to increase the precision of
// the integer-based images that we use.  This is a power of 2.

#define FACTORPOW 3
#define FACTOR (2<<FACTORPOW)


class SSD : public BasicFeature, public PointFeature
{
private:

  // Window parameters -- width, height and reduction factor

  int WINX,WINY;
  int Reduction;

  // Whether the coordinates are even or odd.  Used for compute
  // exactly where to acquire the image.  This is not fully
  // implemented, yet

  int odd_sized_x;
  int odd_sized_y;

  // Saved inverse LHS of equation computed offline and storage for
  // RHS which is computed online

  Matrix Amat;
  Matrix Amat_inv;

  ColVector b;
  ColVector resid;
  ColVector ref_vec;
  ColVector X;
  ColVector illum;

  // Weighting matrix as an image and as a vector

  XVImage<float> WI;
  DiagonalMatrix WM;

  // The scaling matrix for the residuals

  DiagonalMatrix HAT;

  // The residuals themselves

  XVImage<float> residuals;

  // The selection matrix when outlier detection is enabled.

  DiagonalMatrix Selection;

  // The SD of noise and the cutoff for  increasing or decreasing in SD units

  float pixel_sd;
  float outlier_threshold;

  char *name;

  // Various flags indicating how the processing should proceed

  int _incremental_compensation;  // Compensate after other stages or before
  int _compute_residue;           // Compute an adjusted residue
  int _compute_normalized_correlation;       // Compute an normalized correlation
  int _with_check;                // Compute outlier detection
  int _with_integration;          // Integrate the Image errors

  // Integration multiplier which determines how fast errors die out

  float integration_multiplier;

  // Some stuff for monitoring matches

  int _misses, _max_misses;
  double _correlation_threshold;

  // Bits indicating which portions of the processing should be performed

  unsigned _later_stages;          

  // Some display technology
  
  CWindow *win_live, *win_residual, *win_outliers, *win_basis, *win_hat;

 protected:

  // Some local variables the we don't want to keep allocating in the
  // inner loop.  They are the full resolution new Image and the reduced
  // resolution image

  Image new_image;
  Image new_refer;
  XVImage <float> small_XVrf;
  Image new_small_refer;
  Image *basis;

  // Some initialization routines.

  virtual void main_init(char *name = NULL);
  virtual void main_init(Image &x, char *name = NULL);

  // Relevant Image information to keep around about the original
  // template that we are tracking.

  Image reference;
  Image refer;
  Image small_refer;
  Image Dx_refer;
  Image Dy_refer;
  Image Dt;

  // Additional information needed to compute scaling

  Image grabimage;

  // These two are coefficients that can be adjusted to modify how
  // the window responds.  Normally, they are set to be the reduction
  // factor of the window.

  float Ax,Ay;
  int inner_iters;
  int outer_iters;
  int erode_iters;
  int dilate_iters;
  
  // Some state values that shouldn't be messed with by the user

  inline float set_residual(float x) { return state[B_RES] =x;};  
  inline float set_correlation(float x) { return state[B_RES] =x;};  

  // A small worker routine.

  int acquire_image(Image &new_image);

  // Used by multiresolution tracking

  void resize(int winx, int winy, int reduction);

 public:

  // Used by multiresolution tracking

  inline virtual void set_state(SSD &s) {
    memcpy(state,s.state,sizeof(float)*ncoords);
  }

  SSD(Video *pv, int winx=40, int winy=40, int reduction=4);
  SSD(Video *pv, char *lbasis, int reduction=4);
  SSD(const SSD &s);

  // SSD initialization

  virtual int state_init (float *state_in);
  virtual int state_init(float in_x,float in_y, float r = 0.0);
  virtual int interactive_init (CWindow &w, float r = 0.0);
  virtual int image_init(Image &x,float x=0, float y=0);

  void  get_illumination(char *name,Matrix &);
  void  get_size(char *name,int reduction);

  // Basicfeature required stuff

  virtual BasicFeature *dup() { return (BasicFeature*) new SSD(*this);};

  virtual int update();
  virtual int show(int color_in=no_color);

  // Getting/setting information

  inline int   width() {return WINX;};
  inline int   height() {return WINY;};
  inline int   reduction() {return Reduction;};

  inline float set_threshold(float x) {return outlier_threshold = x;};
  inline float set_pixel_sd(float x) {return pixel_sd = x;};
  inline void  set_iiter(int x) {inner_iters = x;};
  inline void  set_oiter(int x) {outer_iters = x;};

  inline void  set_morph(int e,int d) {erode_iters = e;
				       dilate_iters = d;};

  inline virtual float x() const { return state[B_TRANSX]; }
  inline virtual float y() const { return state[B_TRANSY]; }
  inline virtual float set_x(float x_in) { return state[B_TRANSX] = x_in; }
  inline virtual float set_y(float y_in) { return state[B_TRANSY] = y_in; }

  float orientation() {return state[B_ROT];};
  float set_orientation(float x) {return state[B_ROT]=x;};
  
  // Scaling

  float scale()      { return state[B_SCALE];}
  float scale_corr() { return max2(abs(cos(orientation())),
				  abs(sin(orientation())));};
  float set_scale(float x) {return state[B_SCALE] = x;};
  float scalex()           {return scale()/aspect_ratio();};
  float scaley()           {return scale()*aspect_ratio();};

  float aspect_ratio() {return state[B_ASPECT_RATIO];};
  float set_aspect_ratio(float x) {return state[B_ASPECT_RATIO]  = x;};
  
  float shear() {return state[B_SHEAR];}
  float set_shear(float x) {return state[B_SHEAR] =x;}

  float brightness() {return state[B_BRIGHT];};
  float relative_contrast() {return state[B_CONTRAST];};

  // Images that one might want

  // Live image after warping

  Image &live_image() {new_small_refer >>= Reduction;return new_small_refer;};
  
  // Basis elements

  Image basis_element(int i) {
    XVImage<float> temp(height()/Reduction-1,width()/Reduction-1);
    if (i < Amat.n_of_cols()) {
      Amat.Column(i) >> temp;
      return make_showable(temp.scale());
    }
    else {
      cerr << "index out of range" << endl;
      return Image(0,0);
    }
      
  }

  XVImage<float> xvbasis_element(int i) {
    XVImage<float> temp(height()/Reduction-1,width()/Reduction-1);
    if (i < Amat.n_of_cols()) {
      Amat.Column(i) >> temp;
    }
    else {
      cerr << "index out of range" << endl;
      temp = 0;
    }

    return temp;
      
  }

  Image residual_image() {
    if (_compute_residue) {
      return make_showable(residuals);
    }
    else {
      cout << "no residual computed" << endl;
      return Image(0,0);
    }
  }

  Image outlier_image() {
    if (_with_check) {
      XVImage<float> temp(height()/Reduction-1,width()/Reduction-1);
      Selection.diagonal() >> temp;
      return make_showable(temp * 150);
    }
    else {
      cout << "no outliers computed" << endl;
      return Image(0,0);
    }
  }

  Image corrected_image() {
    XVImage<float> temp(height()/Reduction-1,width()/Reduction-1);
    ColVector Y(X);
    int i;

    // Zero out the motion fields

    int next_one = 2;

    // Then do rotation

    if (isset(_later_stages,B_ROT)) {
      next_one++;
    }

    if (isset(_later_stages,B_SCALE)) {
      next_one++;
    }
    
    // This skips over brightness and contrast.

    for (i=0;i<next_one+2;i++)
      Y[i] = 0;

    // This should be the "residual" after compensating for illumination

    illum = (b - Amat * Y)/(1<<Reduction);

    illum >> temp;
    return make_showable(temp.scale());
  }

  ColVector &soln() { return X;};

  // Some values that can't be set by the user

  inline float residual() { return state[B_RES];};  
  inline float correlation() { return state[B_RES];};  

  virtual inline void set_coefficients(float Ax_in=1.0,
				       float Ay_in=1.0,
				       float Mx = 0.5)
    { Ax=Ax_in*Reduction;
      Ay=Ay_in*Reduction;
      integration_multiplier = Mx;
    };

  // Compute an adjusted residue?

  void compute_residue(int stupid_bool_val =1 ) {
    _compute_residue = stupid_bool_val;
    if (_compute_residue)
      _compute_normalized_correlation = 0;

  }

  void compute_correlation(int stupid_bool_val =1 ) {
    _compute_normalized_correlation = stupid_bool_val;
    if (_compute_normalized_correlation) {
      _compute_residue = 0;
      _with_check = 0;
    }
  }


  void check_match(double correlation_val,int max_misses = 1) {
    compute_correlation(1);
    _correlation_threshold = correlation_val;
    _max_misses = max_misses;
  }


  // Check and make sure that the computation reduce SSD error.

  void check_residue(int stupid_bool_val = 1) {
    _with_check = stupid_bool_val;
    if (_compute_residue)
      _compute_normalized_correlation = 0;
  }
  
  // Integrate errors.  If enabled, must adjust residues

  void with_integration(int stupid_bool_val = 1) {
    _with_integration = stupid_bool_val;
    if (stupid_bool_val)
      compute_residue();
  }

  // Toggles for the various stages.  In most cases, these cannot be
  // changed after the tracker is initialized

  void compute_brightness(int stupid_bool_val = 1) {
    bittoggle(_later_stages,B_BRIGHT,stupid_bool_val);
  }

  void compute_translation(int stupid_bool_val = 1) {
    bittoggle(_later_stages,B_TRANS,stupid_bool_val);
  }

  void compute_rotation(int stupid_bool_val = 1) {
    bittoggle(_later_stages,B_ROT,stupid_bool_val);
  }

  void compute_scale(int stupid_bool_val = 1) {
    bittoggle(_later_stages,B_SCALE,stupid_bool_val);
//    if (stupid_bool_val) 
//      bittoggle(_later_stages,B_ASPECT_RATIO,0);
  }

  void compute_aspect_ratio(int stupid_bool_val = 1) {
    bittoggle(_later_stages,B_ASPECT_RATIO,stupid_bool_val);
//    if (stupid_bool_val) 
//      bittoggle(_later_stages,B_SCALE,0);
  }

  void compute_shear(int stupid_bool_val = 1) {
    bittoggle(_later_stages,B_SHEAR,stupid_bool_val);
  }

  void compute_quad(int stupid_bool_val = 1) {
    bittoggle(_later_stages,B_QUAD,stupid_bool_val);
  }

  void show_live(CWindow *x) {win_live = x;};
  void show_basis(CWindow *x) {win_basis = x;};
  void show_outliers(CWindow *x) {win_outliers = x;};
  void show_residual(CWindow *x) {win_residual = x;};
  void show_hat(CWindow *x) {win_hat = x;};

  // For debugging purposes and other instructional uses

  void show_dt (CWindow &w, int scalef = 1, int x=0, int y = 0) {
    w.show((Dt*((int)(1.0/(2.0*FACTOR)))+128).magnify(Reduction*scalef),x,y);
  }

  void show_new_refer (CWindow &w, int scalef = 1, int x=0, int y = 0) {
    w.show((new_small_refer.scale()).magnify(Reduction*scalef),x,y);
  }

  void show_reference(CWindow &w) {
    w.show((refer.scale().magnify(Reduction)));};


  void show_weights(CWindow &w) {
    w.show((make_showable(WI)).magnify(Reduction));};

  void show_residual(CWindow &w) {
    w.show(((residual_image()).abs()).magnify(Reduction));
  };

  void show_selection(CWindow &w) {
    Image temp(width()/Reduction-1,height()/Reduction-1);
    Selection.diagonal()*255.0 >> temp;
    w.show(temp.magnify(Reduction));
  };


  void show_A_column(CWindow &w,int i) {
    XVImage<float> temp(height()/Reduction-1,width()/Reduction-1);
    if (i < Amat.n_of_cols()) {
      Amat.Column(i) >> temp;
    w.show(make_showable(temp.scale()).magnify(Reduction));
    }
  };

  void show_A(CWindow &w) {
    XVImage<float> temp(height()/Reduction-1,width()/Reduction-1);
    int ww = width()-Reduction;
    w.resize(ww*Amat.n_of_cols(),height()-Reduction);
    int i;
    for (i=0;i < Amat.n_of_cols();i++) {
      Amat.Column(i) >> temp;
      w.show(make_showable(temp.scale()).magnify(Reduction),i*ww);
    }
  };


};



#endif
