/*
    Copyright (C) 1996 Gregory D. Hager
    (Yale Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright abnd permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//--------------------------------------------------------------------------
//
// SSD.cc
//
//-------------------------------------------------------------------------

#include <fstream.h>
#include "SSD.hh"
#include "XVImage.hh"
#include "XVImage.icc"

//#define Txx int
//#include "XVImage.icc"
//#undef Txx


// Hidden variable which enables printing a bunch of stuff; makes it
// a little simpler to debug without recompiling

#define IFACTOR 100000

float gw(float x,float y,float s1,float s2) {
  return exp(-sqr(x/s1) - sqr(y/s2));
};

XVImage<float>
compute_gauss_weighting(int width, int height, float sigmaw, float sigmah)
{
  int i,j;

  XVImage<float> temp(height,width);

  // Use a general Gaussian weighting

  for (i=0;i<height;i++)
    for (j=0;j<width;j++)
      temp[i][j] = gw(i-height/2.0,j-width/2.0,sigmaw,sigmah);
		
  return temp;
}

template <class Txx>
XVImage<Txx>
dilate(XVImage<Txx> &old)
{
  XVImage<float> temp(old.height(),old.width());

  for (i=1;i<(old.height()-1);i++)
    for (j=1;j<(old.width()-1);j++)
      temp[i][j] = min2(min2(old[i+1][j],old[i-1][j]),
		       min2(old[i][j+1],old[i][j-1]));
		
  return temp;
}

void
dilate(ColVector &oldv,int ncols)
{
  XVImage<float> old(oldv.n_of_rows()/ncols,ncols);
  int i,j;

  oldv >> old;
  XVImage<float> temp(old);
  for (i=1;i<(old.height()-1);i++)
    for (j=1;j<(old.width()-1);j++)
      temp[i][j] = min2(old[i][j],min2(min2(old[i+1][j],old[i-1][j]),
				     min2(old[i][j+1],old[i][j-1])));
		
  oldv << temp;
}


void
erode(ColVector &oldv,int ncols)
{
  XVImage<float> old(oldv.n_of_rows()/ncols,ncols);
  int i,j;

  oldv >> old;
  XVImage<float> temp(old);
  for (i=1;i<(old.height()-1);i++)
    for (j=1;j<(old.width()-1);j++)
      temp[i][j] = max2(old[i][j],max2(max2(old[i+1][j],old[i-1][j]),
				     max2(old[i][j+1],old[i][j-1])));
		
  oldv << temp;
}

//DiagonalMatrix
XVImage<float>
compute_gradient_weighting(const XVImage<float> &Dx, const XVImage<float> &Dy)
{

  int i,j;
  XVImage<float> temp(Dx.width(),Dx.height());

  // Use a general Gaussian weighting

  for (i=0;i<Dx.width();i++)
    for (j=0;j<Dx.height();j++)
      temp[i][j] = 1/(sqr(Dx[i][j]) + sqr(Dy[i][j]) + 1.0);
		
  return temp;
}

ColVector
HAT_mat(Matrix &A,Matrix &Ai,float sd)
{
  ColVector temp(A.n_of_rows());

  int i;
  
  for (i=0;i<A.n_of_rows();i++) {

    // this system #define should happen in site_config.h
    // if you have problems check version of g++ and
    // either #define or #undefine EGCS  
    temp[i] = 1.0/(1.0-(A.Row(i) * Ai.Column(i))[0][0]);


    temp[i] /= sd;
  }

  return temp;
}

FrReal one_over_H(FrReal x) {return sqrt(1.0/(1.0 - x) );}

//-----------------------------------------------------------------------------
//  Constructors.
//-----------------------------------------------------------------------------

void
SSD::resize(int winx, int winy, int reduction) {

  // Get Image sizes on an even boundary

  winx = round(((float)winx)/reduction) * reduction;
  winy = round(((float)winy)/reduction) * reduction;

  // Store some useful values

  WINX=winx;
  WINY=winy;
  Reduction=reduction;

  set_coefficients(1.5,1.5,0.5);

  refer.resize(winy/reduction,winx/reduction);
  small_refer.resize(winy/reduction-1,winx/reduction-1);
  small_XVrf.resize(winx/reduction-1,winy/reduction-1);
  Dx_refer.resize(winy/reduction-1,winx/reduction-1);
  Dy_refer.resize(winy/reduction-1,winx/reduction-1);
  new_image.resize(winy,winx);
  new_refer.resize(winy/reduction,winx/reduction);
  Dt.resize(winy/reduction-1,winx/reduction-1);
  residuals.resize(winy/reduction-1,winx/reduction-1);

  Amat.resize(small_refer.width()*small_refer.height(),0);

  odd_sized_x = (winx/reduction) & 0x1;
  odd_sized_y = (winy/reduction) & 0x1;

}

SSD::SSD(Video *pv, int winx, int winy, int reduction):
     BasicFeature(N_SSD_PARMS,pv,"SSD_Tracker"),
     PointFeature(this)
{ 

  _incremental_compensation = 0;
  _compute_residue          = 1;
  _with_check               = 0;
  _with_integration         = 0;

  _later_stages = 0;

  _max_misses = 0;
  _misses = 0;
  

  compute_translation();
  compute_brightness();

  integration_multiplier = 0.49;    // Low enough to make differences
				    // eventually go to zero

  pixel_sd    = 5.0;
  outlier_threshold = 5.0;

  win_live = NULL;
  win_residual = NULL;
  win_outliers = NULL;
  win_basis = NULL;
  win_hat = NULL;

  name = NULL;
  resize(winx, winy, reduction);

  inner_iters = 1;
  outer_iters = 1;
  erode_iters = 1;
  dilate_iters = 1;
}


SSD::SSD(Video *pv, char *name_in, int reduction):
     BasicFeature(N_SSD_PARMS,pv,"SSD_Tracker"),
     PointFeature(this)
{ 

  _incremental_compensation = 0;
  _compute_residue          = 1;
  _with_check               = 0;
  _with_integration         = 0;

  _later_stages = 0;

  compute_translation();
  compute_brightness();

  _max_misses = 0;
  _misses = 0;

  pixel_sd = 5.0;
  outlier_threshold = 5.0;

  integration_multiplier = 0.49;    // Low enough to make differences
				    // eventually go to zero

  win_live = NULL;
  win_residual = NULL;
  win_outliers = NULL;
  win_basis = NULL;
  win_hat = NULL;

  name = name_in;
  get_size(name,reduction);

  inner_iters = 1;
  erode_iters = 1;
  dilate_iters = 1;

}

ColVector
unitize(const ColVector &x) {
  ColVector temp(x);
  temp -= x.Sum()/x.n_of_rows();
  temp /= sqrt(temp.ip());
  return temp;
}

template <class Txx>
XVImage<Txx>
unitize(const XVImage<Txx> &x) {
  XVImage<Txx> temp(x);
  temp -= x.sum()/(x.width()*x.height());
  temp /= sqrt(temp.ip(temp));
  return temp;
}

float turnoffs = 0.0;
void
update_selection(DiagonalMatrix &s,ColVector &r, float threshold, float pv)
{
  int i;
  float th = threshold*pv;

  turnoffs = 0.0;

  ColVector temp;
  temp = r.Map(abs);
  //  cout << temp.t();

  for (i=0;i<temp.n_of_rows();i++) {
    if (temp[i] > th) {
      s(i) = th/temp[i];
      turnoffs += 1;
    }
    else
      s(i) = 1.0;
  }
 
  turnoffs /= r.n_of_rows();

}
    
      

SSD::SSD(const SSD &s): 
     BasicFeature (s),
     PointFeature(this),
     Amat(s.Amat),
     Amat_inv(s.Amat_inv), 
     b(s.b),
     ref_vec(s.ref_vec),
     X(s.X), 
     WI(s.WI),
     WM(s.WM), 
     HAT(s.HAT), 
     residuals(s.residuals),
     Selection(s.Selection),
     new_image(s.new_image),
     new_refer(s.new_refer),
     small_XVrf(s.small_XVrf),
     refer(s.refer),
     small_refer(s.small_refer),
     Dx_refer(s.Dx_refer),
     Dy_refer(s.Dy_refer),
     Dt(s.Dt),
     grabimage(s.grabimage)
{ 

  WINX=s.WINX;
  WINY=s.WINY;
  Reduction=s.Reduction;
  
  Ax=s.Ax;
  Ay=s.Ay;

  odd_sized_x = s.odd_sized_x;
  odd_sized_y = s.odd_sized_y;

  outlier_threshold = s.outlier_threshold;
  pixel_sd = s.pixel_sd;

  win_live = s.win_live;
  win_residual = s.win_residual;
  win_outliers = s.win_outliers;
  win_basis = s.win_basis;
  win_hat = s.win_hat;

  _max_misses = 0;
  _misses = 0;

  _incremental_compensation = s._incremental_compensation;
  _compute_residue          = s._compute_residue;
  _with_check               = s._with_check;
  _with_integration         = s._with_integration;
  _later_stages             = s._later_stages;
  integration_multiplier    = s.integration_multiplier;
  inner_iters = s.inner_iters;
  erode_iters = s.erode_iters;
  dilate_iters = s.dilate_iters;
  name = s.name;

}


void
SSD::get_size(char *name, int reduction)
{ 
  float fheight,fwidth, fnbasis;  // Just cause matlab generates floats
  int width, height, nbasis;
  ifstream bimages(name);
  
//  use_basis_only = 1;

  bimages >> fnbasis;
  bimages >> fheight;
  bimages >> fwidth;

  // The floats should all be whole numbers

  nbasis = (int)fnbasis;
  width = (int)fwidth;
  height = (int)fheight;

  // This should allocate everything of interest

  resize(width,height,reduction);
}
  
void
SSD::get_illumination(char *name,Matrix &A)
{ 
  int i;
  float fheight,fwidth, fnbasis;  // Just cause matlab generates floats
  int width, height, nbasis;
  ifstream bimages(name);
  
//  use_basis_only = 1;

  bimages >> fnbasis;
  bimages >> fheight;
  bimages >> fwidth;

  // The floats should all be whole numbers

  nbasis = (int)fnbasis;
  width = (int)fwidth;
  height = (int)fheight;

  // This should allocate everything of interest

//  resize(width,height,Reduction);

  // Now, read the basis and scale it up to a reasonable range of values
  // I assume the vectors themselves are unit vectors.  Hopefully,
  // this will go away if and when image templates work.

  float *buffer = new float[nbasis*width*height];

  for (i=0;i<nbasis*width*height;i++) {
    bimages >> buffer[i];
    buffer[i] *= IFACTOR;
  }

  bimages.close();

  // If the basis was not a multiple of the reduction factor, we have to 
  // figure out how many pixels to skip front and back of each row and column

  int skipcf =  (width-WINX)/2;
  int skipcb = width-WINX - skipcf;
  int skiprf =  (height-WINY)/2;

  // Get the images in the proper form

  Image *basiss;
  basis = new Image[nbasis];
  basiss = new Image[nbasis];
  for (i=0;i<nbasis;i++) {
    basis[i].resize(WINY,WINX);
    basiss[i].resize(WINY/Reduction,WINX/Reduction);
    }

  float *val = buffer + skiprf*width*nbasis;
  int column,row;

  for (row=0;row<WINY;row++) {
    val += skipcf*nbasis;
    for (column=0;column<WINX;column++)
      for (i=0;i<nbasis;i++) {
	(basis[i])[row][column] = round((*val++));
      }
    val += skipcb*nbasis;
  }

  delete buffer;

  // Reduce the resolution


  for (i=0;i<nbasis;i++) {
    basis[i].reduce_resolution(Reduction,basiss[i]);
  }

  for (i=0;i<nbasis;i++) {
    const Image &tempi = compressy(compressx(basiss[i]));
    A = add_column(A,tempi);
  }


  delete(basiss);

}

void SSD::main_init(Image &x, char *name)
{

  int i,j;

  reference = x;

  // Increase the precision so we don't loose anything in the integer world

  reference <<= Reduction;

  // Reduce the resolution

  reference.reduce_resolution(Reduction,refer);

  // Compute the smoothed Image needed for Dt.

  small_refer=compressx(compressy(refer));

  // Compute the derivatives

  small_XVrf = XVImage<float>(small_refer);
  XVImage<float> XVrf(refer);

//  XVrf = unitize(XVrf);

//  XVImage <float> XVDx;
//  XVImage <float> XVDy;

//  XVDx = smoothDx(XVrf);
//  XVDy = smoothDy(XVrf);

  Dx_refer = smoothDx(refer);
  Dy_refer = smoothDy(refer);

  XVImage <float> XVDx(Dx_refer);
  XVImage <float> XVDy(Dy_refer);

  // Now, set up the translation
  
  if (isset(_later_stages,B_TRANS)) {
    Amat = add_column(Amat,XVDx);
    Amat = add_column(Amat,XVDy);
  }

  // Now, set up rotation

  if (isset(_later_stages,B_ROT)) {

    XVImage<float> temp(XVDx.height(),XVDx.width());

    // Generate the info needed to compute scale

    float midx = (XVDx.width()-1)/2.0;
    float midy = (XVDx.height()-1)/2.0;

    for (i=0;i<XVDx.width();i++)
      for (j=0;j<XVDx.height();j++)
	temp[j][i] = XVDx[j][i]*(j-midy) - 
	             XVDy[j][i]*(i-midx); 

    Amat = add_column(Amat,temp);

  }

  // Now, set up scale

  if (isset(_later_stages,B_SCALE)) {

    XVImage<float> temp(XVDx.height(),XVDx.width());

    // Generate the info needed to compute scale

    float midx = (XVDx.width()-1)/2.0;
    float midy = (XVDx.height()-1)/2.0;

    for (i=0;i<XVDx.width();i++)
      for (j=0;j<XVDx.height();j++)
	temp[j][i] = XVDx[j][i]*(i-midx) + 
	             XVDy[j][i]*(j-midy); 

    Amat = add_column(Amat,temp);
  }


  // Now, set up aspect ratio

  if (isset(_later_stages,B_ASPECT_RATIO)) {

    XVImage<float> temp(XVDx.height(),XVDx.width());

    // Generate the info needed to compute scale

    float midx = (XVDx.width()-1)/2.0;
    float midy = (XVDx.height()-1)/2.0;

    for (i=0;i<XVDx.width();i++)
      for (j=0;j<XVDx.height();j++)
	temp[j][i] = XVDx[j][i]*(i-midx) - 
	             XVDy[j][i]*(j-midy); 

    Amat = add_column(Amat,temp);
  }

  if (isset(_later_stages,B_SHEAR)) {

    XVImage<float> temp(XVDx.height(),XVDx.width());

    // Generate the info needed to compute scale

    float midx = (XVDx.width()-1)/2.0;
    float midy = (XVDx.height()-1)/2.0;

    for (i=0;i<XVDx.width();i++)
      for (j=0;j<XVDx.height();j++)
	temp[j][i] = XVDx[j][i]*(j-midy);

    Amat = add_column(Amat,temp);
  }


  // Compute a weighting matrix to handle getting rid of
  // window edge effects

  WI = compute_gauss_weighting(Dx_refer.width(),
				  Dx_refer.height(),
				  Dx_refer.width()/2.0,
				  Dx_refer.height()/2.0);
    WI = compute_gradient_weighting(XVDx,XVDy);
    WI = WI.scale();
    ColVector TC;
    WM = (TC << WI);
    WM = 1.0;

  // Continue setting up the system based on whether residuals need
  // to be computed.

    ref_vec << small_refer;
    ref_vec = unitize(ref_vec);
    

  if (_compute_residue || _with_check) {

// #define nonormalize
#ifndef nonormalize
    // Brightness

    ColVector tempC(Amat.n_of_rows());
    tempC = 100.0;
    Amat = add_column(Amat,tempC);

    // Contrast

    Amat = add_column(Amat,small_refer);
#endif


    // Illumination basis

    if (name != NULL) 
      get_illumination(name,Amat);

    // Compute and store the solution matrix

    Amat_inv = (Amat.t()*WM*Amat).i() * Amat.t() * WM;

    // Compute the so-called HAT matrix; note the scaling by
    // both Reduction (because I scale incoming images) is cancelled
    // by the variance reduction due to averaging.
    // Multiplying by this matrix will yield unit variance data.

    if (_with_check) {
      HAT = HAT_mat(Amat,Amat_inv,/*(1<<Reduction)**/pixel_sd);
      (Selection.diagonal()).resize(Amat.n_of_rows());
      Selection = 1.0;
    }

  }
  else {
    Matrix B(Amat.n_of_rows(),0);

    // Brightness, contrast, and illumination as above

    ColVector tempC(Amat.n_of_rows());
    tempC = 100.0;
    B = add_column(B,tempC);
    B = add_column(B,small_refer);

    if (name != NULL) 
      get_illumination(name,B);

    const Matrix &H = (B.t() * WM * B).i() * (B.t() * WM);

    // The RHS premultiplier

    const Matrix &RHS = Amat.t() * WM - (Amat.t() * WM * B) * H;

    // The LHS

    const Matrix &LHS  = RHS * Amat;

    Amat_inv = LHS.i() * RHS;
    
  }
  

  // do some display work

  if (win_live != NULL)
    win_live->open(width()-reduction(),height()-reduction(),"Live Video",NULL);

  if (win_residual != NULL) 
    win_residual->open(width()-reduction(),height()-reduction(),
		       "Residuals", NULL);


  if (win_hat != NULL) {
    win_hat->open(width()-reduction(),height()-reduction(),
		       "Residuals", NULL);
    XVImage<float> temp(height()/Reduction-1,width()/Reduction-1);
    HAT.diagonal() >> temp;
//    temp = log(temp);
    win_hat->show(make_showable(temp.scale()).magnify(Reduction));
  }


  if (win_outliers != NULL)
    win_outliers->open(width()-reduction(),height()-reduction(),
		       "Outliers",NULL);

  if (win_basis != NULL) {
    win_basis->open((width()-reduction())*Amat.n_of_cols(),
		    height()-reduction(), "Basis", NULL);
    show_A(*win_basis);
  }

}

void SSD::main_init(char *name)
{

  Image temp(width(),height());
//  temp.grab(*source(),x(),y(),orientation());
  source() -> grab(temp,int(x()),int(y()),orientation());
  main_init(temp,name);
}
/*
void
SSD::set_state(MultiScale_SSD &s)
 {
   memcpy(state,s.state,sizeof(float)*ncoords);
 }

*/
int SSD::state_init (float *state_in)
{ 
  memset(state,0,sizeof(float)*ncoords);
  memcpy(state,state_in,sizeof(float)*ncoords);

  main_init();
  return 1;

}


int SSD::image_init (Image &im,float x, float y)
{ 

  memset(state,0,sizeof(float)*ncoords);

  set_x(x);
  set_y(y);
  set_orientation(0.0);
  set_scale(1.0);
  set_aspect_ratio(1.0);
  set_shear(0.0);
  
  main_init(im);
  return 1;

}

int SSD::state_init (float in_x,float in_y, float r)
{ 
  memset(state,0,sizeof(float)*ncoords);

  set_x(in_x);
  set_y(in_y);
  set_orientation(r);
  set_scale(1.0);
  set_aspect_ratio(1.0);
  set_shear(0.0);
  
  main_init(name);
  return 1;
}


int SSD::interactive_init (CWindow &w,float r) {
  cout <<"got here: SSD::interactive_init" << endl;
  
  position* pos = w.get_region (*source(), WINX, WINY, "SSD");
  //return two corners
  state_init( (pos[0].x+pos[1].x)/2, (pos[0].y+pos[1].y)/2, r );
  return 1;
}


int SSD::acquire_image(Image &new_image)
{

  // Grab the image, adjusting back to a reasonable set of coordinates.
  // Grab_aligned does this automatically; otherwise we do it manually.

  if (orientation() == 0.0) {
    set_x(round(x()));
    set_y(round(y()));
    if (!(source()->grab_aligned(new_image,int(x()),int(y()))))
      return 0;
  }
  else {
    set_x(round(x()));
    set_y(round(y()));
    if (!(source()->grab(new_image,(int)x(),(int)y(),orientation())))
      return 0;
  }

  new_image <<= Reduction;
  return 1;
}
  
template <class T>
T logit(T x) {return log(sqr(x) + 1);}

// Combined updating function.


int SSD::update()
{ 
  int oi;

  for (oi = 0;oi<outer_iters;oi++) {

  // If Scaling, first get an Image of the right size; add a couple of 
  // pixels padding  to make sure that there is some extra space around 
  // it for resampling.  Otherwise, just grab a normal sized image.

  // These are corrected for the problems with bresenham rotation

  float sx = scalex()*scale_corr();
  float sy = scaley()*scale_corr();

  if ((sx != 1.0) || (sy != 1.0) || (shear() != 0.0) /*||
      quad_coef() != 0.0*/) {
    int width  = (int)(sx*SSD::width())+3;
    int height = (int)(sy*SSD::height())+3;

    Image grabimage(width,height);

//    width += (int)(height*abs(shear())/2.0 + 1);
//    height += (int)(fabs(quad_coef()) +1);

//    Image grabimage1(width,height);
  
//    acquire_image(grabimage1);
    if (!(acquire_image(grabimage)))
      return 0;
    
    // Now shear it.

/*    if (quad_coef() == 0.0)*/
//      grabimage1.Shear(grabimage,-shear());
/*    else
//      grabimage1.Shear_quad_h(grabimage,-quad_coef());*/

    // Now, make it the right size for our devious purposes.
    
    if (Reduction > 1)
      grabimage.scale(new_image, 1/sx, 1/sy);
    else
      grabimage.scale(new_refer, 1/sx, 1/sy);
  }
  else  {
    if (Reduction > 1)
      if (!acquire_image(new_image))
	return 0;
    else
      if (!acquire_image(new_refer))
	return 0;
  }

  // Reduce the resolution and get the smoothed image
  
  if (Reduction > 1)
    new_image.reduce_resolution(Reduction,new_refer);

  // Do the normal SSD_processing.  This should set the translational
  // part of the state as well as modifying the temporal derivatives
  // based on the offset calculations.

  new_small_refer = compressx(compressy(new_refer));

  if (_with_check)
    b << ((XVImage<int>)new_small_refer - (XVImage<int>)small_refer);
  else
    b << new_small_refer;

  int i;
  for (i=0;i<inner_iters;i++) {

    if (_with_check) 
      X  = Amat_inv * (Selection * b);
    else
      X = Amat_inv * b;


    // Right now, these two ways of doing things are mutually exclusive!!
    // The first assumes that Amat is only pose parameters
    
    if (_compute_normalized_correlation) {
      set_correlation(ref_vec.ip(unitize(b - Amat*X)));
      if (_max_misses > 0) {
	if (correlation() < _correlation_threshold)
	  _misses++;
	else
	  _misses = 0;
      }
    }
      
    // Otherwise, compute residual and possibly check it.

    if (_with_check || _compute_residue) {
      resid = b - Amat*X;
      resid /= (1<<Reduction); // because I scale it above
      if (_with_check) {
	ColVector x;
	x = (Selection * resid);
	x >> residuals;
      }
      else
	resid >> residuals;
    }

    if (_with_check) {
      int k;

      update_selection(Selection,resid,outlier_threshold,pixel_sd);

      for (k=0;k<erode_iters;k++)
	erode(Selection.diagonal(),residuals.width());
      for (k=0;k<dilate_iters;k++)
	dilate(Selection.diagonal(),residuals.width());
    }
  }

  // Store the final residue value

    if (_compute_residue ) 
      set_residual(sqrt((residuals.ip(residuals)/
			 (residuals.width()*residuals.height()))));

  // Now, propagate the updates to the parameters.



    if ((_max_misses == 0) || (_misses < _max_misses)) {

      X[0] *= Ax;
      X[1] *= Ay;

      if (isset(_later_stages,B_TRANS)) {
	set_x(x() - (X[0]*cos(orientation()) + X[1]*sin(orientation()))*scale());
	set_y(y() - (-X[0]*sin(orientation()) + X[1]*cos(orientation()))*scale());
      }
      
      int next_one = 2;

      // Then do rotation
      
      if (isset(_later_stages,B_ROT)) {
	set_orientation(orientation()-X[next_one]);
	next_one++;
      }

      if (isset(_later_stages,B_SCALE)) {
	set_scale(scale() * (1 - X[next_one]));
	next_one++;
      }

      
      // or two scales

      if (isset(_later_stages,B_ASPECT_RATIO)) {
	set_aspect_ratio(aspect_ratio() * (1+ X[next_one++]));
	next_one++;
      }

      int xx=0;
      while (next_one < X.n_of_rows())
	state[FIRST_ILLUM+(xx++)] = X[next_one++];

/*
  if (isset(_later_stages,B_SHEAR)) {
    shift_shear(Dt,(~(1<<B_SHEAR) & _later_stages));
  }

  if (isset(_later_stages,B_QUAD)) {
    shift_quad(Dt,(~(1<<B_QUAD) & _later_stages));
  }

  }
*/

    }
    else
      return 0;

    // Display stuff
    
    if (win_live != NULL)
      show_new_refer(*win_live);

    if (win_residual != NULL)
      show_residual(*win_residual);

    if (win_outliers != NULL)
      show_selection(*win_outliers);
  }    
  return 1;
  }

// Draw the window on the video.
 
int SSD::show (int color_in)
{
  int colr = color_in == no_color? color: color_in;
  float s = sin(-orientation());
  float c = cos(-orientation());
  float sw = width() * scalex();
  float sh = height() * scaley();
  float c1 = (c * sw -  s * sh)/2;
  float c2 = (s * sw +  c * sh)/2;

  float d1 = (-c * sw - s * sh)/2;
  float d2 = (-s * sw + c * sh)/2;
    
 /*
  v->line(x(),(y()-(float)WINY/2.),WINX,0.,colr);
  v->line(x(),(y()+(float)WINY/2.),WINX,0.,colr);
  v->line((x()-(float)WINX/2.),y(),WINY,M_PI/2,colr);
  v->line((x()+(float)WINX/2.),y(),WINY,M_PI/2,colr);
*/

  v->line2(x()+d1,y()+d2,x()+c1,y()+c2,colr);
  v->line2(x()-d1,y()-d2,x()-c1,y()-c2,colr);
  v->line2(x()+d1,y()+d2,x()-c1,y()-c2,colr);
  v->line2(x()-d1,y()-d2,x()+c1,y()+c2,colr);

  return 1;
}

// Clear the window on the video.

#ifdef MSSSD
static Color colorrange[6] = {blue_color,green_color,red_color,
			      purple_color,cyan_color,yellow_color};

MultiScale_SSD::MultiScale_SSD(Video *pv, int winx, int winy, 
			       int reductionmin, int reductionmax)  :
     /*PointFeature(this)*/ CompFeature(N_SSD_PARMS,"MultiScale SSD")
{
  int i;

  WINX = winx;
  WINY = winy;
  Reductions = new int[nstages = (reductionmax-reductionmin+1)];
  ssds = new SSD*[nstages];
  SSD *temp;
  
  for (i=0;i<nstages;i++) {
    Reductions[i] = 1<< (i+reductionmin);
    temp = new SSD(pv,winx,winy,Reductions[i]);
    temp->set_color(colorrange[i%6]);
    ssds[i] = 
      ((SSD *)((*this) += *temp));
    delete temp;
  }

  which = nstages - 1;  // start at lowest resolution
  set_display(pv);

}

MultiScale_SSD::MultiScale_SSD(MultiScale_SSD &s) : CompFeature(s)/*:
     PointFeature(this)*/
{
  int i;
  SSD *temp;

  WINX = s.WINX;
  WINY = s.WINY;
  Reductions = new int[s.nstages];
  ssds = new SSD*[s.nstages];
  memcpy(Reductions,s.Reductions,sizeof(int)*nstages);
  
  for (i=0;i<nstages;i++) {
    temp = new SSD(*(s.ssds[i]));
    ssds[i] = (SSD *)((*this)+= *temp);
    delete temp;
  }

  which = s.which;
}


int MultiScale_SSD::interactive_init (CWindow &w) {
  position* pos = w.get_region (*source(), WINX, WINY, "SSD");
		//return two corners
  
  state_init( (pos[0].x+pos[1].x)/2, (pos[0].y+pos[1].y)/2 );
  return 1;
}


int MultiScale_SSD::image_init(Image &x) {
  MAP(image_init(x));
  return 1;
}


MultiScale_SSD::update() {

  float nx,ny,maxv;

  nx = ((ssds[which])->x());
  ny = ((ssds[which])->y());
  
  maxv = max2(fabs(nx-lastx),fabs(ny-lasty))/(Reductions[which]);

    if (maxv < 0.1) 
      if  (which > 0)
	which --;

    if (maxv > 0.25) 
      if  (which < (nstages-1))
	which ++;

    lastx = nx;
    lasty = ny;

  ssds[which]->update();

  compute_state();
  if (is_top_level())
    propagate();

  return 1;
}


MultiScale_SSD::show(int color_in) {
  return ssds[which]->show(color_in);
}

MultiScale_SSD::constraints() {
  int i;
  for (i=0;i<nstages;i++) {
    ssds[i]->set_state(*this);
  }
  return 1;
}

#endif

