/*
    Copyright (C) 1996 Gregory D. Hager
    (Yale Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//--------------------------------------------------------------------------
//
// GlobalVelocity.hh
//
//-------------------------------------------------------------------------

#ifndef GlobalVelocity_H
#define GlobalVelocity_H 1

#include "Tracker.hh"
#include "Video.hh"
#include "Image.hh"
#include "FTypes.hh"

//----------------------------------------------------------------------------
//  GlobalVelocity base class declaration
//----------------------------------------------------------------------------


/* This class implements a highly modified Horn-style interframe motion estimator
   It does a robust estimate of global motion (median filtered from several image
   samples), and it uses a simple velocity predictor.
*/


/*

   The state vector parameter definitions are:

   0: vx
   1: vy
   2: vx residual
   3: vy residual

*/

// These definitions are for the processing bitmap

#define N_GV_PARMS 4
#define VX          0
#define VY          1
#define VX_RESIDUAL 2
#define VY_RESIDUAL 3

// A structure used internally

typedef  struct {
  float val;
  float residual;
} estimate;


class GlobalVelocity : public BasicFeature
{
private:

  // This is how the image is sampled; the first two are supplied and
  // the rest is computed from that

  int number_of_samples, sample_width;
  int offsetx, step_sizex;
  int offsety, step_sizey;

  void compute_sampling(int width_in, int &offset, int &step_size);

  // Reduction is roughly the number of pixels of motion that need to be recovered.
  // This leads to a convolution mask of 2*reduction + 3 pixels in width.

  int mask_size;
  int filter_width(int reduction) const {return max2(5,((2*reduction+3)/4)*4+1);};

  // These are the previously filtered images used in the next iteration
  // to save a little computation.

  XVImage<int> *oldsamples_x;
  XVImage<int> *oldsamples_y;
  XVImage<int> *newsamples_x;
  XVImage<int> *newsamples_y;


  // Flag to let us know if we've bootstrapped the pipeline

  int first_time;
  
  estimate robust_x_motion(XVImage<int> newsamples[],
			   XVImage<int> oldsamples[], int bias);
  
  void initialize_everything(int nsamps, int samp_width, int reduction);

 protected:

  Image Dx_refer;
  Image Dy_refer;
  Image Dt;

  inline float set_vx_residual(float x) { return state[VX_RESIDUAL] =x;};  
  inline float set_vy_residual(float x) { return state[VY_RESIDUAL] =x;};  

 public:

  GlobalVelocity(Video *pv, int nsamples, int max_velocity, 
                 int sample_width =1);
  GlobalVelocity(const GlobalVelocity &s);

  virtual ~GlobalVelocity();

  // Basicfeature required stuff

  virtual BasicFeature *dup() { return (BasicFeature*) new GlobalVelocity(*this);};

  virtual int update();
  virtual int show(int color_in=no_color) //{return 1;};
    {
      int xval = (int)abs(vx()*5);
      int yval = (int)abs(vy()*5);
      source()->line(xval/2+1,0,xval,0,color_in);
      source()->line(xval/2+1,1,xval,0,color_in);
      source()->line(xval/2+1,2,xval,0,color_in);

      source()->line(0,yval/2+1,yval,M_PI/2.0,color_in);
      source()->line(1,yval/2+1,yval,M_PI/2,color_in);
      source()->line(2,yval/2+1,yval,M_PI/2,color_in);

      return 1;
    }

#ifdef IRLS
  inline float set_threshold(float x) {return outlier_threshold = x;};
  inline float set_pixel_sd(float x) {return pixel_sd = x;};
  inline void  set_iiter(int x) {inner_iters = x;};
  inline void  set_oiter(int x) {outer_iters = x;};

  inline void  set_morph(int e,int d) {erode_iters = e;
				       dilate_iters = d;};
#endif

  inline virtual float vx() const { return state[VX]; }
  inline virtual float vy() const { return state[VY]; }
  inline virtual float vx_residual() const { return state[VX_RESIDUAL]; }
  inline virtual float vy_residual() const { return state[VY_RESIDUAL]; }
  inline virtual float set_vx(float x_in) { return state[VX] = x_in; }
  inline virtual float set_vy(float y_in) { return state[VY] = y_in; }

};


#endif
