/*
    Copyright (C) 1996 Gregory D. Hager
    (Yale Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright abnd permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//--------------------------------------------------------------------------
//
// SSD.cc
//
//-------------------------------------------------------------------------

#include <fstream.h>
#include "GlobalVelocity.hh"
#include "XVImage.hh"
//#define Txx int
//#include "XVImage.icc"
//#undef Txx

//----------------------------------------------------------------------------
//  Utility functions
//----------------------------------------------------------------------------


void 
GlobalVelocity::compute_sampling(int width_in, int &offset, int &step_size)
{
  width_in -= sample_width+1;
  step_size = width_in/number_of_samples;
  offset = (width_in - step_size*number_of_samples)/2;
}
  


//-----------------------------------------------------------------------------
//  Constructors.
//-----------------------------------------------------------------------------

void
GlobalVelocity::initialize_everything(int n_samples, int max_velocity,
				      int sample_width) 
{

  int i;

  (*this).number_of_samples = n_samples;
  (*this).sample_width = sample_width;
  (*this).mask_size = filter_width(max_velocity);
  
  compute_sampling(source()->width(),offsety,step_sizey);
  compute_sampling(source()->height(),offsetx,step_sizex);

  oldsamples_x = new XVImage<int>[n_samples];
  oldsamples_y = new XVImage<int>[n_samples];
  newsamples_x = new XVImage<int>[n_samples];
  newsamples_y = new XVImage<int>[n_samples];

  // Size the new images right; the others will be sized by filtering

  for (i=0;i<number_of_samples;i++) {
    newsamples_x[i].resize(sample_width,source()->width());
    oldsamples_x[i].resize(sample_width,source()->width()-mask_size+1);
    newsamples_y[i].resize(sample_width,source()->height());
    oldsamples_y[i].resize(sample_width,source()->height()-mask_size+1);
  }

  first_time = 1;

}

GlobalVelocity::GlobalVelocity(Video *pv, int n_samples, int max_velocity,
			       int sample_width) :
     BasicFeature(N_GV_PARMS,pv,"GlobalVelocity_Tracker")
{ 

  initialize_everything(n_samples, max_velocity, sample_width);

}


GlobalVelocity::GlobalVelocity(const GlobalVelocity &s): 
     BasicFeature (s)
{ 

  initialize_everything(s.number_of_samples, (s.mask_size-3)/2, s.sample_width);

}

GlobalVelocity::~GlobalVelocity()
{
  delete newsamples_x;
  delete newsamples_y;
  delete oldsamples_x;
  delete oldsamples_y;
}

estimate
motion_est_x(const XVImage<int> newimage, XVImage<int> &oldfilteredimage, 
	     int filter_width, int bias)
{

  XVImage<int> newfilteredimage;
  XVImage<int> nf_cropped;
  XVImage<int> of_cropped;

  XVImage<int> dt;
  XVImage<int> dx;
  double trans;
  estimate est;

  // filter the new image

  double_box_filter_x(newfilteredimage, newimage,filter_width);

  // Get the pieces we'll work with

  if (bias > 0) {
    nf_cropped = newfilteredimage.subimage(bias,0,
					   newfilteredimage.width(),
					   newfilteredimage.height());
    of_cropped = oldfilteredimage.subimage(0,0,
					   oldfilteredimage.width()-bias,
					   oldfilteredimage.height());
  }
  else   {
    nf_cropped = newfilteredimage.subimage(0,0,
					   newfilteredimage.width()+bias,
					   newfilteredimage.height());
    of_cropped = oldfilteredimage.subimage(-bias,0,
					   oldfilteredimage.width(),
					   oldfilteredimage.height());
  } 

  // Compute all the derivatives we need

  box_filter(dt,nf_cropped-of_cropped,1,3);
  prewitt_filter_dx(dx, nf_cropped+of_cropped);

  // Save the filtered image (extra copy operation here that we could save)

  oldfilteredimage = newfilteredimage;

  // Compute the motion
  // The gain here is due to the need to deal with filtering factors
  // as we're not using unit-sum filters.

  double dxdt = (double)dx.ip(dt);
  double dxdx = (double)dx.ip(dx);
  double dtdt = (double)dt.ip(dt);

  est.val =  -dxdt/dxdx; 
  est.residual = dtdt + dxdx*est.val*est.val-2*dxdt*est.val;
  est.val/= .75;
  est.val += bias;

  return est;
}



int
comp_val(const void *x, const void *y)
{
  return (((estimate *)x)->val < ((estimate*)y)->val) ? 
       -1 : ((((estimate *)x)->val == ((estimate *)y)->val) ? 0 : 1);
}

int
comp_residual(const void *x, const void *y)
{
  
  return (((estimate*)x)->residual < ((estimate *)y)->residual) ? -1 : 
    ((((estimate*)x)->residual == ((estimate*)y)->residual) ? 0 : 1);
}

estimate
robust_estimate(estimate *x,int nmem)
{
 
  // Get the good fits at the bottom

  qsort(x,nmem,sizeof(double),comp_residual);

  // Get the median of the lower half (this should be done better)

  nmem/=2;
  qsort(x,nmem,sizeof(double),comp_val);

  //  if ((nmem/2)*2 == nmem)
    //    return (x[nmem/2-1] + x[nmem/2])/2.0;
    return x[nmem/2];
      //  else
      //    return x[nmem/2];
}

estimate
GlobalVelocity::robust_x_motion(XVImage<int> newsamples[],
				XVImage<int> oldsamples[], int bias)
{
  int sample_pt;
  int i;
  estimate xs[number_of_samples];

  for (i=0;i<number_of_samples;i++) {
    //    cout << i << " " ;
    xs[i] = motion_est_x(newsamples[i],	 oldsamples[i],	 mask_size, bias);
  }

  return robust_estimate(xs,number_of_samples);

}
  
void
transpose_grab(Video &v,XVImage<int> &im,int x, int y)
{
  int r = im.height();
  int c = im.width();

  im.reshape(c,r);
  v.grab_region(im,x,y);
  im.reshape(r,c);
}
  

int 
GlobalVelocity::update()
{

  int i;
  estimate e;

  // Check if we've been here before and get initial data if not

  if (first_time) {
    XVImage<int> tempx(newsamples_x[0]);
    XVImage<int> tempy(newsamples_y[0]);

    for (i=0;i<number_of_samples;i++) {
      source()->grab_region(tempx,0,offsetx+i*step_sizex);
      transpose_grab(*(source()),tempy,offsety+i*step_sizey,0);
    
      double_box_filter_x(oldsamples_x[i], tempx, mask_size);
      double_box_filter_x(oldsamples_y[i], tempy, mask_size);
    }

    first_time = 0;
  }

  // Grab the new samples

  for (i=0;i<number_of_samples;i++) {
    //    cout << offsetx+i*step_sizex << " " << offsety+i*step_sizey << endl;
    source()->grab_region(newsamples_x[i],0,offsetx+i*step_sizex);
    transpose_grab(*(source()),newsamples_y[i],offsety+i*step_sizey,0);
  }

  e = robust_x_motion(newsamples_x,oldsamples_x,(int)vx());
  set_vx(e.val);
  set_vx_residual(e.residual);

  e = robust_x_motion(newsamples_y,oldsamples_y,(int)vy());

  set_vy(e.val);
  set_vy_residual(e.residual);

  return 1;
}
		      
			  
