#include <Pipe_op.hh>
#include <Tools.hh>
#include <RGB2int.hh>
#include <math.h>


#ifndef DBL_MAX
#include <float.h>
#endif


// need this in addition to Image::subimage for const Image objects
Image subimage(const Image &im, int lx, int ly, int ux, int uy) {
  int *im_data;
  int im_width;
  int i,j;

  if ((lx < 0) || (ly < 0) ||
      (ux >= im.width()) || (uy >= im.height()))
    panic("subimage: attempt to acquire subimage outside the range of parent image.");

  Image processed(ux-lx+1,uy-ly+1);

  im_width = im.width();
  im_data = im.data();

  uy++;

  for (i=ly;i<uy;i++) {
    memcpy(processed[i-ly],im_data+(i*im_width)+lx,(sizeof(int)*(ux-lx+1)));
  }

  return processed;
}

// pixel square root
Image sqroot(const Image& im)
{
  unsigned int im_size;
  int *processed_data;
  int i, j;
  
  Image processed = im;

  im_size = processed.width() * processed.height();

  processed_data = processed.data();

  for (i=0; i<im_size; i++) {
    *processed_data = (int) rint(sqrt((double) *processed_data));
    processed_data++;
  }
  return processed;
}

// change RGB color image to grayscale

Image grayscale(const Image& im) {
  double mono_val;
  int *im_data, *gray_image_data;
  int i, im_size;

  Image gray_image(im.width(),im.height());

  im_data = im.data();
  gray_image_data = gray_image.data();

  im_size = im.width()*im.height();

  // average RGB values to get grayscale value
  for (i=0; i<im_size; i++) {
    mono_val = compute_red(*im_data);
    mono_val += compute_green(*im_data);
    mono_val += compute_blue(*im_data);
    mono_val /= 3;

    *gray_image_data = (int) rint(mono_val);
    im_data++;
    gray_image_data++;
  }
  return gray_image;
}

// Perform an image projection --- last one determines
// whether to normalize the result --- since we
// are using image, right now this normalizes to
// a fixed value of magnitude <= 1000

Image project(const Image& im,float rc, float gc, float bc, int normalize) {
  double mono_val;
  int *im_data, *gray_image_data;
  int i, im_size;
  float r,b,g;
  double norm = sqrt(rc*rc+bc*bc+gc*gc);
  rc/=norm;
  bc/=norm;
  gc/=norm;

  Image gray_image(im.width(),im.height());

  im_data = im.data();
  gray_image_data = gray_image.data();

  im_size = im.width()*im.height();

  // average RGB values to get grayscale value
  for (i=0; i<im_size; i++,im_data++) {
    mono_val = (r = compute_red(*im_data)) *rc;
    mono_val += (g = compute_green(*im_data)) *gc;
    mono_val += (b = compute_blue(*im_data)) *bc;

    // Hope we never see an image with zeros in it..
    if (normalize) {
      mono_val *= 1000.0/sqrt(r*r+g*g+b*b);
    }
    *gray_image_data++ = (int) mono_val;
  }
  return gray_image;
}

// Binary Threshold
Image threshold(const Image& im, int thresh, int val)
{
  unsigned int im_size;
  int *im_data, *processed_data;
  int i, j;
  
  Image processed(im.width(),im.height());

  im_size = im.width() * im.height();

  im_data = im.data();
  processed_data = processed.data();

  for (i=0; i<im_size; i++) {
    // below threshold, black background
    if (*im_data < thresh)
      *processed_data = 0;
    else
      *processed_data = val;

    im_data++;
    processed_data++;
  }
  return processed;
}

// convolve im with mask
Image convolve(const Image &im, const Image& mask)
{
  Image newim(im);
  return im.convolve(newim,mask);
}
Image erode(const Image &im, const Image& mask)
{
  Image newim(im); 
  return im.erode(newim,mask);
}
/*
Image convolve(const Image& im, const Image& mask)
{
  unsigned int im_height, im_width, mask_height, mask_width;
  int mask_midx, mask_midy;
  int *im_data, *mask_data, *processed_data;
  int i, j, k, k_end, l, l_end;

  im_height = im.height();
  im_width = im.width();

  Image processed = im;  // convolved image, init with Image im values

  mask_height = mask.height();
  mask_width = mask.width();
 
  if ((mask_width % 2) == 0)
    mask_midx = mask_width/2;
  else
    mask_midx = (mask_width-1)/2;
  if ((mask_height % 2) == 0)
    mask_midy = mask_height/2;
  else
    mask_midy = (mask_height-1)/2;
 
  im_data = im.data();
  processed_data = processed.data();

  for (i=0; i<im_height; i++) {
    for (j=0; j<im_width; j++) {
      // pixel out of bounds, skip; should we try renormalizing?
      if (((i-mask_midy) < 0) || ((i+mask_midy) >= im_height))
	break;
      // pixel out of bounds, skip; should we try renormalizing?
      if (((j-mask_midx) < 0) || ((j+mask_midx >= im_width)))
	continue;

      mask_data = mask.data();

      k_end = i+mask_midy+1;
      l_end = j+mask_midx+1;

      processed_data[i*im_width+j] = 0;

      // convolve
      for (k=i-mask_midy; k<k_end; k++) {
	for (l=j-mask_midx; l<l_end; l++) {
	  processed_data[i*im_width+j] += *mask_data++ 
	    * (im_data[k*im_width+l]);
	}
      }
    }
  }
  return processed;
}
	*/			   

// x derivative
Image dx(const Image& im)
{
  static Image mask(3,1);
  static int make_mask = 1;

  // make masks the first time around
  if (make_mask) { 
    int *mask_data = mask.data();

    *mask_data++ = 1;
    *mask_data++ = 0;
    *mask_data++ = -1;

    make_mask = 0;  // only make the mask once
  }

  // convolve im with mask
  Image processed = convolve(im,mask);

  return processed;
}

// y derivative
Image dy(const Image& im)
{
  static Image mask(1,3);
  static int make_mask = 1;

  // make masks the first time around
  if (make_mask) { 
    int *mask_data = mask.data();

    *mask_data++ = 1;
    *mask_data++ = 0;
    *mask_data++ = -1;

    make_mask = 0;  // only make the mask once
  }

  // convolve im with mask
  Image processed = convolve(im,mask);

  return processed;
}

// Sobel edge detector
Image sobel(const Image& im)
{
  static Image mask_x(3,3);
  static Image mask_y(3,3);
  static int make_mask = 1;
  unsigned int im_size;
  int *processed_data, *processed_y_data;
  int i;

  // make masks the first time around
  if (make_mask) { 
    int *mask_x_data = mask_x.data();
    int *mask_y_data = mask_y.data();

    // make x mask
    *mask_x_data++ = -1;
    *mask_x_data++ = -2;
    *mask_x_data++ = -1;
    *mask_x_data++ = 0;
    *mask_x_data++ = 0;
    *mask_x_data++ = 0;
    *mask_x_data++ = 1;
    *mask_x_data++ = 2;
    *mask_x_data++ = 1;

    // make y mask
    *mask_y_data++ = -1;
    *mask_y_data++ = 0;
    *mask_y_data++ = 1;
    *mask_y_data++ = -2;
    *mask_y_data++ = 0;
    *mask_y_data++ = 2;
    *mask_y_data++ = -1;
    *mask_y_data++ = 0;
    *mask_y_data++ = 1;

    make_mask = 0;  // only make the mask once
  }

  // look for edges in x direction first by convolving
  Image processed = convolve(im,mask_x);

  // check y direction by convolving
  Image processed_y = convolve(im,mask_y);

  // take absolute values for magnitude of edge values
  processed.abs();
  processed_y.abs();

  processed_y_data = processed_y.data();
  processed_data = processed.data();

  im_size = im.width() * im.height();

  // take max of x and y edge values
  for (i=0; i<im_size; i++) {
    if (*processed_y_data > *processed_data)
      *processed_data = *processed_y_data;
    
    processed_data++;
    processed_y_data++;
  }
  return processed;
}
#ifdef BLORT
//template <class T>
//XVImage<T> variance(const XVImage<T> &input,int xsize, int ysize)
Image variance(const Image &input,int xsize, int ysize)
{
  //  XVImage<T> result1(input.height(),input.width());
 Image result1(input.width(),input.height());

  // Just to make life easy, assume we want an odd number

  int xhalf, yhalf;
  xsize = (xhalf = xsize/2)*2 + 1;
  ysize = (yhalf = ysize/2)*2 + 1;

  register double mean = 0,variance = 0,temp;

  int i,j;
  //  T rowptrin_back rowptrin_front,rowptrout, rowend;
  int *rowptrin_back, *rowptrin_front, *rowptrout, *rowend;
  double div = xsize;//*ysize;
  double divsq = div*div;

  // Do the rows

  for (i=0;i<input.height();i++) {
    variance = mean =0;
    rowptrin_back = rowptrin_front = input[i];
    rowend = rowptrin_back+input.width()-xhalf;
    rowptrout = result1[i];

    // pad the output for now

    for (j=0;j<xhalf;j++)
      *rowptrout++ = 0;

    // Load up the variance

    for (j=0;j<xsize;j++,rowptrin_front++) {
      mean += *rowptrin_front;
      variance += sqr(*rowptrin_front);
    }
    mean/=div;
    variance/=div;

    // save the first value

    *rowptrout++ = variance - mean*mean;

    // Do the line

    for (;rowptrin_front < rowend;rowptrin_back++,rowptrin_front++) {
      mean += (*rowptrin_front - *rowptrin_back)/div;
      variance += (sqr(*rowptrin_front) - sqr(*rowptrin_back))/div;
      //      cout << div << " " << mean << " " << sqrt(variance) << endl;
      *rowptrout++ = variance - mean*mean;
    }
    for (j=0;j<xhalf;j++)
      *rowptrout++ = 0;

  }
  return result1;

}
#endif
