#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <XConsole.hh>
#include <XWindow.hh>
#include <Video.hh>
#include <Image.hh>


template < class T >
class Unbounded_Array {
private:
  int size_block;
  int num_elts;
  int size;
  T *array;
public:
  Unbounded_Array(void);
  T &operator[](int index);
};


struct Pipe_Params {
  int threshold;
  int factor;
  int width;
  int height;
  float rc,bc,gc;
  int normalize;
  double sigma_x;
  double sigma_y;
  double maskx,masky;
  int lx, ly, ux, uy;
  int translation;
};

class Pipe {
protected:
  int id;
  Unbounded_Array<Pipe*> preds;
  unsigned int num_preds;
  Image output;
  int frame_count, input_frame_count;
  unsigned int height, width;
  XWindow *w;
  virtual const char *get_name()  = 0;
  const char *display_label;
  int normalized_display;
public:
  Pipe(void);
  void add_pred(Pipe *p_in);    // Add a predecessor to the pipe
  virtual void  main_update();  // This is what is called internally
  virtual void update(void) {};   // This is what the user writes
  virtual void modify_params(Pipe_Params &pp);
  const Image &get_output(void) const;
  inline int get_frame_count(void);
  unsigned int get_height(void);
  unsigned int get_width(void);
  int get_id(void);
  void display_output(XConsole &c, int normalize = 1,char *label = NULL);
};

inline Pipe::Pipe()
{
  frame_count = 0;
  input_frame_count = 0;
  num_preds = 0;
  w = NULL;
}

inline void Pipe::modify_params(Pipe_Params &pp) 
{
  cerr << "No parameters to modify in " << get_name() << endl;
}

inline void Pipe::add_pred(Pipe *p_in)
{
  preds[num_preds++] = p_in;
  input_frame_count = p_in->get_frame_count();
}

inline void Pipe::main_update()
{
  char buff[200];
  update();
  if (w != NULL) {
    sprintf(buff,"%d %s",frame_count,display_label);
    if (normalized_display) {
      w->show(output.scale(),buff);
    }
    else
      w->show(output,buff);
  }
}

inline int Pipe::get_id(void)
{
  return id;
}

inline const Image &Pipe::get_output(void) const
{
  return output;
}

inline int Pipe::get_frame_count(void)
{
  return frame_count;
}

inline unsigned int Pipe::get_height(void)
{
  return height;
}

inline unsigned int Pipe::get_width(void)
{
  return width;
}


class Pipeline {
protected:
  Unbounded_Array<Pipe*> pipes;
  unsigned int num_pipes;
  Pipe *first, *last;
  Image input;
  Image output;
  int frame_count, v_frame_count;
  Video *v;
  float xloc, yloc, angle;
  int ws, hs;
  unsigned int v_height, v_width;
  unsigned int height, width;

public:
  Pipeline(Video *v_in);
  Pipeline(Pipeline &pl_in);
  Pipeline(Pipe *p_in);
  void add_pipe(Pipe *p_in);
  Pipe *get_last(void);
  void grab_video(void);
  void update(void);
  Image &get_output(void);
  Image &get_input(void);
  unsigned int get_height(void);
  unsigned int get_width(void);
  int get_frame_count(void);
  int get_v_frame_count(void);
  unsigned int get_num_pipes(void);
  Pipe *get_pipe(int i);
  Video *get_video(void);
  Pipe *operator[](int id_in);
  void display_output(XConsole &c, int normalize = 1, char *label = NULL);
};

inline Pipe *Pipeline::operator[](int id_in)
{
  for (unsigned int i=0; i<num_pipes; i++) {
    if (pipes[i]->get_id() == id_in)
      return pipes[i];
  }
  cerr << "Error: Pipe id = " << id_in << " not found.\n";
  return pipes[0];
}

inline void Pipeline::grab_video(void)
{
  if (v != NULL) {
    v->grab(input,(int)xloc,(int)yloc,angle,ws,hs);
    v_frame_count++;
  }
}

inline void Pipeline::update(void)
{
  last->main_update();
  
  output = last->get_output();

  frame_count++;
}

inline Pipe *Pipeline::get_last(void)
{
  return last;
}

inline Image &Pipeline::get_input(void)
{
  return input;
}

inline Image &Pipeline::get_output(void)
{
  return output;
}

inline unsigned int Pipeline::get_height(void)
{
  return height;
}

inline unsigned int Pipeline::get_width(void)
{
  return width;
}

inline int Pipeline::get_frame_count(void)
{
  return frame_count;
}

inline int Pipeline::get_v_frame_count(void)
{
  return v_frame_count;
}

inline unsigned int Pipeline::get_num_pipes(void)
{
  return num_pipes;
}

inline Pipe *Pipeline::get_pipe(int i)
{
  return pipes[i];
}

inline Video *Pipeline::get_video(void)
{
  return v;
}


// mono video class constructed with a pipeline (graph) as parameter
// taken from MonoVideo class and MPEG device class
class Pipeline_Video : public ColorVideo {
  Image vimage;
  Pipeline *pl;
  int auto_step_mode;       // Whether to just increment with each grab
  XConsole con;
  
  XWindow *win;
  
protected:
 
  int map_color(Color colin);
  void construct_name(char *buff);
  
public:
 
  Pipeline_Video(Pipeline *pl_in, int height, int width);
  
  ~Pipeline_Video();

  char *device_info();
  
  // Set it up so the default video interfaces work
  
  virtual int pre_grab();
  
  virtual void *current_frame_ptr();
 
  // this one does a grab on the image, but doesn't forward sequence
  
  virtual int grab_static(int *im,
			  int x, int y, int width, int height,
			  float angle = 0.0,
			  int sampw = 1, int samph = 1,
			  short int mask = 0xff);
  
  // Set auto_step_mode
  int auto_step(int bval);
 
  // forward() has to be called explicitly to forward image sequence
 
  int forward();
  
  int show (int x, int y,
	    int width, int height,          // show image 
             int *image);

 void point (int x, int y, Color color = 1, int size=1);
 
  void line (float x, float y, int length,   // draw an overlayed line
             float angle,
             int color = 1,int samp = 1);
 
  void clearline (float x, float y,          // clear overlayed line
                  int length, float angle, int samp = 1);

  // Draw and clear a circle
 
  void circle (float x, float y, int r, int color = 1);
  void clearcircle(float x, float y,  int r);
 
  // Draw and clear a cubic-polynomial segment
 
  void cubic (float x, float y, int length, float angle,
              float a, float b, float c, int color = 1);
  void clearcubic (float x, float y, int length, float angle,
		   float a, float b, float c);
};

inline Pipeline_Video::~Pipeline_Video(void)
{

}

inline int Pipeline_Video::forward()
{
  pl->update();

  vimage = pl->get_output();

  return 1;
}
 
inline int Pipeline_Video::show (int x, int y,int width, int height, int *im)
{ 
  if (!(in_view(x,y,width,height)))
    return 0;
  
  win->show(im,width,width,height);
  
  return 1;
}

inline int Pipeline_Video::map_color(Color colin)
{
  switch (colin) {
  case default_color:
    //    case green_color:
    return 2;
  case default_color2:
    //    case red_color:
    return 1;
  case default_color3:
    //    case blue_color;
    return 3;
  case erase_color:
    return 0;
  default:
    return colin;
  }
}

inline void Pipeline_Video::construct_name(char *buff)
{
    sprintf(buff,"Pipeline Video");
}

inline char *Pipeline_Video::device_info()
{
  return "Pipeline Video Class";
}
    
inline int Pipeline_Video::pre_grab()
{
  if (auto_step_mode)
    return forward();
  else return 1;
}

inline void *Pipeline_Video::current_frame_ptr()
{
  return vimage.data();
}

inline int Pipeline_Video::grab_static(int *im,
				       int x, int y, int width, int height,
				       float angle = 0.0,
				       int sampw = 1, int samph = 1,
				       short int mask = 0xff)
{
  int tmp = auto_step_mode, res;
  auto_step_mode = 0;
  res = grab(im, x, y, width, height, angle, sampw, samph, mask);
  auto_step_mode = tmp;
  return res;
}

inline int Pipeline_Video::auto_step(int bval)
{
  return auto_step_mode = bval;
}
  
inline void Pipeline_Video::point (int x, int y, Color color, int size)
{
 
}
 
inline void Pipeline_Video::line (float x, float y, int length, float angle,
                     Color color, int samp)
{                     
  win->line((int) x, (int)y, length, angle, color);
}

inline void Pipeline_Video::clearline(float x, float y,    
				      int length, float angle, int samp = 1)
{
  line (x, y, length, angle, 0, samp);
}
 
inline void Pipeline_Video::circle (float x, float y, int r, Color color)
{
 
}
 
inline void Pipeline_Video::clearcircle(float x, float y,  int r)
{
  circle(x,y,r,0);
}

inline void Pipeline_Video::cubic (float x, float y, int length, float angle,
                   float a, float b, float c, Color color)
{
  
}

inline void Pipeline_Video::clearcubic(float x, float y, int length, 
				       float angle, float a, float b, float c)
{
  cubic(x,y,length,angle,a,b,c,0);
}

class Pipe_Raw : public Pipe {
protected:
  Pipeline *pl;
  virtual const char *get_name() {return "Pipe_Raw";};	
public:
  Pipe_Raw(Pipeline *pl_in);
  virtual void update(void);
};


class Pipe_Plus : public Pipe {
  virtual const char *get_name()  {return "Pipe_Plus";};	
public:
  Pipe_Plus(Pipe *p1_in, Pipe *p2_in);
  virtual void update(void);
};

class Pipe_Minus : public Pipe {
  virtual const char *get_name()  {return "Pipe_Minus";};	
public:
  Pipe_Minus(Pipe *p1_in, Pipe *p2_in);
  virtual void update(void);
};

class Pipe_Mult : public Pipe {
  virtual const char *get_name()  {return "Pipe_Mult";};	
public:
  Pipe_Mult(Pipe *p1_in, Pipe *p2_in);
  virtual void update(void);
};

class Pipe_And : public Pipe {
  virtual const char *get_name()  {return "Pipe_And";};	
public:
  Pipe_And(Pipe *p1_in, Pipe *p2_in);
  virtual void update(void);
};

class Pipe_Sq : public Pipe {
  virtual const char *get_name()  {return "Pipe_Sq";};	
public:
  Pipe_Sq(Pipe *p_in);
  virtual void update(void);
};

class Pipe_Sqrt : public Pipe {
  virtual const char *get_name()  {return "Pipe_Sqrt";};	
public:
  Pipe_Sqrt(Pipe *p_in);
  virtual void update(void);
};

class Pipe_Subimage : public Pipe {
  virtual const char *get_name()  {return "Pipe_Subimage";};	
protected:
  int lx, ly, ux, uy;
public:
  Pipe_Subimage(int id_in, Pipe *p_in, int lx_in, int ly_in, int ux_in, 
		int uy_in);
  virtual void update(void);
  virtual void modify_params(Pipe_Params &pp);
};

class Pipe_Delay : public Pipe {
  virtual const char *get_name()  {return "Pipe_Delay";};	
protected:
  Image delayed;
public:
  Pipe_Delay(Pipe *p_in);
  virtual void update(void);
};

class Pipe_Threshold : public Pipe {
  virtual const char *get_name()  {return "Pipe_Theshold";};	
protected:
  int thresh;
  int val;
public:
  Pipe_Threshold(int id_in, Pipe *p_in, int thresh_in, int val_in = 255);
  virtual void update(void);
  virtual void modify_params(Pipe_Params &pp);
};

inline void Pipe_Threshold::modify_params(Pipe_Params &pp)
{
  thresh = pp.threshold;
}

class Pipe_Reduce : public Pipe {
  virtual const char *get_name()  {return "Pipe_Reduce";};	
protected:
  int factor;
public:
  Pipe_Reduce(int id_in, Pipe *p_in, int factor_in);
  virtual void update(void);
  virtual void modify_params(Pipe_Params &pp);
};

inline void Pipe_Reduce::modify_params(Pipe_Params &pp)
{
  factor = pp.factor;
}

class Pipe_Project : public Pipe {
  virtual const char *get_name()  {return "Pipe_Project";};	
protected:
  float rc,gc,bc;
  int normalize;
public:
  Pipe_Project(int id_in, Pipe *p_in, float rc, float gc, float bc,int norm = 0);
  virtual void update(void);
  virtual void modify_params(Pipe_Params &pp);
};

inline void Pipe_Project::modify_params(Pipe_Params &pp)
{
  rc = pp.rc;
  gc = pp.gc;
  bc = pp.bc;
  normalize = pp.normalize;
}

class Pipe_Abs : public Pipe {
  virtual const char *get_name()  {return "Pipe_Abs";};	
public:
  Pipe_Abs(Pipe *p_in);
  virtual void update(void);
};


class Pipe_Gray : public Pipe {
  virtual const char *get_name()  {return "Pipe_Gray";};	
public:
  Pipe_Gray(Pipe *p_in);
  virtual void update(void);
};


class Pipe_Dx : public Pipe {
  virtual const char *get_name()  {return "Pipe_Dx";};	
public:
  Pipe_Dx(Pipe *p_in);
  virtual void update(void);
};

class Pipe_Dy : public Pipe {
  virtual const char *get_name()  {return "Pipe_Dy";};	
public:
  Pipe_Dy(Pipe *p_in);
  virtual void update(void);
};

class Pipe_Sobel : public Pipe {
  virtual const char *get_name()  {return "Pipe_Sobel";};	
public:
  Pipe_Sobel(Pipe *p_in);
  virtual void update(void);
};

class Pipe_Gaussian : public Pipe {
  virtual const char *get_name()  {return "Pipe_Gaussian";};	
protected:
  Image maskx, masky;
  double sigmax, sigmay;
  void make_mask(int mask_width, int mask_height);
public:
  Pipe_Gaussian(int id_in, Pipe *p_in, 
		int width_in, double sigmax_in,
		int height_in = -1, double sigmay_in = -1);
  virtual void update(void);
  virtual void modify_params(Pipe_Params &pp);
};

class Pipe_Convolve : public Pipe {
  virtual const char *get_name()  {return "Pipe_Convolve";};	
protected:
  Image mask;
  double sigma;
  unsigned int mask_height, mask_width;
  int mask_midx, mask_midy, mask_weight;
  void make_mask(void);
public:
  Pipe_Convolve(int id_in, Pipe *p_in, const Image &mask_in);
  virtual void update(void);
};

class Pipe_Erode : public Pipe {
  virtual const char *get_name()  {return "Pipe_Erode";};	
protected:
  Image mask;
  unsigned int mask_height, mask_width;
  int mask_midx, mask_midy;
  void make_mask(void);
public:
  Pipe_Erode(int id_in, Pipe *p_in, const Image &mask_in);
  virtual void update(void);
};
/*
class Pipe_Variance : public Pipe {
  virtual const char *get_name()  {return "Pipe_Variance";};	
protected:
  unsigned int mask_height, mask_width;
public:
  Pipe_Variance(int id_in, Pipe *p_in, int height_in, int width_in);
  virtual void update(void);
  virtual void modify_params(Pipe_Params &pp);
};
*/
class Pipe_Laplacian_Gaussian : public Pipe {
  virtual const char *get_name()  {return "Pipe_Laplacian_Gaussian";};	
protected:
  Image mask;
  double sigma;
  unsigned int mask_height, mask_width;
  int mask_midx, mask_midy, mask_weight;
  void make_mask(void);
public:
  Pipe_Laplacian_Gaussian(int id_in, Pipe *p_in, int width_in, 
			  double sigma_in);
  virtual void update(void);
  virtual void modify_params(Pipe_Params &pp);
};

class Pipe_Convtrack : public Pipe {
  virtual const char *get_name()  {return "Pipe_Track";};	
protected:
  int lx, ly, ux, uy;
  int trans;
  int lx_best, ly_best, ux_best, uy_best;
  unsigned int mask_height, mask_width, im_height, im_width, mask_size;
  double mask_val;
  void make_mask(void);
public:
  Pipe_Convtrack(int id_in, Pipe *p_in, int lx_in, int ly_in, int ux_in, 
		 int uy_in, int trans_in);
  virtual void update(void);
  virtual void modify_params(Pipe_Params &pp);
};


