#include <Pipe.hh>
#include <Pipe_op.hh>
#include <math.h>


#ifndef DBL_MAX
#include <float.h>
#endif


template < class T >
Unbounded_Array < T >::Unbounded_Array(void) {
  size_block=3;
  size = size_block;
  num_elts = 0;
  array = new T[size_block];
}
 
template < class T >
T &Unbounded_Array < T >::operator[](int index)
{
  int i;
 
  if (index >= size) {
    T *array_old = new T[size];
    memcpy(array_old,array,size*sizeof(T));
    delete array;
    size += size_block;
    array = new T[size];
    memcpy(array,array_old,size*sizeof(T));
  }
  return array[index];
}

void Pipe::display_output(XConsole &c, int normalize, char *labelin)
{
  w = new XWindow(c);
  normalized_display = normalize;
  if (labelin != NULL)
    display_label = labelin;
  else
    display_label = get_name();

  //  w->Set_Name((char *)get_name()); //Set_Name should be const
}


Pipeline::Pipeline(Video *v_in)
{
  frame_count = 0;
  v = v_in;
  ws = hs = 1;
  angle = 0;
  v_height = v->height();
  v_width = v->width();
  xloc = v_width/2.0;
  yloc = v_height/2.0;
  height = v_height;
  width = v_width;
  v_frame_count = 0;
  input.resize(height,width);
  output.resize(height,width);
  num_pipes = 0;
  first = last = NULL;
  add_pipe(new Pipe_Raw(this));
}

Pipeline::Pipeline(Pipeline &pl_in)
{
  frame_count = 0;
  v = pl_in.v;
  if (v != NULL) {
    ws = hs = 1;
    angle = 0;
    v_height = v->height();
    v_width = v->width();
    xloc = v_width/2.0;
    yloc = v_height/2.0;
    height = v_height;
    width = v_width;
    v_frame_count = pl_in.get_v_frame_count();
  }
  else {
    height = pl_in.height;
    width = pl_in.width;
  }

  input.resize(height,width);
  output.resize(height,width);
  input = pl_in.input;
  output = pl_in.output;
  num_pipes = 0;
  first = last = NULL;

  for (int i=0; i<pl_in.num_pipes; i++)
    add_pipe(pl_in.pipes[i]);
}

Pipeline::Pipeline(Pipe *p_in)
{
  frame_count = 0;
  v = NULL;
  height = p_in->get_height();
  width = p_in->get_width();
  input.resize(height,width);
  output.resize(height,width);
  num_pipes = 0;
  first = last = NULL;
  add_pipe(p_in);
}


void Pipeline::display_output(XConsole &c, int normalize, char *label)
{
  get_last()->display_output(c, normalize, label);
}

void Pipeline::add_pipe(Pipe *p_in)
{
  pipes[num_pipes] = p_in;

  if (num_pipes == 0)
    first = p_in;

  last = p_in;
  height = last->get_height();
  width = last->get_width();
  output.resize(height,width);
  num_pipes++;
}


Pipeline_Video::Pipeline_Video(Pipeline *pl_in, int height, int width) 
  : ColorVideo(4)
{
  // initializations
  pl = pl_in;
  auto_step_mode = 1;

  win = new XWindow(con);

  // Use this to set size
  vimage.resize(height,width);

  set_size(vimage.height(),vimage.width());
    
  tracker_cleanup(this);
} 


Pipe_Raw::Pipe_Raw(Pipeline *pl_in) : Pipe()
{
  pl = pl_in;
  height = pl_in->get_height();
  width = pl_in->get_width();
  output.resize(height,width);
}

void Pipe_Raw::update(void)
{
  input_frame_count++;

  if (pl->get_v_frame_count() < input_frame_count) {
    pl->grab_video();
  }
 
  output = pl->get_input();

  //  if (*output.data() > 255)
  //    output = grayscale(output);

  frame_count++;
}


Pipe_Plus::Pipe_Plus(Pipe *p1_in, Pipe *p2_in)
{
  add_pred(p1_in);
  add_pred(p2_in);
  height = p1_in->get_height();
  width = p1_in->get_width();
  output.resize(height,width);
}

void Pipe_Plus::update(void)
{
  input_frame_count++;

  if (preds[0]->get_frame_count() < input_frame_count) {
    preds[0]->main_update();
  }
  if (preds[1]->get_frame_count() < input_frame_count) {
    preds[1]->main_update();
  }

  output = ((const XVImage<int> &)preds[0]->get_output()) + ((const XVImage<int> &)preds[1]->get_output());

  frame_count++;
}


Pipe_Minus::Pipe_Minus(Pipe *p1_in, Pipe *p2_in)
{
  add_pred(p1_in);
  add_pred(p2_in);
  height = p1_in->get_height();
  width = p1_in->get_width();
  output.resize(height,width);
}

void Pipe_Minus::update(void)
{
  input_frame_count++;

  if (preds[0]->get_frame_count() < input_frame_count) {
    preds[0]->main_update();
  }
  if (preds[1]->get_frame_count() < input_frame_count) {
    preds[1]->main_update();
  }

  output = (const XVImage<int> &)preds[0]->get_output() - (const XVImage<int>&)preds[1]->get_output();

  frame_count++;
}

Pipe_Mult::Pipe_Mult(Pipe *p1_in, Pipe *p2_in)
{
  add_pred(p1_in);
  add_pred(p2_in);
  height = p1_in->get_height();
  width = p1_in->get_width();
  output.resize(height,width);
}

void Pipe_Mult::update(void)
{
  input_frame_count++;

  if (preds[0]->get_frame_count() < input_frame_count) {
    preds[0]->main_update();
  }
  if (preds[1]->get_frame_count() < input_frame_count) {
    preds[1]->main_update();
  }

  output = ((const XVImage<int> &)preds[0]->get_output()) * ((const XVImage<int> &)preds[1]->get_output());

  frame_count++;
}

Pipe_And::Pipe_And(Pipe *p1_in, Pipe *p2_in)
{
  add_pred(p1_in);
  add_pred(p2_in);
  height = p1_in->get_height();
  width = p1_in->get_width();
  output.resize(height,width);
}

void Pipe_And::update(void)
{
  input_frame_count++;

  if (preds[0]->get_frame_count() < input_frame_count) {
    preds[0]->main_update();
  }
  if (preds[1]->get_frame_count() < input_frame_count) {
    preds[1]->main_update();
  }

  output = ((const XVImage<int> &)preds[0]->get_output()) * ((const XVImage<int> &)preds[1]->get_output());

  frame_count++;
}



Pipe_Sq::Pipe_Sq(Pipe *p_in)
{
  add_pred(p_in);
  height = p_in->get_height();
  width = p_in->get_width();
  output.resize(height,width);
}

void Pipe_Sq::update(void)
{
  input_frame_count++;

  if (preds[0]->get_frame_count() < input_frame_count) {
    preds[0]->main_update();
  }

  output = square((const XVImage<int> &)preds[0]->get_output());

  frame_count++;
}


Pipe_Sqrt::Pipe_Sqrt(Pipe *p_in)
{
  add_pred(p_in);
  height = p_in->get_height();
  width = p_in->get_width();
  output.resize(height,width);
}

void Pipe_Sqrt::update(void)
{
  input_frame_count++;
    
  if (preds[0]->get_frame_count() < input_frame_count) {
    preds[0]->main_update();
  }

  output = sqroot(preds[0]->get_output());

  frame_count++;
}


Pipe_Subimage::Pipe_Subimage(int id_in, Pipe *p_in, int lx_in, int ly_in, 
			     int ux_in, int uy_in)
{
  id = id_in;
  add_pred(p_in);
  lx = lx_in;
  ly = ly_in;
  ux = ux_in;
  uy = uy_in;
  height = uy-ly+1;
  width = ux-lx+1;
  output.resize(height,width);
}
 
void Pipe_Subimage::update(void)
{
  input_frame_count++;

  if (preds[0]->get_frame_count() < input_frame_count) {
    preds[0]->main_update();
  }

  output = subimage(preds[0]->get_output(),lx,ly,ux,uy);

  frame_count++;
}

void Pipe_Subimage::modify_params(Pipe_Params &pp)
{
  lx = pp.lx;
  ly = pp.ly;
  ux = pp.ux;
  uy = pp.uy;

  height = uy-ly+1;
  width = ux-lx+1;
  output.resize(height,width);
}


Pipe_Delay::Pipe_Delay(Pipe *p_in)
{
  add_pred(p_in);
  height = p_in->get_height();
  width = p_in->get_width();
  output.resize(height,width);
  delayed.resize(height,width);

  int *delayed_data = delayed.data();
  unsigned int output_size = width * height;

  // init with black, empty image
  for (int i=0; i<output_size; i++) {
    *delayed_data++ = 0;
  }
}

void Pipe_Delay::update(void)
{
  input_frame_count++;

  if (preds[0]->get_frame_count() < input_frame_count) {
    preds[0]->main_update();
  }

  output = delayed;
  delayed = preds[0]->get_output();

  frame_count++;
}


Pipe_Threshold::Pipe_Threshold(int id_in, Pipe *p_in, int thresh_in, int val_in) : Pipe()
{
  id = id_in;
  add_pred(p_in);
  height = p_in->get_height();
  width = p_in->get_width();
  output.resize(height,width);
  val = val_in;
  thresh = thresh_in;
}

void Pipe_Threshold::update(void)
{
  input_frame_count++;
 
  if (preds[0]->get_frame_count() < input_frame_count) {
    preds[0]->main_update();
  }

  output = threshold(preds[0]->get_output(),thresh, val);

  frame_count++;
}


Pipe_Reduce::Pipe_Reduce(int id_in, Pipe *p_in, int factor_in) : Pipe()
{
  id = id_in;
  add_pred(p_in);
  height = p_in->get_height()/factor_in;
  width = p_in->get_width()/factor_in;
  output.resize(height,width);
  factor = factor_in;
}

void Pipe_Reduce::update(void)
{
  input_frame_count++;
 
  if (preds[0]->get_frame_count() < input_frame_count) {
    preds[0]->main_update();
  }

  output = preds[0]->get_output().reduce_resolution(factor);

  frame_count++;
}


Pipe_Project::Pipe_Project(int id_in, Pipe *p_in, float rc_in, float gc_in, float bc_in,
			  int norm) : Pipe()
{
  id = id_in;
  add_pred(p_in);
  height = p_in->get_height();
  width = p_in->get_width();
  output.resize(height,width);
  normalize = norm;
  rc = rc_in;
  gc = gc_in;
  bc = bc_in;

}

void Pipe_Project::update(void)
{
  input_frame_count++;
 
  if (preds[0]->get_frame_count() < input_frame_count) {
    preds[0]->main_update();
  }

  output = project(preds[0]->get_output(),rc,gc,bc,normalize);

  frame_count++;
}


Pipe_Abs::Pipe_Abs(Pipe *p_in) : Pipe()
{
  add_pred(p_in);
  height = p_in->get_height();
  width = p_in->get_width();
  output.resize(height,width);
}

void Pipe_Abs::update(void)
{
  input_frame_count++;

  if (preds[0]->get_frame_count() < input_frame_count) {
    preds[0]->main_update();
  }

  output = (preds[0]->get_output()).abs();

  frame_count++;
}


Pipe_Gray::Pipe_Gray(Pipe *p_in) : Pipe()
{
  add_pred(p_in);
  height = p_in->get_height();
  width = p_in->get_width();
  output.resize(height,width);
}

void Pipe_Gray::update(void)
{
  input_frame_count++;

  if (preds[0]->get_frame_count() < input_frame_count) {
    preds[0]->main_update();
  }

  output = grayscale(preds[0]->get_output());

  frame_count++;
}

Pipe_Dx::Pipe_Dx(Pipe *p_in) : Pipe()
{
  add_pred(p_in);
  height = p_in->get_height();
  width = p_in->get_width();
  output.resize(height,width);
}

void Pipe_Dx::update(void)
{
  input_frame_count++;

  if (preds[0]->get_frame_count() < input_frame_count) {
    preds[0]->main_update();
  }

  output = dx(preds[0]->get_output());

  frame_count++;
}



Pipe_Dy::Pipe_Dy(Pipe *p_in) : Pipe()
{
  add_pred(p_in);
  height = p_in->get_height();
  width = p_in->get_width();
  output.resize(height,width);
}

void Pipe_Dy::update(void)
{
  input_frame_count++;
    
  if (preds[0]->get_frame_count() < input_frame_count) {
    preds[0]->main_update();
  }

  output = dy(preds[0]->get_output());

  frame_count++;
}


Pipe_Sobel::Pipe_Sobel(Pipe *p_in)
{
  add_pred(p_in);
  height = p_in->get_height();
  width = p_in->get_width();
  output.resize(height,width);
}

void Pipe_Sobel::update(void)
{
  input_frame_count++;

  if (preds[0]->get_frame_count() < input_frame_count) {
    preds[0]->main_update();
  }

  output = sobel(preds[0]->get_output());

  frame_count++;
}


Pipe_Gaussian::Pipe_Gaussian(int id_in, Pipe *p_in, 
			     int width_in, double sigmax_in,
			     int height_in, double sigmay_in)
{

  if (height_in < 0) height_in = width_in;
  if (sigmay_in < 0) sigmay_in = sigmax_in;

  id = id_in;
  add_pred(p_in);
  height = p_in->get_height();
  width = p_in->get_width();
  sigmax = sigmax_in;
  sigmay = sigmay_in;
  output.resize(height,width);

  make_mask(width_in, height_in);
}

void Pipe_Gaussian::modify_params(Pipe_Params &pp)
{
  sigmax = pp.sigma_x;
  sigmay = pp.sigma_y;

  make_mask((int)pp.maskx,(int)pp.masky);
}

// Make the separable masks

void Pipe_Gaussian::make_mask(int mask_width, int mask_height)
{
  int i;
  double dx_sq, sigma_sqx, sigma_sqy;
  double mask_midx  = (mask_width-1)/2;
  double mask_midy  = (mask_height-1)/2;
  
  sigma_sqx = sigmax * sigmax;
  sigma_sqy = sigmay * sigmay;
  



  maskx.resize(1,mask_width);
  for (i=0; i<mask_width; i++) {
    dx_sq = (mask_midx-i) * (mask_midx-i);
    maskx[0][i] = (int) rint(100.0 * exp(-(dx_sq+dx_sq)/(2.0*sigma_sqx)));
  }

  masky.resize(mask_height,1);
  for (i=0; i<mask_height; i++) {
    dx_sq = (mask_midy-i) * (mask_midy-i);
    masky[i][0] = (int) rint(100.0 * exp(-(dx_sq+dx_sq)/(2.0*sigma_sqy)));
  }

}

void Pipe_Gaussian::update(void)
{
  input_frame_count++;

  if (preds[0]->get_frame_count() < input_frame_count) {
    preds[0]->main_update();
  }

  // Check for 1-D convolutions; otherwise a separable 2-D

  if (masky.height() < 2)
    (preds[0]->get_output()).convolve(output,maskx);
  else if (maskx.width() < 2)
    (preds[0]->get_output()).convolve(output,masky);
  else
    (preds[0]->get_output()).convolve(output,maskx,masky);

  // convolve with gaussian mask

  /*  int *output_data = output.data();
  
  int *arg_data = preds[0]->get_output().data();
  
  unsigned int output_size = output.width() * output.height();
  
  // normalize convolution values
  for (int i=0; i<output_size; i++) {
    // normalize only if value was convolved
    if (*output_data != *arg_data) {
      *output_data = (int) rint(((double)*output_data)/mask_weight);
    }
    output_data++;
    arg_data++;
  }*/

  frame_count++;
}

Pipe_Convolve::Pipe_Convolve(int id_in, Pipe *p_in, const Image &mask_in)
{
  id = id_in;
  add_pred(p_in);
  height = p_in->get_height();
  width = p_in->get_width();
  mask_width = mask_in.width();
  mask_height = mask_in.height();
  mask_midx = (mask_width-1)/2;
  mask_midy = (mask_height-1)/2;

  output.resize(height,width);
  mask = mask_in;
}


void Pipe_Convolve::update(void)
{
  input_frame_count++;

  if (preds[0]->get_frame_count() < input_frame_count) {
    preds[0]->main_update();
  }

  // convolve with guassian mask
  output = convolve(preds[0]->get_output(),mask);
  /*  int *output_data = output.data();
  
  int *arg_data = preds[0]->get_output().data();
  
  unsigned int output_size = output.width() * output.height();
  
  // normalize convolution values
  for (int i=0; i<output_size; i++) {
    // normalize only if value was convolved
    if (*output_data != *arg_data) {
      *output_data = (int) rint(((double)*output_data)/mask_weight);
    }
    output_data++;
    arg_data++;
  }*/

  frame_count++;
}

Pipe_Erode::Pipe_Erode(int id_in, Pipe *p_in, const Image &mask_in)
{
  id = id_in;
  add_pred(p_in);
  height = p_in->get_height();
  width = p_in->get_width();
  mask_width = mask_in.width();
  mask_height = mask_in.height();
  mask_midx = (mask_width-1)/2;
  mask_midy = (mask_height-1)/2;

  output.resize(height,width);
  mask = mask_in;
}


void Pipe_Erode::update(void)
{
  input_frame_count++;

  if (preds[0]->get_frame_count() < input_frame_count) {
    preds[0]->main_update();
  }

  // erode using mask

  output = erode(preds[0]->get_output(),mask);
  
  /*  int *output_data = output.data();
  
  int *arg_data = preds[0]->get_output().data();
  
  unsigned int output_size = output.width() * output.height();
  
  // normalize convolution values
  for (int i=0; i<output_size; i++) {
    // normalize only if value was convolved
    if (*output_data != *arg_data) {
      *output_data = (int) rint(((double)*output_data)/mask_weight);
    }
    output_data++;
    arg_data++;
  }*/

  frame_count++;
}
/*
Pipe_Variance::Pipe_Variance(int id_in, Pipe *p_in, int height_in, 
			     int  width_in)
{
  id = id_in;
  add_pred(p_in);
  height = p_in->get_height();
  width = p_in->get_width();
  mask_height = height_in;
  mask_width = width_in;

  output.resize(height,width);
}

void Pipe_Variance::update(void)
{
  input_frame_count++;

  if (preds[0]->get_frame_count() < input_frame_count) {
    preds[0]->main_update();
  }
  // compute variance 

  output = variance(preds[0]->get_output(),mask_height,mask_width);

  frame_count++;
}

void Pipe_Variance::modify_params(Pipe_Params &pp)
{
  mask_width = pp.width;
  mask_height = pp.height;
}

*/
Pipe_Laplacian_Gaussian::Pipe_Laplacian_Gaussian(int id_in, Pipe *p_in, 
						 int width_in, double sigma_in)
{
  mask_width = mask_height = width_in;
  mask_midx = mask_midy = (mask_width-1)/2;
  add_pred(p_in);;
  sigma = sigma_in;
  height = p_in->get_height();
  width = p_in->get_width();

  output.resize(height,width);
  mask.resize(mask_width,mask_width);
  make_mask();
}

void Pipe_Laplacian_Gaussian::make_mask(void)
{
  int i, j, mask_val;
  double dy_sq, r, r_sq, sigma_sq, sigma_4th;  
  int *mask_data;

  mask_data = mask.data();

  sigma_sq = sigma * sigma;
  sigma_4th = sigma_sq * sigma_sq;
  mask_weight = 0;
  mask_val = 0;

  for (i=0; i<mask_height; i++) {
    dy_sq = (mask_midx-i) * (mask_midy-i);
    for (j=0; j<mask_width; j++) {
      r_sq = (mask_midx-j)*(mask_midx-j) + dy_sq;
      r = sqrt(r_sq);
      mask_val = (int) rint(100.0 * ((r_sq-(2.0*sigma_sq))/sigma_4th)
			    * exp(-r/(2.0*sigma_sq)));
      mask_data[i*mask_width+j] = mask_val;
      mask_weight += mask_val;
    }
  }
}

void Pipe_Laplacian_Gaussian::update(void)
{
  input_frame_count++;

  if (preds[0]->get_frame_count() < input_frame_count) {
    preds[0]->main_update();
  }

  // convolve with laplacian of guassian mask
  output = convolve(preds[0]->get_output(),mask);

  int *output_data = output.data();

  int *arg_data = preds[0]->get_output().data();

  unsigned int output_size = output.width() * output.height();

  // normalize convolution values
  for (int i=0; i<output_size; i++) {
    // normalize only if value was convolved
    if (*output_data != *arg_data) {
      *output_data = (int) rint(((double)*output_data)/mask_weight);
    }
    output_data++;
    arg_data++;
  }
  frame_count++;
}

void Pipe_Laplacian_Gaussian::modify_params(Pipe_Params &pp)
{
  mask_width = mask_height = pp.width;
  mask_midx = mask_midy = (mask_width-1)/2;
  sigma = pp.sigma_x;

  mask.resize(mask_width,mask_width);
  make_mask();
}


Pipe_Convtrack::Pipe_Convtrack(int id_in, Pipe *p_in, int lx_in, int ly_in, 
			       int ux_in, int uy_in, int trans_in)
{
  id = id_in;
  lx = lx_in;
  ly = ly_in;
  ux = ux_in;
  uy = uy_in;
  trans = trans_in;
  add_pred(p_in);

  lx_best = lx;
  ly_best = ly;
  ux_best = ux;
  uy_best = uy;

  mask_width = ux-lx+1;
  mask_height = uy-ly+1;

  height = mask_height;
  width = mask_width;

  output.resize(height,width);
  make_mask();
}

void Pipe_Convtrack::make_mask(void)
{
  mask_size = mask_width * mask_height;

  // mask is the region to track
  output = subimage(preds[0]->get_output(),lx_best,ly_best,ux_best,uy_best);

  int *mask_data = output.data();

  mask_val = 0;

  // mask_val is best convolved value for location of mask
  for (int i=0; i<mask_size; i++) {
    mask_val += (*mask_data) * (*mask_data);
    mask_data++;
  }
}

void Pipe_Convtrack::update(void)
{
  input_frame_count++;

  if (preds[0]->get_frame_count() < input_frame_count) {
    preds[0]->main_update();
  }

  Image input = preds[0]->get_output();
  int x_start, x_end, y_start, y_end, i, j, k, k_end, l, l_end;
  int *mask_data, *input_data;
  unsigned int input_width;
  double val, min = DBL_MAX;

  x_start = lx_best-trans;
  y_start = ly_best-trans;
  x_end = lx_best+trans;
  y_end = ly_best+trans;

  if (x_start < 0)
    x_start = 0;
  if ((x_end+mask_width-1) >= input.width())
    x_end = input.width() - mask_width + 1;
  if (y_start < 0)
    y_start = 0;
  if ((y_end+mask_height-1) >= input.height())
    y_end = input.height() - mask_height + 1;
  
  input_data = input.data();
  input_width = input.width();

  // look for best match with mask by taking maximum of convolution
  for (i=y_start; i<y_end; i++) {
    for (j=x_start; j<x_end; j++) {
      mask_data = output.data();
      
      val = 0;

      k_end = i+mask_height+1;
      l_end = j+mask_width+1;

      // convolve
      for (k=i; k<k_end; k++) {
	for (l=j; l<l_end; l++) {
	  val += *mask_data++ * (input_data[k*input_width+l]);
	}
      }

      // found better match
      if (fabs(mask_val-val) < min) {
	min = fabs(mask_val-val);
	lx_best = j;
	ly_best = i;
      }
    }
  }
  
  ux_best = lx_best + mask_width - 1;
  uy_best = ly_best + mask_height - 1;
  
  if (lx_best < 0)
    lx_best = 0;
  if ((ux_best+mask_width-1) >= input.width())
    ux_best = input.width() - mask_width;
  if (ly_best < 0)
    ly_best = 0;
  if ((uy_best+mask_height-1) >= input.height())
    uy_best = input.height() - mask_height;

  // update mask to match found
  output = subimage(input,lx_best,ly_best,ux_best,uy_best);
  
  mask_data = output.data();

  mask_val = 0;

  // update mask_val
  for (i=0; i<mask_size; i++) {
    mask_val += (*mask_data) * (*mask_data);
    mask_data++;
  }
  frame_count++;
}

void Pipe_Convtrack::modify_params(Pipe_Params &pp)
{
  lx = pp.lx;
  ly = pp.ly;
  ux = pp.ux;
  uy = pp.uy;
  trans = pp.translation;

  lx_best = lx;
  ly_best = ly;
  ux_best = ux;
  uy_best = uy;
  mask_width = ux-lx+1;
  mask_height = uy-ly+1;
  height = mask_height;
  width = mask_width;

  output.resize(height,width);
  make_mask();
}



