/*
    Copyright (C) 1993 Gregory D. Hager (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//-----------------------------------------------------------------------------
//
//  FTypes.hh
//
//  Simple static and dynamic typing system for tracking objects.
//  Essentially, as an object inherits more types, it keeps a map
//  of the object sufficient to perform dynamic casting properly.
//
//  8/1/94
//  Gregory D. Hager
//
//-----------------------------------------------------------------------------

#ifndef _FTYPES_H
#define _FTYPES_H

#include "BasicFeature.hh"
#include "BaseType.hh"
#include "Geometry.hh"

// There are two types of information about class that is useful
// in this typing scheme.   One type just says that the feature
// "offers" some information.  The other says that those parameters
// can be set.  The current parameter groups are
//
//   Position
//   Scale
//   Orientation
//   Length/Width of acquisition window
// 
// This is in contrast to the attributes of the feature itself
// which might also have some of these properties.

class Positionable : virtual public BaseType {
  
  static const char *_typeid; //typeid is a  reserved word

protected:

  Positionable(BasicFeature *f)  {
    set_handle(f);
    f->add_type(_typeid,this);
  }

public:

  // State info getters

  virtual float x() const = 0;
  virtual float y() const = 0;

  // State info setters

  virtual float set_x(float) PANIC("Positionable");
  virtual float set_y(float) PANIC("Positionable");

  static const char *info() {return _typeid;};

};

class Orientable : virtual public BaseType {
  
  static const char *_typeid;

protected:

  Orientable(BasicFeature *f) {
    set_handle(f);
    f->add_type(_typeid,this);
  }

public:

  // State info getters

  virtual float orientation() const = 0;

  // State info setters

  virtual float set_orientation(float) PANIC("Orientable");

  static const char *info() {return _typeid;};

};

class PointFeature : public Positionable
{

  static const char *_typeid;

protected:

  PointFeature(BasicFeature *f) : Positionable(f) {
    f->add_type(_typeid,this);
  }

public:

  // State info getters

  virtual point2D coordinates() const {
    point2D temp;
    temp.x = x();
    temp.y = y();
    return temp;
  }

  virtual point3D hcoordinates() const
    {
    point3D temp;
    temp.x = x();
    temp.y = y();
    temp.z = 1.0;
    return temp;
  }

};

class LineFeature :  public Orientable  
{
  
  static const char *_typeid;

protected:

  LineFeature(BasicFeature *f) : Orientable(f) {
    f->add_type(_typeid,this);
  }

public:

  virtual line2D coordinates() const {
    lineAD temp = coordinatesAD();
    line2D temp1;
    temp1.x = -sin(temp.o);
    temp1.y = cos(temp.o);
    temp1.z = -temp.d;
    
    return temp1;
  }

  virtual lineAD coordinatesAD() const {
    line2D temp = coordinates();
    lineAD temp1;

    temp1.o = atan2(temp.y,-temp.x);
    temp1.d = -temp.z/sqrt(temp.x*temp.x+temp.y*temp.y);

    return temp1;
  }


  static const char *info() {return _typeid;};

};


class LineSegment :  public LineFeature, public Positionable  {
  
  static const char *_typeid;

protected:

  LineSegment(BasicFeature *f) : LineFeature(f), Positionable(f) {
    f->add_type(_typeid,this);
  }

public:

  virtual line2D coordinates() const {
    line2D temp1;
    temp1.x = -sin(orientation());
    temp1.y = cos(orientation());
    temp1.z = -(x() * temp1.x + y()*temp1.y);
    
    return temp1;
  }


  virtual point2D endpt1() const = 0;
  virtual point2D endpt2() const = 0;
  virtual double  length() const {return distance(endpt1(),endpt2());};

  static const char *info() {return _typeid;};

};


#endif

