/*                                                                -*-c++-*- 
    Copyright (C) 1993 Gregory D. Hager and Sidd Puri (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//-----------------------------------------------------------------------------
//
//  BasicFeature.hh
//
//  Declaration of BasicFeature class.  This class is a virtual base class
//  for all trackable BasicFeatures.  It provides a standard state variable
//  interface and virtual functions for initialization to tracking functions.
//
//  6 - 7 - 93
//  Sidd Puri
//
//  7 - 31 - 94
//  Gregory D. Hager
//
//  09/04/98 Greg Hager added a little infrastructure to allow
//  for synchronization with framegrabber
//  The trick is that "state_update" is now the function with causes
//  the next frame to get called in, whereas update doesn't.
//  Track has been modified to  use state update.  If all this
//  works, then it'll be possible to get everybody working together
//  on one frame.
//-----------------------------------------------------------------------------

#ifndef BasicFeature_h
#define BasicFeature_h
#include <stdio.h>
#include <string.h>
#include "Lists.hh"
#include "CWindow.hh"
#include "typeinfo.hh"
#include "Tools.hh"
#include "rate.hh"

#if (XV_OS == XV_IRIX)
#include <sys/time.h>
#else
#include <time.h>
#endif

// For backwards compatibility with previous releases

#define Thing BasicFeature

//-----------------------------------------------------------------------------
//  BasicFeature class declaration
//-----------------------------------------------------------------------------

class BasicFeature  {
  friend class CompFeature;
  friend class FExtension;
  friend class Contour;

protected:

  // State related information

  int ncoords;
  int ntimesteps;
  int current_state;
  float *state;            // active state vector
  float **state_arr;       // set of state vectors

  // Other information for this feature

  Color color;
  Video *v;                  // The video source this BasicFeature uses

  // Typing information 

  const char *name;
  List<_typeinfo *> ftypes;   // This shouldn't be a pointer
  FeatureType featuretype;
  int top_level_feature;

  // Timing and sequencing information
  TimingInfo tinfo;
  int framenr;

  // Some other bookeeping

  void set_nontop()  {top_level_feature = 0;};

  Color map_color(Color c_in);

  void init_states(int coords_in,int timesteps_in = 2,int start=0);
  void resize_states(int timesteps_in);
  void delete_states();

protected:

  int  copy_to_update;        // Make 1 only if copying should be used
                              // to maintain graphics display consistency.
  
  void forward(int n=1) {current_state = (current_state + n)%ntimesteps;
			 state = state_arr[current_state];
		       }
  virtual void replicate_forward() 
    {copy(state_arr[(current_state + 1)%ntimesteps],
				 state_arr[current_state],ncoords);
			    forward();
			  }

  void back(int n = 1) {n %= ntimesteps;
			current_state = current_state + 
			  ((n <= current_state)? -n : ntimesteps-n);
			state = state_arr[current_state];
		      }

  void erase_last() {back();clear();forward();};

public:

  // Creation and Deletion

  BasicFeature (int ncoords_in, Video *vin, 
	 const char *namein = "BasicFeature", Color color_in = default_color);

  BasicFeature (const BasicFeature &t);

  virtual ~BasicFeature()          {delete_states();ftypes.clear();};

  // Something for types

  void add_type(const char *t,void *ptr);
  void *has_type(const char*);
  int modifiable() const {return ((featuretype==Basic) ||
			    (featuretype==Composite));};

  // Two basic initialization routines.  Either supply a state to start 
  // with,  use an external CWindow to interactively initialize.

  virtual int state_init (float *state_in) 
    {copy(state,state_in,ncoords); return 1;};
  virtual int interactive_init (CWindow &) {return 0;};
//  virtual int interactive_init (CWindow &) = 0;

  //  This duplication function must be redefined in any class that
  //  adds variables to the class.  It ensures that the class can
  //  be duplicated independent of type.

  virtual BasicFeature *dup() = 0;

  // For accessing the video source
  
  Video *source() {return v;};

  // use only if you know what you are doing.....
  void setSource(Video* newv) {v = newv;};

  // For changing color

  virtual Color set_color(Color x) {Color temp = color;color = x;return temp;};

  // For accessing state information generically

  virtual float operator[] (int n) { return state[n]; }

  // And special methods for accessing position values,
  // so these  become independent on the actual
  // storage in the state vector.

  // virtual float x (void) { cerr << "This feature does not have an x() method\n";
  //                          return 0; 
  //                        }

  // virtual float y (void) { cerr << "This feature does not have an x() method\n";
  //                          return 0; 
  //                        }

  // Simple run-time typing

  int is_top_level() const {return top_level_feature;};

  // These are the functions used for the tracking cycle.

  virtual int show (Color color_in= default_color) = 0;
  //  virtual int clear (){back();show(erase_color);forward();return 1;};
  virtual int clear (){show(erase_color);return 1;};
  virtual int update ()  = 0;
  int state_update () {

      // Hack to synchronize with source if that
      // is desired

      if (source() != NULL) {
	source()->check_sequence(framenr);
      }

      // we do our thing

      framenr++;
      return update();
  }

  void use_copy_to_update() {copy_to_update = 1;};

  virtual void init_timing(int interval, int do_asynchronous_reporting = 0) 
  {
    tinfo.setprinting(do_asynchronous_reporting);
    tinfo.setinterval(interval);
    if (interval < 1)
      {
	cout << "unreasonable timing interval ("
	     << interval << ") in call to initialize timing" << endl
	     << "for top_level feature" << endl;
      };
  };
  
  float get_rate()
  {
    if (top_level_feature) 
      return tinfo.rate();
    else
      return -1;
  };
      
  // The default tracking cycle.  
  virtual int track() 
    { 
      int res;

      if (copy_to_update) {
	BasicFeature *temp = dup(); // why is this here?
	res = state_update();
	temp->clear(); // why is this here?
	if (res)
	  show (color);
	delete temp; // why is this here?
      }
      else {
	replicate_forward();
	res = state_update();
	erase_last();
	if (res)
	  show(color);
      }

      if (is_top_level()) 
	{
	  tinfo.tick();
	};           

      return res;
      
    }
  
  // This is a base function needed for CompFeature to work.  It
  // can be redefined if a basic feature has a state propagation rule.
  
  virtual int propagate () {return 1;}

  // Define the output operator

  friend ostream&  operator << (ostream& s, BasicFeature& x);

  // Make sure equality is not used for assignment

  BasicFeature &operator = (BasicFeature &) {
    panic("Error: assignment not defined for Feature networks");
    return *this;
  };

  // This is only here for typing
  
  static const char *info() {return NULL;};

};


//-----------------------------------------------------------------------------
#endif

