/*
    Copyright (C) 1993 Gregory D. Hager
    Yale Computer Science Robotics and Vision Laboratory

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//-----------------------------------------------------------------------------
//
//  BasicFeature.cc
//
//  Implementation of BasicFeature class.  This class is a virtual base class
//  for all trackable BasicFeatures.  It provides a standard state variable
//  interface and virtual functions for initialization to tracking functions.
//
//  7 - 31 - 94
//  Gregory D. Hager
//
//-----------------------------------------------------------------------------

#include "BasicFeature.hh"
#include <string.h>


class Video;

void BasicFeature::init_states(int coords_in,int timesteps_in,int start) {
  ncoords = coords_in;
  ntimesteps = timesteps_in;

  state_arr = new float*[timesteps_in];
  state_arr[0] = new float[timesteps_in*ncoords];
  
  int i;
  for (i=1;i<ntimesteps;i++)
    state_arr[i] = state_arr[0]+i*ncoords;

  state = state_arr[start];
  current_state = start;
}
    
void BasicFeature::resize_states(int timesteps_in) {
  if (timesteps_in < ntimesteps)
    panic("Cannot reduce temporal state information");

  float *temp = state_arr[0];
  int tnc = ncoords*ntimesteps;
  delete state_arr;

  init_states(ncoords,timesteps_in,current_state);
  copy(state_arr[0],temp,tnc);
  
  delete temp;
}

void BasicFeature::delete_states() {
  delete state_arr[0];
  delete state_arr;
}

BasicFeature::BasicFeature (int ncoords_in, Video *vin, 
			    const char *namein, Color color_in) 
: tinfo(-1)

{
  
  init_states(ncoords_in);
  copy_to_update = 0;
  color = color_in;
  v = vin;
  name = namein;
  top_level_feature = 1;
  framenr = 0;
  featuretype = Basic;
}

BasicFeature::BasicFeature (const BasicFeature &t) :
tinfo(t.tinfo)
{
  copy_to_update       = t.copy_to_update;
  color                = t.color;
  v                    = t.v;
  name                 = t.name;
  featuretype          = t.featuretype;
  top_level_feature    = t.top_level_feature;
  name                 = t.name;
  framenr              = t.framenr;
  
  // Make a new state array and copy that info across

  init_states(t.ncoords,t.ntimesteps,t.current_state);
  copy(state_arr[0],t.state_arr[0],ncoords*ntimesteps);
}

Color BasicFeature::map_color(Color c_in) {

  if (c_in == no_color)
    return color;
  else
    return c_in;
}
  

void *
BasicFeature::has_type(const char *c)
{
  if (name == c)  return (void *)this;
  else
    for(ftypes.front();!~ftypes;++(ftypes))
      if (*(*ftypes) == c)
	return (void *)(*ftypes)->cast_from();

  return NULL;
}

void BasicFeature::add_type(const char *t,void *ptr) {
    ftypes += new _typeinfo(t,ptr); 
}

ostream& operator << (ostream& s, BasicFeature &x) {
    int i;
    s << x.name << "   " << x.ncoords << endl;
    for (i=0;i<x.ncoords;i++)
      s << x.state[i] << "  ";
    if (x.is_top_level())
      s << "(Tracking at " << x.tinfo.rate() << " Hz)";
    s << endl;
    return s;
}
