/*                                                                -*-c++-*- 
    Copyright (C) 1993 Gregory D. Hager (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//-----------------------------------------------------------------------------
//
//  BASETYPE.hh
//
//  Simple static and dynamic typing system for tracking objects.
//  Essentially, as an object inherits more types, it keeps a map
//  of the object sufficient to perform dynamic casting properly.
//
//  8/1/94
//  Gregory D. Hager
//
//-----------------------------------------------------------------------------

#ifndef _BASETYPE_H
#define _BASETYPE_H

#include "BasicFeature.hh"
#include "typeinfo.hh"

/*--------------------- Typing Structure ----------------------------------

   The typing structure used here is a bit complex, unfortunately.
   Basically, it works as follows.  Each object is a BasicFeature, a
   FeatureGroup, or a CompositeFeature.  A FeatureGroup can consist of
   any collection of features.  However, CompositeFeatures will not
   accept FeatureGroups as subsidiaries.  Thus, if the convention is
   adhered to that CompositeFeatures manage the states of their
   children, then a CompositeTree can propagate state up and down the
   tree.

   In addition, there is type casting based on the "functionality" of
   features.  This is done by essentially splitting feature parameters
   into groups with particular access functions.  The "setting"
   functions are by default disabled.  Furthermore, all FeatureGroup
   children are automatically cast as "const" instances.  Hence, the
   compiler disallows use of the setting functions unless an explicit
   cast to a nonconst type is done.  Even then, the system will abort
   with a runtime error if an attempt is made to push constraints
   through a FeatureGroup.

--------------------------------------------------------------------------*/


// The first define looks a little strange in order to handle both
// BasicFeatures and handles (types) to BasicFeatures.  The
// (&(*(obj))) makes sure every object becomes a BasicFeature.  The
// cast to BasicFeature * is to get rid of the const typing so that
// the has_type function can operate (it cannot be declared as a const
// function because of the way lists operate).

#define can_cast(type,obj) \
        (((BasicFeature *)(&(*(obj))))->has_type(type::info()))

#define ptr_cast(type,obj) (((obj) && ((obj)->modifiable())) ? \
			    (type *)can_cast(type,(obj)) : (type *)NULL)

#define const_ptr_cast(type,obj) ((obj) ? \
			    (const type *)can_cast(type,(obj)) : (const type *)NULL)

#define child_as(type,n) (ptr_cast(type,child(n)))

#define const_child_as(type,n) (ptr_cast(type,child(n)))

// By default, the setting functions will cause a runtime 
// panic if not overloaded in the derived class

void vfnpanic(const char *where);
#define PANIC(n) {vfnpanic(n);return 0;}

class BaseType 
{
friend class BasicFeature;
  
  BasicFeature *handle;

protected:
  void  set_handle(BasicFeature *h) {handle = h;};

public:

  BasicFeature *get_handle() {return handle;};


  // BaseType is often used as a virtual base.   In order
  // to avoid constructor calls where the user shouldn't
  // need to know about BaseType, an empty constructor is
  // included.  The convention is that any class derived 
  // from BaseType performs a set_handle call to actually
  // initialize that pointer.

  BaseType () {};
  BaseType(BasicFeature *f) : handle(f) {}
  BaseType (BaseType &b) : handle(b.get_handle()) {};


  operator BasicFeature*()    {return get_handle();}
  operator BasicFeature&()    {return *get_handle();}
  BasicFeature *operator& ()    {return get_handle();}

  BasicFeature *operator->() {return get_handle();}
  BasicFeature &operator*()  {return *get_handle();}

  // Type inquiry as to whether you can set the paramters
  // of the object
  
  int modifiable() {return handle->modifiable();}

};

#endif
