/* 
    Copyright (C) 1993 Gregory D. Hager and Sidd Puri (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//-----------------------------------------------------------------------------
//
//  Pattern.hh
//
//  Declaration of Pattern class.  An object of this class is a description of
//  a pattern (e.g. an edge) which can be located in an image.  Pattern itself
//  is a rather minimal base class, from which specific patterns are derived.
//
//  7 - 21 - 93
//  Sidd Puri
//
//-----------------------------------------------------------------------------

#ifndef Pattern_h
#define Pattern_h

#include "Tracker.hh"
#include "Tools.hh"

//-----------------------------------------------------------------------------
//  Default parameters
//-----------------------------------------------------------------------------

int   const default_mwidth = 8;
float const default_alpha  = deg_rad (15);
int   const default_edge_threshold = 5;
int   const default_width = 40;
float const default_sensitivity = 1.0;

//-----------------------------------------------------------------------------
//  Structures for return values
//-----------------------------------------------------------------------------

struct offset {				     // returned by find
  float x, angle, val;
};

struct peak {				     // returned by interpolation
					     // functions
  float x, val;
};

struct SE_out {                    // Shortedge output struct.
  int numpks;                      // number of peaks. 
  int *pk;                         // array of pk locations.
  int *pkstrength;                 // array of pk strengths.
  int *regionmode;                 // array of intensity mode values
                                   //  between peaks.
  int *imval;                      // array of actual intensity values.
};

//-----------------------------------------------------------------------------
//  Pattern base class declaration
//-----------------------------------------------------------------------------

class Pattern {

protected:

  // Edge thresholds are defined by the expected difference in the
  // light/dark transition.  The actual threshold is computed
  // based on the size of the window and the size of the mask;

  int lthreshold, uthreshold;
  float _sensitivity;
  int mwidth;

public:
  virtual ~Pattern() {};

  // A virtual constructor needed by Line

  virtual Pattern *dup() = 0;

					     // amount of padding needed to
					     // detect the pattern on the edges
					     // of an image
#ifndef SGI  					       
  virtual int xpad (int width, int height) = 0;
  virtual int ypad (int width, int height) = 0;

  virtual offset find (Image *im) = 0;
#else
  virtual int xpad (int width, int height) {return 0;};
  virtual int ypad (int width, int height) {return 0;};

  virtual offset find (Image *im) {offset x; return x;};
#endif

  float sensitivity() { return _sensitivity; };
  void set_sensitivity(float sen) { _sensitivity =sen; };

  int lower_threshold(int image_height) {
    return (mwidth/2) * image_height * lthreshold;
  }

  int upper_threshold(int image_height) {
    return (mwidth/2) * image_height * uthreshold;
  }
  
};

//-----------------------------------------------------------------------------
//  Absmedge class declaration
//  This class implements a pattern that looks for an absolute edge with a
//  -2, -1, 0, 1, 2 mask.
//-----------------------------------------------------------------------------

class Absmedge: public Pattern {
protected:
  int *mask;
  float alpha;				     // angle offset at which to check
					     // for tilted line
  float maxdiffangle;			     // maximum amount of angle offset

public:
  Absmedge (int mwidth_in = default_mwidth, float alpha_in = default_alpha);
  Absmedge (Absmedge &);
  ~Absmedge() { delete mask; }

  virtual Pattern *dup() {return new Absmedge(*this);};

  int xpad (int, int height)
    { return mwidth-1 + 2 * round (half (height) * tan (alpha)); }
  int ypad (int , int) { return 0; }
  offset find (Image *im);

};


//-----------------------------------------------------------------------------
//  Edge class declaration
//  This is a simple, fast, signed edge with a -1, 1 mask.
//-----------------------------------------------------------------------------

typedef enum {none, lightdark, darklight, any} linetype;

class Edge: public Pattern {

  float alpha;

  // By convention, a line is named by the brightness change when
  // facing the the direction of the line.  By default, a line
  // is unsigned.  By setting the transition type to "any,"
  // the line starts unsigned and changes once it locks on.

  linetype trans_type;

public:
  Edge (int mwidth_in = default_mwidth, float alpha_in = default_alpha) {
    mwidth = mwidth_in; alpha = alpha_in; trans_type = any;
    if (odd (mwidth)) mwidth++;		     // the algorithm only makes sense
					     // for even mask widths
    lthreshold = default_edge_threshold;
    uthreshold = 256;
    _sensitivity = default_sensitivity;
  }

  Edge (Edge &e) {*this = e;};

  virtual Pattern *dup() {return new Edge(*this);};

  int xpad (int , int height)
    { return mwidth-1 + 2 * round (half (height) * tan (alpha)); }
  int ypad (int , int ) { return 0; }
  offset find (Image *im);

  linetype &transition_type() {return trans_type;}
};


//-----------------------------------------------------------------------------
//  Edge class declaration
//  This is a simple, fast, signed edge with a -1, 1 mask.
//-----------------------------------------------------------------------------

class Max_Edge: public Pattern {

  // By convention, a line is named by the brightness change when
  // facing the the direction of the line.  By default, a line
  // is unsigned.  By setting the transition type to "any,"
  // the line starts unsigned and changes once it locks on.

  linetype trans_type;

public:
  Max_Edge (int mwidth_in = default_mwidth, int min_threshold = 0) {
    mwidth = mwidth_in; trans_type = any;
    if (odd (mwidth)) mwidth++;		     // the algorithm only makes sense
					     // for even mask widths
    lthreshold = min_threshold;
    _sensitivity = default_sensitivity;
  }

  Max_Edge (Max_Edge &e) {*this = e;};

  virtual Pattern *dup() {return new Max_Edge(*this);};

  int xpad (int , int)
    { return mwidth-1; }
  int ypad (int , int ) { return 0; }
  offset find (Image *im);

  linetype &transition_type() {return trans_type;}
};

class GeneralEdge: public Pattern {
protected:
  int *mask;
  float alpha;				     // angle offset at which to check
					     // for tilted line
  float maxdiffangle;			     // maximum amount of angle offset
  int lower_threshold,upper_threshold;

  linetype trans_type;

public:
  GeneralEdge (int mwidth_in = default_mwidth, linetype trans_type_in = any, 
	       float alpha_in = default_alpha); 
  GeneralEdge (GeneralEdge &);
  ~GeneralEdge() { delete mask; }

  virtual Pattern *dup() {return new GeneralEdge(*this);};

  int xpad (int, int height)
    { return mwidth-1 + 2 * round (half (height) * tan (alpha)); }
  int ypad (int , int) { return 0; }
  offset find (Image *im);

};


class GaussEdge: public GeneralEdge {

public:
  GaussEdge (int mwidth_in = default_mwidth, 
	     linetype trans_type_in = any, float alpha_in = default_alpha);
  

  GaussEdge (GaussEdge &e) : GeneralEdge(e) {};

  virtual Pattern *dup() {return new GaussEdge(*this);};
};

//-----------------------------------------------------------------------------
//  Shortedge class declaration
//  This is a simple, fast signed edge of length one.  Used primarily
//    for contour tracking.
//-----------------------------------------------------------------------------

class Shortedge: public Pattern {

  // Shortedges are Edges of length 1, so no angle is required. 

  int maxpeaks;                              // half of image width

public:
  Shortedge (int mwidth_in = default_mwidth);

  Shortedge (Shortedge &s) {*this = s;};
  virtual Pattern *dup() {return new Shortedge(*this);};


  int xpad (int , int )
    { return mwidth-1 + 2; }
  int ypad (int , int ) { return 0; }

  // This just returns the offset of the strongest peak.
  offset find (Image *im);

  // This fills  lowout  with relevant info.  An upscale find fcn.
  void getlowout_ct (Image *im, SE_out *lowout);
};

//-----------------------------------------------------------------------------
#endif

