/*
    Copyright (C) 1991 Gregory D. Hager and Sidd Puri (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//-----------------------------------------------------------------------------
//
//  Pattern.cc
//
//  Definition of Pattern subclass member functions.
//
//  7 - 21 - 93
//  Sidd Puri
//
//-----------------------------------------------------------------------------

#include <string.h>
#include <stdio.h>
#include "Image.hh"
#include "Pattern.hh"

//-----------------------------------------------------------------------------
//  Private functions
//-----------------------------------------------------------------------------

static inline void sumcor (int *sum, int *cor, int cwidth, int mwidth,
			   linetype l=any) {
  int val = 0;				     // initialize running value
  int i;
  for (i = 0; i < mwidth/2; i++)
    val -= sum[i];
  for (; i < mwidth; i++)
    val += sum[i];

  int *left  = sum - 1;
  int *mid   = sum - 1 + mwidth/2;
  int *right = sum - 1 + mwidth;

  switch (l) {

  case any:
    
    cor[0] = abs (val);

    for (i = 1; i < cwidth; i++) {	     // run across the vector
      val += left[i] - ( mid[i]<<1) + right[i];
      cor[i] = abs (val);
    }
    break;

  case lightdark:

    cor[0] = -val;

    for (i = 1; i < cwidth; i++) {	     // run across the vector
      val += left[i] - ( mid[i]<<1) + right[i];
      cor[i] = -val;
    }
    break;

  case darklight:
  case none:

    cor[0] = val;
    for (i = 1; i < cwidth; i++) {	     // run across the vector
      val += left[i] - ( mid[i]<<1) + right[i];
      cor[i] = val;
    }
    break;

  }

}

static peak interparb (int left, int mid, int right) {
					     // apply quadratic interpolation
  const float a = half (left - 2 * mid + right);
  peak result;

  if (fabs(a) < TINY) {
    result.x = 0.0;
    result.val = mid;
    return result;
  }

  const float b = half (right - left);


  result.x = -b / (2 * a);
  result.val = a * result.x * result.x + b * result.x + mid;

  return result;
}

static peak interparbangle (float left, float mid, float right) {
					     // apply quadratic interpolation
					     // for angles
  const float a = (left - 2 * mid + right) / 2;
  const float b = (right - left) / 2;
  peak result;

  if (fabs(a) < TINY) {
    if (fabs(b) < TINY) {
      result.x = 0.0;
      result.val = mid;
    }
    else
      if (right > left) {
	result.x = 1.0;
	result.val = right;
      }
      else {
	result.x =  -1.0;
	result.val = left;
      }
    return result;
  }

  if (a >= 0) {
    if (right > left) {
      result.x = 1.0;
      result.val = right;
    }
    else {
      result.x =  -1.0;
      result.val = left;
    }
  }
  else {
    result.x = -b / (2 * a);
    result.val = a*result.x*result.x + b*result.x + mid;
  }

  return result;
}
/*
static peak interbell (float left, float mid, float right) {
					     // apply interpolation on a bell
  peak result;
  if (fabs(mid) < TINY) {
    result.x = result.val = 0;
    return result;
  }

  float a = abs (left + right - 2 * left * right / mid);
  if (fabs(a) < TINY) return 0.0;
  float result = (right - left) / (2 * a);
  return result;
}
*/
static peak findpeak (int *cor, int cwidth, int threshold) {
//  peak best = { 0, 0 };
	peak best;
	best.x = 0;
	best.val = 0;

  for (int i = 1; i < cwidth - 1; i++)
    if (cor[i] >= cor[i - 1] && cor[i] >= cor[i + 1]) {
      peak nexttry = interparb (cor[i - 1], cor[i], cor[i + 1]);
      if ( nexttry.val >= best.val + threshold ||
	  (nexttry.val >= best.val - threshold && abs (nexttry.x) < abs (best.x))) {
	best.x = i + nexttry.x - half (cwidth);
	best.val = nexttry.val;
      }
    }
  if (best.val == 0) best.val = cor[cwidth/2];

  return best;
}

static peak findpeak (int *cor, int cwidth, int threshold, linetype l) {
//  peak best = { 0, 0 };
	peak best;
	best.x = 0;
	best.val = 0;
  int i;

  if (l == none) {
    for (i = 1; i < cwidth - 1; i++)
      if (cor[i] >= cor[i - 1] && cor[i] >= cor[i + 1]) {
	peak nexttry = interparb (cor[i - 1], cor[i], cor[i + 1]);
	nexttry.x = i + nexttry.x - half (cwidth);
	if ((nexttry.val - (nexttry.x+1)*threshold) > 
	    ((best.val) - (best.x+1)*threshold)) {
	  best.x = nexttry.x;
	  best.val = nexttry.val;
	}
      }

    for (i = 1; i < cwidth - 1; i++)
      if (-cor[i] >= -cor[i - 1] && -cor[i] >= -cor[i + 1]) {
	peak nexttry = interparb (-cor[i - 1], -cor[i], -cor[i + 1]);
	nexttry.x = i + nexttry.x - half (cwidth);
	if ((nexttry.val - (nexttry.x+1)*threshold) > 
	    ((abs(best.val)) - (best.x+1)*threshold)) {
	  best.x = nexttry.x;
	  best.val = -nexttry.val;
	}
      }
  }
  else {
    int besti = 1;
    for (i = 2; i < cwidth - 1; i++)
      if (cor[i] >= cor[besti])	besti  = i;
    
    if (cor[besti] > threshold) {
      best = interparb (cor[besti - 1], cor[besti], cor[besti + 1]);
      best.x = besti + best.x - half (cwidth);
    }
  }

  if (best.val == 0) best.val = cor[cwidth/2];

  return best;
}

static float average (peak a, peak b, peak c) {
  return (a.x * a.val + b.x * b.val + c.x * c.val) / (a.val + b.val + c.val);
}

//- mode computation ---------------------------------------------------
//   Output is x, where all numbers from x to x+(DIVBY-1) were thrown
//      into bin x/HALF_DIVBY, and that bin contained the most numbers
//      after tally (in case of tie, first such bin is chosen).
//----------------------------------------------------------------------

int const NUMBINS = 26;                   // NUMBINS * DIVBY ~= 256
int const DIVBY = 10;
int const HALF_DIVBY = 5;

int mode (int *arr, int start, int finish) {
  int i;
  int max=0;
  int bin[NUMBINS*2];

  memset(bin, 0, sizeof(int)*NUMBINS*2);

  for (i=start;i<finish;i++) 
    bin[2*(arr[i]/DIVBY)] += 1;
  for (i=start;i<finish;i++)
    bin[2*((arr[i]+(HALF_DIVBY))/DIVBY)-1] += 1; 
  
  for (i=0;i<NUMBINS*2;i++) {
    if (bin[i] >bin[max]) max = i;
  }

  return max*(HALF_DIVBY);
}

//- findpeaks ------------------------------------------------------------
//   Find all peaks and compute mode values between peaks and boundaries.
//   Return output in SE_out structure,  data .
//------------------------------------------------------------------------

int const UP = 1;
int const DOWN = 0;

int findpeaks (int *cors, int *is, int cwidth, int, 
	       int thresh, SE_out *data) {
  int i, last;
  int pks = 0;

  int *cordiffs = new int[cwidth];

  // compute direction of slope (UP or DOWN)
  for (i=0;i<cwidth;i++)
    if ((cors[i] > thresh) && (cors[i] > cors[i+1]))
      cordiffs[i] = DOWN;
    else
      cordiffs[i] = UP;

  // find peaks (defined as an UP-DOWN sequence)
  last = cordiffs[0];
  for (i=1;i<cwidth;i++) {
    if ((last == UP) && (cordiffs[i] == DOWN)) {
      data->pk[pks] = i; 
      data->pkstrength[pks] = cors[i];      // would i+1 be better?
      pks++;
    }
    last = cordiffs[i];
  }

  // compute mode values between peaks
  data->regionmode[0] = mode(is, 0, data->pk[0]-1);
  for (i=1;i<pks;i++)
    data->regionmode[i] = mode(is, data->pk[i-1],data->pk[i]-1);
  data->regionmode[pks] = mode(is, data->pk[pks-1], cwidth-1); 

#ifndef __GNUC__
  delete [] cordiffs;
#endif

  return pks;
}
  
// Just for debugging Shortedge stuff.
void printSE_outp (SE_out *s, int wid, char *src = "") {
  int i,k=0;
  printf ("%s: SE_out: \n",src);
  printf ("   numpks: %d\n",s->numpks);
  printf ("   num     imval     pk      pkstrength     regionmode \n");
  for (i=0;i<wid; i++) {
    if (s->pk[k] == i) { 
      printf ("   %2d:     %3d        %2d       %7d          %3d\n",
	      i,   s->imval[i], s->pk[k], s->pkstrength[k], s->regionmode[k]);
      k++;
    } else {
      printf ("   %2d:     %3d\n",i,s->imval[i]);
    }
  }
  printf ("     :                                         %3d\n",
	  s->regionmode[k]);
}


//-----------------------------------------------------------------------------
//  Member functions for Absmedge, Edge, and Shortedge patterns.
//-----------------------------------------------------------------------------

Absmedge::Absmedge (int mwidth_in, float alpha_in) {
  mwidth = mwidth_in;
  alpha = alpha_in;
  int i;

  mask = new int[mwidth];		     // initialize mask
  for (i = 0; i < mwidth/2 - 1; i++)
    mask[i] =  -2;
  mask[i++] =  -1;
  if (odd (mwidth))
    mask[i++] = 0;
  mask[i++] =   1;
  for (; i < mwidth; i++)
    mask[i] =   2;

  maxdiffangle = alpha;

  for (i = 0; i < mwidth/2; i++)	     // initialize simple mask
    mask[i] = -1;
  for (; i < mwidth; i++)
    mask[i] =  1;

  lthreshold = default_edge_threshold*2;
  uthreshold = 256;
  _sensitivity = default_sensitivity;
}

Absmedge::Absmedge(Absmedge &e) 
{
  *this  = e;
  mask = new int[mwidth];
  memcpy(mask,e.mask,sizeof(int)*mwidth);
}

//-----------------------------------------------------------------------------

offset Absmedge::find (Image * im)
{
  int width =  im->width();
  int height = im->height();
  int *image =  im->data();

  float tanalpha = tan (alpha);		     // set up constants
  int pad = round (half (height) * tanalpha);
  int cwidth = width - 2 * pad - (mwidth - 1);

#ifdef __GNUC__
  int cors[3 * cwidth + 7 * pad];	     // allocate space for correlations
#else
  int *cors = new int[3 * cwidth + 7 * pad];	   
#endif
  int *midcor = &cors[                 pad], //   at angle,
      *poscor = &cors[    cwidth + 3 * pad], //   at angle + alpha, and
      *negcor = &cors[2 * cwidth + 5 * pad]; //   at angle - alpha
  int i;
//  for (int i = cwidth - 1; i >= 0; i--)
//    midcor[i] = poscor[i] = negcor[i] = 0;
  memset(cors,0,sizeof(int)*(3*cwidth + 7*pad));

  for (int row = 0; row < height; row++) {   // compute correlations
    int *imagerow = &image[row * width];
    int offset = round ((half (height) - row) * tanalpha);
					     // the following loop is optimized
					     // handle with care
    int loopend = cwidth + abs (offset);
    for (int col = -abs (offset); col < loopend; col++) {
      int *imageplace = &imagerow[pad + col];
      int sum = 0;
      register int *regmask = mask;
      for (i = mwidth - 1; i >= 0; i--)
	sum += imageplace[i] * regmask[i];
      sum = abs (sum);
      midcor[col]          += sum;
      poscor[col + offset] += sum;
      negcor[col - offset] += sum;
    }
  }

  int thresh = int(lower_threshold(height)*_sensitivity);
			     // compute delta from correlations
  peak midpeak = findpeak (midcor, cwidth, thresh),
       pospeak = findpeak (poscor, cwidth, thresh),
       negpeak = findpeak (negcor, cwidth, thresh);
  peak maxpeak = interparbangle (negpeak.val, midpeak.val, pospeak.val);
  offset delta;
//  delta.angle = interbell (negpeak.val, midpeak.val, pospeak.val) * alpha;
  delta.angle = maxpeak.x* alpha;
  delta.val = maxpeak.val;
  if ((delta.angle) >  maxdiffangle) delta.angle =  maxdiffangle;
  if ((delta.angle) < -maxdiffangle) delta.angle = -maxdiffangle;
  float ratio = delta.angle/maxdiffangle;
/*  if (delta.angle >= alpha)
    delta.x = pospeak.x;
  else if (delta.angle >= 0)
    delta.x =  delta.angle * pospeak.x + (1-delta.angle) * midpeak.x;
  else if (delta.angle > -alpha)
    delta.x = -delta.angle * negpeak.x + (1+delta.angle) * midpeak.x;
  else
    delta.x = negpeak.x;
*/

  if (ratio >= 0)
    delta.x =  ratio * pospeak.x + (1-ratio) * midpeak.x;
  else 
    delta.x = -ratio * negpeak.x + (1+ratio) * midpeak.x;


#ifndef __GNUC__
  delete [] cors;
#endif

  return delta;
}

//-----------------------------------------------------------------------------

offset Edge::find (Image * im)
{
  int width =  im->width();
  int height = im->height();
  int *image =  im->data();

  //cout<<"width= "<<width<<" height="<<height<<endl;cout.flush();

  float tanalpha = tan (alpha);		     // set up constants
  int pad = round (half (height) * tanalpha);
  int swidth = width - 2*pad;
  int cwidth = swidth - (mwidth - 1);

#ifdef __GNUC__
  int sums[3 * swidth + 7 * pad];	     // allocate space for sums
#else
  int *sums = new int[3 * swidth + 7 * pad];
#endif
  int *midsum = &sums[                pad],  //   at angle,
      *possum = &sums[    swidth + 3 * pad], //   at angle + alpha, and
      *negsum = &sums[2 * swidth + 5 * pad]; //   at angle - alpha

  memset(sums,0,sizeof(int)*(3*swidth + 7*pad));

  for (int row = 0; row < height; row++) {   // compute sums
    int *imagerow = &image[row * width];
    int offset = round ((half (height) - row) * tanalpha);
					     // the following loop is optimized
					     // handle with care
    int loopend = swidth + abs (offset);
    for (int col = -abs (offset); col < loopend; col++) {
      int val = imagerow[pad + col];
      midsum[col]	   += val;
      possum[col + offset] += val;
      negsum[col - offset] += val;
    }
  }

#ifdef __GNUC__
  int cors[3 * cwidth];			     // allocate space for correlations
#else
  int *cors = new int[3*cwidth];
#endif

  int *midcor = &cors[0         ],
      *poscor = &cors[    cwidth],
      *negcor = &cors[2 * cwidth];

  // compute correlations from sums

  sumcor (midsum, midcor, cwidth, mwidth, transition_type());  
  sumcor (possum, poscor, cwidth, mwidth, transition_type());
  sumcor (negsum, negcor, cwidth, mwidth, transition_type());    

  int thresh = int(lower_threshold(height)*_sensitivity);
		     // compute delta from correlations

  peak midpeak,pospeak,negpeak;
  offset delta;
  if (transition_type() == none) {
    midpeak = findpeak (midcor, cwidth, thresh, transition_type());
    pospeak = findpeak (poscor, cwidth, thresh, transition_type());
    negpeak = findpeak (negcor, cwidth, thresh, transition_type());

//  cout << negpeak.val << " " << midpeak.val << " " << pospeak.val << " ";
/*
  delta.angle = interbell (negpeak.val, midpeak.val, pospeak.val) * alpha;
					     // this approximation works, but
					     // the following gives much better
					     // behavior; why exactly this is I
					     // haven't been able to figure out
  float fudge = atan (height/20) * M_PI_2/alpha;
  delta.angle = atan (delta.angle * fudge) / fudge;
*/


    float maxval = max2(max2(negpeak.val,midpeak.val),pospeak.val);
    float minval = min2(min2(negpeak.val,midpeak.val),pospeak.val);
    if (-minval < maxval)
      transition_type()  = darklight;
    else
      transition_type() = lightdark;
    delta.val = max2((float)(-minval),maxval);
    delta.x = 0.0;
  }
  else{

    midpeak = findpeak (midcor, cwidth, thresh);
    pospeak = findpeak (poscor, cwidth, thresh);
    negpeak = findpeak (negcor, cwidth, thresh);

    peak maxpeak = interparbangle(negpeak.val, midpeak.val, pospeak.val);
    delta.val = maxpeak.val;
    
    float &ratio = maxpeak.x;
    if (ratio > 1.0) ratio = 1.0;
    if (ratio < -1.0) ratio = -1.0;
    delta.angle = alpha*ratio;
    if (ratio >= 0)
      delta.x =  ratio * pospeak.x + (1-ratio) * midpeak.x;
    else 
      delta.x = -ratio * negpeak.x + (1+ratio) * midpeak.x;
  }

#ifndef __GNUC__
  delete [] sums;
  delete [] cors;
#endif

  if (delta.val<lower_threshold(height) )
	delta.val=0;
//  else if (delta.x>1) 
//	cout<<"offset="<<delta.x<<","<<delta.angle<<endl; cout.flush();

  return delta;
}

//-----------------------------------------------------------------------------

offset Max_Edge::find (Image* im)
{
  int width =  im->width();
  int height = im->height();
  int *image =  im->data();

  int swidth = width;
  int cwidth = swidth - (mwidth - 1);

#ifdef __GNUC__
  int sums[swidth];	     // allocate space for sums
#else
  int *sums = new int[swidth];
#endif
  memset(sums,0,sizeof(int)*swidth);

  int *imagerow = image;
  for (int row = 0; row < height; row++) {   // compute sums
    for (int col = 0; col < swidth; col++) {
      sums[col] += *imagerow++;
    }
  }

#ifdef __GNUC__
  int cors[cwidth];			     // allocate space for correlations
#else
  int *cors = new int[cwidth];
#endif

  // compute correlations from sums

  sumcor (sums, cors, cwidth, mwidth, transition_type());  

  int thresh = int(lower_threshold(height)*_sensitivity);
		     // compute delta from correlations

  peak midpeak;
  offset delta;
  delta.angle = 0;

  midpeak = findpeak (cors, cwidth, 0, transition_type());
  cout << "Peak " << midpeak.x << " " << midpeak.val << endl;

  if (midpeak.val > lower_threshold(height)) {
    delta.val = midpeak.val;
    delta.x = midpeak.x;
  }
  else
    delta.val = 0;

#ifndef __GNUC__
  delete [] sums;
  delete [] cors;
#endif

  return delta;
}

//-----------------------------------------------------------------------------

Shortedge::Shortedge (int mwidth_in) {
  mwidth = mwidth_in;
  if (odd (mwidth)) mwidth++;		     // the algorithm only makes sense
                                             // for even mask widths
  lthreshold = default_edge_threshold;
  uthreshold = 256;
  lthreshold = int(default_sensitivity);
}


offset Shortedge::find (Image *im) {

  int width =  im->width();
  int height = im->height();
  int *image = im->data();
  int cwidth = width - mwidth - 1;

  // compute sums

  int *sums = new int[width];
  memset(sums,0,sizeof(int)*(width));
  for (int col = 0; col < width; col++) {
    sums[col] += image[col];
  }

  // compute correlations from sums.
  int *cors = new int[cwidth];
  sumcor (sums, cors, cwidth, mwidth);  

  int thresh = int(lower_threshold(height)*_sensitivity);

  // find maximum peak.
  peak res = findpeak (cors, cwidth, thresh);
 
#ifndef __GNUC__
  delete [] sums;
  delete [] cors;
#endif

  offset delta;
  delta.x = res.x;
  delta.angle = 0.0;
  delta.val = res.val;

  return delta;                                      // dummy return value
}


void Shortedge::getlowout_ct (Image *im, SE_out *lowout) {

  int width =  im->width();
  int height = im->height();
  int *image = im->data();
  int cwidth = width - mwidth - 1;

  // compute sums

  int *sums = new int[width];
  memset(sums,0,sizeof(int)*(width));
  for (int col = 0; col < width; col++) {
    sums[col] += image[col];
  }

  // compute correlations from sums.
  int *cors = new int[cwidth];
  sumcor (sums, cors, cwidth, mwidth);  

  int thresh = int(lower_threshold(height)*_sensitivity); // better threshold?

  // find peaks, set output values.
  lowout->numpks = findpeaks (cors, image, cwidth, width, thresh, lowout);
 
  memcpy(lowout->imval, image, width*sizeof(int));

#ifndef __GNUC__
  delete [] sums;
  delete [] cors;
#endif
}

// Gedge

GaussEdge::GaussEdge (int mwidth_in, linetype line_type_in,float alpha_in) {
  double sigma;
  int i;

  mwidth = 2*mwidth_in+1;
  alpha = alpha_in;
  trans_type = line_type_in;

  mask = new int[mwidth];		     // initialize mask
  mask[mwidth_in] = 0;
  sigma = ((double)mwidth_in)/4.0;

  // compute a normalized derivative of a Gaussian

  for(i=1;i<=mwidth_in;i++) {
      mask[mwidth_in-i] = - (mask[mwidth_in +i] = 
			(int) ((i*exp( -i*i/(2.0*sigma*sigma)))*100+0.5));
      cout << mask[mwidth_in-i] << " ";
    }
  cout << endl;

  maxdiffangle = alpha;

  lthreshold = default_edge_threshold;
  uthreshold = 100000;
  _sensitivity = default_sensitivity;
}

//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//  Member functions for GeneralEdge, Edge, and Shortedge patterns.
//-----------------------------------------------------------------------------

GeneralEdge::GeneralEdge (int mwidth_in, linetype line_type_in, float alpha_in) {
  mwidth = mwidth_in;
  alpha = alpha_in;
  trans_type = line_type_in;
  int i;

  mask = new int[mwidth];		     // initialize mask
  for (i = 0; i < mwidth/2 - 1; i++)
    mask[i] =  -2;
  mask[i++] =  -1;
  if (odd (mwidth))
    mask[i++] = 0;
  mask[i++] =   1;
  for (; i < mwidth; i++)
    mask[i] =   2;

  maxdiffangle = alpha;

  for (i = 0; i < mwidth/2; i++)	     // initialize simple mask
    mask[i] = -1;
  for (; i < mwidth; i++)
    mask[i] =  1;

  lthreshold = default_edge_threshold;
  uthreshold = 1000000;
  _sensitivity = default_sensitivity;
}

GeneralEdge::GeneralEdge(GeneralEdge &e) 
{
  *this  = e;
  mask = new int[mwidth];
  memcpy(mask,e.mask,sizeof(int)*mwidth);
}

//-----------------------------------------------------------------------------

inline int modsum(int sum, linetype l){
  switch(l) {
  case lightdark:
    sum = -sum;
    break;
  case any:
    sum = abs(sum);
    break;
  case darklight:
  case none:
    panic("modsum"); // ToDo: is this right? ADR
  }

  return sum;
}

offset GeneralEdge::find (Image * im)
{
  int width =  im->width();
  int height = im->height();
  int *image =  im->data();

  float tanalpha = tan (alpha);		     // set up constants
  int pad = round (half (height) * tanalpha);
  int swidth = width - 2*pad;
  int cwidth = swidth - (mwidth - 1);


#ifdef __GNUC__
  int cors[3 * cwidth + 7 * pad];	     // allocate space for correlations
#else
  int *cors = new int[3 * cwidth + 7 * pad];	   
#endif
  int *midcor = &cors[                 pad], //   at angle,
      *poscor = &cors[    cwidth + 3 * pad], //   at angle + alpha, and
      *negcor = &cors[2 * cwidth + 5 * pad]; //   at angle - alpha
  int i;
  memset(cors,0,sizeof(int)*(3*cwidth + 7*pad));

  if ((trans_type == any ) || (trans_type == none)) {
    for (int row = 0; row < height; row++) {   // compute correlations
      int *imagerow = &image[row * width];
      int offset = round ((half (height) - row) * tanalpha);
					     // the following loop is optimized
					     // handle with care
      int loopend = cwidth + abs (offset);
      for (int col = -abs (offset); col < loopend; col++) {
	int *imageplace = &imagerow[pad + col];
	int sum = 0;
	register int *regmask = mask;
	for (i = mwidth - 1; i >= 0; i--)
	  sum += imageplace[i] * regmask[i];
	sum = modsum(sum,trans_type)/256;
	midcor[col]          += sum;
	poscor[col + offset] += sum;
	negcor[col - offset] += sum;
      }
    }
  }
  else {
#ifdef __GNUG_
    int sums[3 * swidth + 7 * pad];	     // allocate space for sums
#else
    int *sums = new int[3 * swidth + 7 * pad];
#endif
    int *midsum = &sums[                pad],  //   at angle,
        *possum = &sums[    swidth + 3 * pad], //   at angle + alpha, and
        *negsum = &sums[2 * swidth + 5 * pad]; //   at angle - alpha

    memset(sums,0,sizeof(int)*(3*swidth + 7*pad));

    for (int row = 0; row < height; row++) {   // compute sums
      int *imagerow = &image[row * width];
      int offset = round ((half (height) - row) * tanalpha);
					     // the following loop is optimized
					     // handle with care
      int loopend = swidth + abs (offset);
      for (int col = -abs (offset); col < loopend; col++) {
	int val = imagerow[pad + col];
	midsum[col]	   += val;
	possum[col + offset] += val;
	negsum[col - offset] += val;
      }
    }

//    cout << "Fastcode" << endl;
    int imageplace;
    for (int col = 0; col < cwidth; col++) {
      register int *regmask = mask;
      for (i = mwidth - 1; i >= 0; i--) {
	imageplace = col+i;
	midcor[col] += midsum[imageplace]* regmask[i];
	poscor[col] += possum[imageplace]* regmask[i];
	negcor[col] += negsum[imageplace]* regmask[i];
      }
      midcor[col] = modsum(midcor[col],trans_type)/256;
      poscor[col] = modsum(poscor[col],trans_type)/256;
      negcor[col] = modsum(negcor[col],trans_type)/256;
    }

#ifdef __GNUG__
    delete sums;
#endif

  }


/*
  cout << "Mids:      ";
  for (i = 0;i<cwidth;i++)
    cout << midcor[i] << " ";
  cout <<  endl;

  cout << "Positive:  ";
  for (i = 0;i<cwidth;i++)
    cout << poscor[i] << " ";
  cout <<  endl;

  cout << "Negative:  ";
  for (i = 0;i<cwidth;i++)
    cout << negcor[i] << " ";
  cout <<  endl;
*/

  peak midpeak = findpeak (midcor, cwidth, lower_threshold,trans_type),
       pospeak = findpeak (poscor, cwidth, lower_threshold,trans_type),
       negpeak = findpeak (negcor, cwidth, lower_threshold,trans_type);

/*  cout << "Midp: " << midpeak.x << " " << midpeak.val << endl;
  cout << "Posp: " << pospeak.x << " " << pospeak.val << endl;
  cout << "Negp: " << negpeak.x << " " << negpeak.val << endl;
*/
  offset delta = {0,0};

  // if this is a dynamic line, set up the right type 

  int minval = int(min2(midpeak.val,min2(pospeak.val,negpeak.val)));
  int maxval = int(max2(midpeak.val,max2(pospeak.val,negpeak.val)));

  if (trans_type == none) {
    if (minval * maxval > 0) {
      if (minval < 0) {
	cout << "Chose lightdark" << endl;
	trans_type = lightdark;
	midpeak.val = - midpeak.val;
	pospeak.val = - pospeak.val;
	negpeak.val = - negpeak.val;
	maxval = - minval;
      }
      else {
	cout << "Chose darklight" << endl;
	trans_type = darklight;
      }
    }
    else
      return delta;
  }

  // Edge was ambiguous

  if (minval == 0) {
    delta.val = -1;
    return delta;
  }

  // Edge was not found

  if (maxval < lower_threshold)
    return delta;

  // Set up a threshold for next time

  lower_threshold = int(.9 * maxval);

  peak maxpeak = interparbangle (negpeak.val, midpeak.val, pospeak.val);

//  delta.angle = interbell (negpeak.val, midpeak.val, pospeak.val) * alpha;
  delta.angle = maxpeak.x* alpha;
  delta.val = maxpeak.val;
  if ((delta.angle) >  maxdiffangle) delta.angle =  maxdiffangle;
  if ((delta.angle) < -maxdiffangle) delta.angle = -maxdiffangle;
  float ratio = delta.angle/maxdiffangle;

  if (ratio >= 0)
    delta.x =  ratio * pospeak.x + (1-ratio) * midpeak.x;
  else 
    delta.x = -ratio * negpeak.x + (1+ratio) * midpeak.x;


#ifndef __GNUC__
  delete [] cors;
#endif

  return delta;
}


