/*                                                                -*-c++-*-
    Copyright (C) 1993 Gregory D. Hager and Sidd Puri (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//-----------------------------------------------------------------------------
//
//  Line.hh
//
//  Declaration of the Line object, a simple example of a Thing.
//
//  6 - 7 - 93
//  Sidd Puri
//
//-----------------------------------------------------------------------------

#ifndef Line_h
#define Line_h

#include <stream.h>
#include <stdio.h>
#include "Tracker.hh"
#include "FTypes.hh"
#include "Pattern.hh"
#include "Image.hh"
#include "BasicFeature.hh"

//-----------------------------------------------------------------------------
//  Default line parameters
//-----------------------------------------------------------------------------

int   const line_length = 31;
int   const line_width  = 20;
int   const line_statevars = 4;      // x,y, angle and strength

//-----------------------------------------------------------------------------
//  Line class declaration
//-----------------------------------------------------------------------------

class Line: public BasicFeature, public LineSegment
{

  // Length is the length of the line; width is the total size
  // of the area perpendicular to the line to search
  // samp_length and samp_width determine subsampling of these
  // two dimensions.

  int len, wid;
  int samp_length, samp_width;

  // Local pattern and video pointers.

  Pattern *pat;
  Image *im;

  float window_angle() { return -orientation()-M_PI_2; }

  static const char *_typeid;

public:

  // Two ways of creating a line.  By default, a line uses
  // a simple edge, but you can supply your own if you want.

  Line (Video *v,int length_in = line_length, int width_in = line_width);
  Line (Pattern *pat_in, Video *v,
	int length_in = line_length, int width_in = line_width);
  Line (const Line &l);

  ~Line() {delete im;delete pat;}

  int in_view(int x, int y, float o) 
	{return source()->in_view(*im,x,y,o);}
  int in_view(double x, double y, float o) 
	{ return in_view(round(x), round(y), o); }
  int in_view(float x, float y, float o)
	{ return in_view(round(x), round(y), o); }

  // Line initialization 

  int state_init(float *s) {return state_init(s[0],s[1],s[2]);};
  int state_init (float x_in, float y_in, float angle_in = 0);
  int interactive_init (CWindow &w);

  BasicFeature *dup() { return new Line (*this); }

  // Display routines

  int show (Video &v, Color color_in = default_color);
  int clear (Video &v);
  int show(Color color_in = default_color) {return show(*v,color_in);};
  int clear() {return clear(*v);};

  int in_view() {return in_view(round(x()),round(y()),orientation());}

  // Update routines

  int update (Video &v);
  int update(){return update(*v);};

  // Changing sampling.

  int &length_sampling() {return samp_length;};
  int &width_sampling() {return samp_width;};

  // information accessors

  double width()  const {return wid*samp_width;};
  double length() const {return len*samp_length;};
  double rawlength() const {return len; }

  // state variable accessors

  float x()	      const   { return state[0];}	     
  float y()	      const   { return state[1];}
  float orientation() const   { return state[2];}
  float strength()    const   { return state[3];}
  Image *image()      { return im; }

  Video *source()	{ return v; } //video source

  // A mutated version of line coordinates

  line2D coordinates() const {
    line2D temp;
    temp.x = -sin(orientation());
    temp.y = cos(orientation());
    temp.z = -(temp.x * x() + temp.y * y());

    return temp;
  }

  // Endpoints

  point2D endpt1() const {
    point2D temp;
    temp.x = x() - cos(orientation())*length()/2;
    temp.y = y() - sin(orientation())*length()/2;
    return temp;
  };

  point2D endpt2() const {
    point2D temp;
    temp.x = x() + cos(orientation())*length()/2;
    temp.y = y() + sin(orientation())*length()/2;
    return temp;
  };

  // state variable changers

  float set_x(float newx)	      {return state[0] = newx;}	     
  float set_y(float newy)	      {return state[1] = newy;}
  float set_orientation(float newo)   {return state[2] = newo;}
  float set_strength(float news)      {return state[3] = news;}

  // Special function for "sliding" lines

  void move(float len);

  // Function to follow line.
  
  int findnext(int len, int range);

  // Output

  void show (CWindow &w)
    { w.show (*im); }
  void show (char *filename)
    { im->write_image (filename); }

  // Typing

  static const char *info() {return _typeid;};

};

//-----------------------------------------------------------------------------
#endif
