/*
    Copyright (C) 1991 Gregory D. Hager and Sidd Puri (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//-----------------------------------------------------------------------------
//
//  Line.cc
//
//  Definition of Line member functions.
//
//  6 - 7 - 93
//  Sidd Puri
//
//-----------------------------------------------------------------------------

#include <stdlib.h> // for exit()
#include "CWindow.hh"
#include "Video.hh"
#include "Line.hh"

const char *Line::_typeid = "Line";

// Creation

Line::Line (Video *v,int length_in, int width_in):
     BasicFeature (line_statevars,v,_typeid), LineSegment(this)
    { 
    len = length_in; wid = width_in; pat = new Edge;
    samp_length = samp_width = 1;
    im = new Image (wid  + pat->xpad (wid, len),
		    len + pat->ypad (wid, len));
  }


Line::Line (Pattern *pat_in, Video *v,
	    int length_in, int width_in): 
	    BasicFeature (line_statevars,v,_typeid), LineSegment(this)
{
    len = length_in; wid = width_in; pat = pat_in->dup();
    samp_length = samp_width = 1;
    im = new Image (wid  + pat->xpad (wid, len),
		    len + pat->ypad (wid, len));
  }

Line::Line (const Line &l): BasicFeature (l), LineSegment(this)//(void *)this)
    { len = l.len; wid = l.wid; samp_length = l.samp_length;
      samp_width = l.samp_width;
      pat = l.pat->dup(); im  = new Image(*(l.im));
    }

//-----------------------------------------------------------------------------
//  Thing member functions
//-----------------------------------------------------------------------------

int _show = 0;

int Line::state_init (float x_in, float y_in, float angle_in) {
  set_x(x_in);
  set_y(y_in);
  set_orientation(angle_in);
  return 1;
}

int Line::interactive_init (CWindow &w) {
  position pos = w.getpos (*v, "Position line",len);
  state_init (pos.x, pos.y, pos.angle);
  return 1;
}

int Line::show (Video &v, Color color_in) {
  if (in_view(round(x()), round(y()), orientation())) {
        v.line (round(x()), round(y()), round(rawlength()), orientation(), 
	        map_color(color_in), 
	        length_sampling());
        return 1;
  }
  else {
       puts("Line::Show:not in view."); 
       return -1; 
  }
}

int Line::clear (Video &v) {
  v.line (round(x()), round(y()), round(length()) , orientation(), 0, length_sampling());
  return 1;
}

int Line::update (Video &v) {

  double start_x=x(), start_y=y(), start_orientation=orientation();
 
  if (!(v.grab (*im, (int)x(), (int)y(), 
		  window_angle(),width_sampling(),length_sampling())))
    return 0;

  if (_show)
    v.show(*im,40,20);

  // Get canonical offsets

  offset delta = pat -> find(im);

  if (delta.val==0)
	{ /* puts("Line:not found."); */ return 0; }

  // Modify based on sampling

  double s = sin(delta.angle);
  double c = cos(delta.angle);

  delta.x = delta.x*width_sampling()*c + 
            delta.x*length_sampling()*s;

  delta.angle = atan2(width_sampling() * s,
		      length_sampling()* c);

  // Update line parameters

  float to = orientation() -delta.angle;
  if (orientation() < -twopi) set_orientation(orientation() + twopi);
  if (orientation() >  twopi) set_orientation(orientation() - twopi);

  float tx = x() + delta.x * cos (window_angle());
  float ty = y() - delta.x * sin (window_angle());

  if (in_view((int)tx,(int)ty,to)) {
    set_x(tx);
    set_y(ty);
    set_orientation(to);
    set_strength(delta.val);

//printf("update (%.1f, %.1f, %.1f)==>(%.1f, %.1f, %.1f)\n", 
//	start_x, start_y, start_orientation, tx, ty, to);

    return 1;
  }
  else
    { puts("Line:lost track(not in view)."); return 0;}

}

void Line::move(float len) {
  replicate_forward();
  set_x(x() + cos(orientation()) *len);
  set_y(y() + sin(orientation()) *len);
  erase_last();
}


//-----------------------------------------------------------------------------
//  Other functions.
//-----------------------------------------------------------------------------

double const DTHETA = M_PI / 4.0;
int const EACH = 2;

int Line::findnext(int len, int) {
  int i;
  float max=0;
  int maxindex=0;
  float theta;
  float xcenter, ycenter;
  float initangle=orientation();

  xcenter = x() + cos(orientation()) * len;
  ycenter = y() + sin(orientation()) * len;

  // Move and update along line, 
  //   shifting orientations between +/- EACH*DTHETA.
  // Search for maximum edge strength.

  for (i=-EACH;i<EACH+1;i++) {
    set_orientation( theta = initangle+(i*DTHETA));
    set_x( xcenter + cos(theta) * len);
    set_y( ycenter + sin(theta) * len);
    update();
    if ((strength() > max) && (fabs(fabs(orientation()-initangle)-M_PI)>0.5))   {
      max = strength();
      maxindex = i;
    }
  }
  
  // Set new line attributes and update. 

  set_orientation( theta = initangle+(maxindex*DTHETA));
  set_x( xcenter + cos(theta) * len);
  set_y( ycenter + sin(theta) * len);
  update();

  // Return 1 if new line is acute with respect to old, 0 otherwise.

  if ((maxindex == -EACH) || (maxindex == EACH)) return 1;
  else return 0;
}  



//-----------------------------------------------------------------------------

