
/*                                                                -*-c++-*-
    Copyright (C) 1993 Gregory D. Hager and Sidd Puri (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
/* This is the Generalized Intersection Class
 * derived from Multitrack class and using the Line Class
 *
 * G. Hager  5/13/93
 */

#ifndef GILINE_H
#define GILINE_H

#include <math.h>
#include "Line.hh"
#include "CompFeature.hh"
#include "IntersectingLines.hh"

extern "C" void sincos(double,double *,double *);


//======================== GILine Class =========================



class GILine : public CompFeature, public PointFeature
{

private:
  
  float setpoint[2];     // Setpoints governing where the edges should be
                         // relative to the corner
			 // The distance between center of the line(s)
			 // to the intersect

  LineSegment *line1, *line2;   // For convenience.
  IntersectingLines *IL;        // Use intersecting lines as a basis

protected:

  LineSegment *lchild(int n) {return ptr_cast(LineSegment,child(n));};
    

public:

  // avoid using these two, use set_state
  float set_x(float xin)      {return state[0] = xin;}
  float set_y(float yin)      {return state[1] = yin;}

  // Try not to use these if possible -- only for backward compatibility.
  float set_angle1(float ain) {return state[2] = ain;}
  float set_angle2(float ain) {return state[3] = ain;}

  float set_orientation1(float ain) {line1->set_orientation(ain); return compute_state();}
  float set_orientation2(float ain) {line2->set_orientation(ain); return compute_state();}

  GILine(LineSegment &l1, float s1, LineSegment &l2, float s2, Color color_in = default_color);
  GILine(Pattern *e, Video *v, float s1, float s2, 
	 int size = line_length, int sampling = 1, Color color_in = default_color);

  GILine(GILine &g) : CompFeature(g), PointFeature(this) {
    setpoint[0] = g.setpoint[0];
    setpoint[1] = g.setpoint[1];

    line1 = lchild(0);
    line2 = lchild(1);
    
    IL = new IntersectingLines(line1, line2);
  }

  ~GILine() {delete IL;};

  virtual Thing *dup() {return new GILine (*this); };

  // state accessors

  float x() const {return state[0];};
  float y() const {return state[1];};

  // angle1 and angle2 are provided for backward compatibility only.
  // Do not use!!
  // In case you want to know, angle1 is the orientation of line1
  // angle2 = orientation of line1 - orientation of line2

  float angle1() const {return state[2];};
  float angle2() const {return state[3];};

  // orientation1 and orientation2 are the orientations of the lines.
  float orientation1() const {return line1->orientation();}
  float orientation2() const {return line2->orientation();}
  float angle_diff() const {return state[3];}
  float angle_bisector() const { 
    return 
      atan2 ((sin(orientation1())+sin(orientation2()))/2,
	     (cos(orientation1())+cos(orientation2()))/2);
  }
  
  LineSegment *Line1() {return line1;}
  LineSegment *Line2() {return line2;};

  int set_state(float x, float y, float o1, float o2);
  int set_state(int x, int y, float o1, float o2) {
	return set_state( (float)x, (float)y, o1, o2);
  }

  void move(float offx, float offy);
  int constraints();
  int compute_state();

  int interactive_init (CWindow &w);
  int interactive_init (CWindow &w, int direction, int rl);
  int search_init (int direction, int rl);

};


//=========================  Cross Class =============================


class Cross : public GILine
{
  
public:
  
  Cross(LineSegment &l1, LineSegment &l2, Color color_in = default_color) : 
    GILine(l1,0.0,l2,0.0,color_in) {};

  Cross(Video *v, int size = line_length, Color color_in = default_color) : 
    GILine(new Absmedge,v,0,0,size,color_in) {};

  Cross(Pattern *e, Video *v, int size = line_length, Color color_in = default_color) : 
            GILine(e, v, 0, 0, size, 1, color_in) {};  


};


// ========================== Corner Class ================================


class Corner : public GILine
{
  
public:
  
  Corner(LineSegment &l1, LineSegment &l2,  Color color_in = default_color) : 
            GILine(l1, l1.length()/2, l2, l2.length()/2,color_in) {};  

  Corner(Video *v, int size = line_length, Color color_in = default_color) : 
            GILine(new Edge, v, size/2, size/2,color_in) {};  

  Corner(Pattern *e, Video *v, int size = line_length, int sampling = 1, 
	 Color color_in = default_color) : 
            GILine(e, v, (float)(size*sampling)/2.0, 
		   (float)(size*sampling)/2.0, 
		   size, sampling, color_in) {};  
  int interactive_init (CWindow &w); // same old stuff with a corner cursor

};

#endif

