/*
    Copyright (C) 1991 Gregory D. Hager and Sidd Puri (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//-----------------------------------------------------------------------------
//
//  GILine.cc
//
//  Definition of GILine member functions.
//
//  9 - 1 - 93
//  Greg Hager
//
//-----------------------------------------------------------------------------

#include <stdlib.h>
#include <stdio.h>
#include "GILine.hh"

#define MIN_CORNER_DIFF 45 

GILine::GILine(LineSegment &l1, float s1,
	       LineSegment &l2, float s2, Color color_in) : 
	       CompFeature(4,"GILine",color_in), PointFeature(this)
    {

      setpoint[0] = s1;
      setpoint[1] = s2;

      (*this) += l1;
      l2.set_orientation(l2.orientation() - M_PI_2);
      (*this) += l2;

      line1 = lchild(0);
      line2 = lchild(1);

      IL = new IntersectingLines(line1,line2,color_in);
      
    }

GILine::GILine(Pattern *e, Video *v, float s1, float s2, int size, 
	       int sampling, Color color_in) :
        CompFeature(4,"GILine",color_in), PointFeature(this)
{

      setpoint[0] = s1;
      setpoint[1] = s2;

      Line *ltemp = new Line(e,v,size);
      ltemp->length_sampling() = sampling;
      (*this) += *ltemp;
      ltemp->set_orientation(ltemp->orientation() - M_PI_2);
      (*this) += *ltemp;

      line1 = lchild(0);
      line2 = lchild(1);

      IL = new IntersectingLines(line1,line2,color_in);
    }

int GILine::interactive_init (CWindow &w, int direction, int rl) {

    cout<<"interactive_init: direction= "<<direction<<"(1 for forward)"<<endl;
    cout<<"                  turn     = "<<rl       <<"(1 for right)"  <<endl;

    (*line1)->interactive_init(w);
    return search_init(direction,rl);
  };

int GILine::interactive_init (CWindow &w) {
    double ang;

//    (*line1)->interactive_init(w);
//    ((*line1).get_handle())->interactive_init(w);
    (*(*line1)).interactive_init(w);

    line2->set_orientation((ang = line1->orientation())-M_PI_2);

    line2->set_x(line1->x() + 
      cos(ang)*setpoint[0] - cos(ang-M_PI_2)*setpoint[1]);
    line2->set_y(line1->y() +
      sin(ang)*setpoint[0] - sin(ang-M_PI_2)*setpoint[1]);

    return compute_state();
    
    
  }

int Corner::interactive_init (CWindow &w) {
    
    // notic line1 is not a Line type, but LineSegment *sih*
    
    Line *lline1;
    
   if ((lline1 = ptr_cast(Line,Line1())) == NULL)
    panic("Cannot cast Line Segment as Line in search_init");
   
    position pos=w.get_corner_pos( *(lline1->source()), "Position Corner", int(lline1->length()));
   
    // get_corner_pos() returns the (x,y) of intersection as (pos.x, pos.y)
    // orientation of line1 as pos.angle
    // pos.angle has to be rotated PI degrees 
    // to be accord with the new representation  
    // orientation of line2 is always assumed to be perpendicular with line1

    set_state(pos.x, pos.y, pos.angle+M_PI, pos.angle-M_PI_2);

    //return compute_state();
    return 1;
  }

// the current representation for corners is that
// two lines point outwards from intersection
//
// direction=1 		search forward
// rl(right or left)=1	right
//
// notice when search forward, the orientation of line1 will be flip
// to be accord with the new representation

int GILine::search_init (int direction, int rl) {

  int count = 0;
  Line *lline1,*lline2;

  if ((lline1 = ptr_cast(Line,line1)) == NULL)
    panic("Cannot cast Line Segment as Line in search_init");

  if ((lline2 = ptr_cast(Line,line2)) == NULL)
    panic("Cannot cast Line Segment as Line in search_init");

  // flip orientation of line1 if direction  == 1
  // line moves BACKWARD to search.
  if (direction>0)
      lline1->set_orientation( lline1->orientation() + M_PI );

  double o1 = lline1->orientation();
  // right is positive
  lline2->set_orientation( o1 - signum(rl) * M_PI_2);
  double o2 = lline2->orientation();

  lline2->set_x(lline1->x() - cos(o1)*setpoint[0] + cos(o2)*setpoint[1]);
  lline2->set_y(lline1->y() - sin(o1)*setpoint[0] + sin(o2)*setpoint[1]);

  update();

  while ((fabs((lline1->strength() - lline2->strength())/
	       lline1->strength()) > 0.5) && count < 1000) {
      count ++;
      // line moves BACKWARD to search.
      move(-cos(lline1->orientation())*lline2->width()/6,
	   -sin(lline1->orientation())*lline2->width()/6);

      lline2->set_orientation(o1 - signum(rl) * M_PI_2);
      o2 = lline2->orientation();
      lline1->update();
      lline1->show();
      lline2->update();
      lline1->clear();
      compute_state();
      constraints();
    }
  //return (count < 1000);
  //cout<<"search_init: count="<<count<<endl;
  return count<1000;

}

int GILine::set_state(float x, float y, float o1, float o2)
{
  line1->set_orientation(o1);
  line2->set_orientation(o2);

  line1->set_x(x + cos(o1)*setpoint[0]);
  line1->set_y(y + sin(o1)*setpoint[0]);

  line2->set_x(x + cos(o2)*setpoint[1]);
  line2->set_y(y + sin(o2)*setpoint[1]);

  return compute_state();

}

void GILine::move(float offx, float offy) {
  set_state(x() + offx,y() + offy, line1->orientation(), line2->orientation() );
}

#ifndef __GNUC__ 
void sincos(double ang,double *s, double *c){
  *s = sin(ang);*c = cos(ang);
}
#endif

int GILine::compute_state()
{

  // check if the intersection is out of view
  // if so, we have lost track

  Line *lline=ptr_cast(Line, line1); // cast into Line so we can use in_view

  if (!lline->in_view(line1->x()+setpoint[0]*cos(line1->orientation()),
	              line1->y()+setpoint[0]*sin(line1->orientation()),
	              line1->orientation()))
  if (!lline->in_view(line2->x()+setpoint[1]*cos(line2->orientation()),
	              line2->y()+setpoint[1]*sin(line2->orientation()),
	              line2->orientation())) {
		
		cout<<"GILine: lost track."<<endl;
		return 0;
  }

  int res = IL->compute_state();
  set_x(IL->x());
  set_y(IL->y());
  set_angle1(line1->orientation());
  set_angle2(line2->orientation()-angle1());
  
  return res;
}

int GILine::constraints()
{
  float o1,o2;

  // Orientations are easy. 

  line1->set_orientation(o1 = angle1());
  line2->set_orientation(o2 = (o1+angle2()));

  // Positions aren't too much harder

  line1->set_x(x() + setpoint[0] * cos(o1));
  line1->set_y(y() + setpoint[0] * sin(o1));


  line2->set_x(x() + setpoint[1] * cos(o2));
  line2->set_y(y() + setpoint[1] * sin(o2));

  return 1;
}


