/*#########################################
 * svs.h
 *
 * Header file for the SVM library
 *
 *#########################################
 */

#ifndef _svs_h_
#define _svs_h_

/**
 ** svs.h
 **
 ** Copyright 1997 by Kurt Konolige
 **
 ** The author hereby grants to SRI permission to use this software.
 ** The author also grants to SRI permission to distribute this software
 ** to educational institutes for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by SRI, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor SRI make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Kurt Konolige
 ** Senior Computer Scientist
 ** SRI International
 ** 333 Ravenswood Avenue
 ** Menlo Park, CA 94025
 ** E-mail:  konolige@ai.sri.com
 **
 **/

#ifdef __cplusplus
extern "C" {
#endif

#include <stdio.h>

#ifdef UNIX
#include <sys/ipc.h>
#include <sys/shm.h>
#include <string.h>
#define ALIGN8 __attribute__ ((aligned (8)))
#define ALIGN16 __attribute__ ((aligned (16)))
#define FALSE 0
#define TRUE 1
#endif

#ifdef WIN32
#include <windows.h>
#define ALIGN8
#define REALLYIMPORT __declspec(dllimport)
#ifndef LIBCODE
#define IMPORT __declspec(dllimport)
#else
#define IMPORT __declspec(dllexport)
#endif
#else
#define IMPORT
#endif

/* This structure stores the intrinsic parameters of a camera.  The 
 *   first half of this structure deals with parameters recovered
 *   from camera calibration.  The second half allows for the 
 *   parameters to be adjusted for subsampled images and subwindows.
 *   Camera calibration should be done for the largest image size 
 *   possible, and the image subwindow parameters used if smaller 
 *   images are to be used.
 */
typedef struct {
  /* Intrinsic parameters */
  int       pwidth; 	/* [pix]    Width of frame grabber's image */
  int       pheight; 	/* [pix]    Height of frame grabber's image */
  double    dpx; 	/* [mm/pix] X dimension of pixel in frame grabber */
  double    dpy; 	/* [mm/pix] Y dimension of pixel in frame grabber */
  double    sx;		/* []       Scale factor to compensate for any error in dpx   */
  double    Cx;		/* [pix]    Z axis intercept of image plane */
  double    Cy;  	/* [pix]    Z axis intercept of image plane */
  double    f;		/* [mm]     Focal length */
  double    kappa1; 	/* [1/mm^2] First coefficient of radial distortion */
  double    kappa2; 	/* [1/mm^4] Second coefficient of radial distortion */
  float     proj[3][4]; /* projection matrix of rectified image */
  float     rect[3][3]; /* rectification transform after correction for lens distortion */
} svsIP;

//
// The svsSP structure stores all info about a stereo image pair and the stereo
//   calculation.  Used as an argument to most functions.
//

typedef struct {
  int id;

  /* Stereo algorithm parameters */
  int convx, convy;		/* Edge convolution kernel size, pixels */
  int corrxsize, corrysize;	/* Correlation window size, pixels */
  int thresh;			/* Confidence threshold, 0-20 */
  int lr;			/* Left/right check, 1=on, 0=off */
  int ndisp;			/* Number of pixel disparities to search */
  int dpp;                      /* Disparities per pixel */
  int offx, offy;		/* offx is horopter displacement, offy unused */

  /* framegrabber/image parameters absolute parameters */
  int max_linelen;		/* for NTSC fields, 640 */
  int max_lines;		/* for NTSC fields, 240 */
  int max_decimation;		/* 1, 2 or 4 */
  int max_binning;		/* 1 or 2 */
  double gamma;			/* gamma correction needed for display, 1.0 is none */
  int color;			/* 0 for monochrome, 1 for color imager */

  /* digitization parameters */
  int autogain;			/* 1 if available */
  int manualgain;		/* 1 if available */
  int autowhite;		/* 1 if available */
  int manualwhite;		/* 1 if available */

  int gain;			/* Image gain, from 0 to 100; -1 is auto */
  int exposure;			/* Image exposure from 0 to 100, -1 is auto */
  int contrast;			/* Image contrast, from 0 to 100, if auto mode */
  int brightness;		/* Image brightness, from 0 to 100, if auto mode */
  int saturation;		/* Image color saturation, from 0 to 100 */
  int red;			/* Image red gain, from 0 to 100, -1 is auto */
  int blue;			/* Image blue gain, from 0 to 100, -1 is auto */

  /* Image subwindow */
  int ix;			/* Subimage start column */
  int iy;			/* Subimage start row */
  int width;			/* Subimage width, in pixels */
  int height;			/* Subimage height, in pixels */
  int vergence;			/* Subimage vergence between images */
  int rectified;		/* 1 if the image is rectified, else 0 */

  /* warping source subwindow parameters, only used in subwindow warping */
  int subwarp;			/* 1 if we're warping subimage, 0 if not */
  int left_ix, left_iy;		/* source start column and row */
  int left_width, left_height;	/* source width and height */
  int right_ix, right_iy;	/* source start column and row */
  int right_width, right_height; /* source width and height */

  /* Current full-frame image size */
  int decimation;		/* current decimation */
  int binning;			/* current binning */
  int subwindow;		/* 1 = subwindow capability, 0 = none */
  /* Parameters below here are important for redoing warping addresses */
  int linelen;			/* Image line length, in pixels */
  int lines;                    /* Number of image lines */

  /* Internal camera optical parameters */
  int have_rect;		/* 1 if this structure has valid rectification params */
  svsIP  left;			/* Left camera parameters */
  svsIP  right;			/* Right camera parameters */

  /* Transformation between left and right cameras */
  /* Coordinate system is attached to the center of projection of the
     left camera, with the X and Y axis aligned with the u and v axis,
     and Z along the line of sight */
  double Tx;			/* [mm] */
  double Ty;			/* [mm] */
  double Tz;			/* [mm] */
  double Rx;			/* [rad] Yaw */
  double Ry;			/* [rad] Pitch */
  double Rz;			/* [rad] Roll */

  unsigned long reserved;
} svsSP;

IMPORT extern svsSP svsParams;

#define VIDEO_ERROR_FRAME_SIZE   -6
#define VIDEO_ERROR_FRAME_SAMPLE -2
#define VIDEO_ERROR_DEC_MODE     -3
#define VIDEO_ERROR_BIN_MODE     -4
#define VIDEO_ERROR_SUBWINDOW    -5


//#define LINE_SIZE 160
//#define FRAME_SIZE 120

/*
 * Maximum sizes for caches
 */
#define MAXX 1280		/* maximum number of columns in image */
#define MAXY 1032		/* maximum number of rows in image */
#define MAXD 16			/* stereo search range */
#define MAXDS 128		/* abs max search range */
#define MAXW 13			/* max window size */
#define MAXRVAL 2048		/* maximum range value returned */
#define SUBPIXELS 16		/* subpixel interpolation size */

/* size of video, filter kernels */
IMPORT extern int svsConvX, svsConvY, svsCorrX, svsCorrY, svsNumDisp;

#define VID_ROWS 240
#define VID_COLS 320
#define XKERNEL 9
#define YKERNEL 5
#define XCORR  11
#define YCORR  11
#define NUM_DISP 16

/* These macros give offsets and size of the disparity image, relative
 *   to the left intensity image.  Takes svsSP pointer as the argument. 
 *   Offsets are in terms of the left image.
 */
#define ABS(x) (((x) < 0) ? (-(x)) : (x))
#define RESWIDTH(p) ((p)->width - (p)->convx - (p)->corrxsize - 2*(p)->ndisp - \
                     ABS((p)->offx) + 4)
#define RESLEN(p)   ((p)->height - (p)->convy - (p)->corrysize - ABS((p)->offy) + 2)
#define RESLEFT(p)  (((p)->convx +(p)->corrxsize + (p)->ndisp + (p)->ndisp - 4) /2 \
                     + ((p)->offx < 0 ? -(p)->offx : 0))
#define RESTOP(p)   (((p)->convy + (p)->corrysize - 2) / 2 + \
		     ((p)->offy < 0 ? -(p)->offy : 0))

/* These macros give offsets and size of the edge image, relative
 *   to the left intensity image.  Takes svsSP pointer as the argument. 
 */
#define EDGEWIDTH(p) (((p)->linelen) - (p)->convx + 1)
#define EDGELEN(p)   (((p)->height) - (p)->convy + 1)
#define EDGELEFT(p)  ((p)->convx/2)
#define EDGETOP(p)   ((p)->convy/2)

/* video input */
#define IN_STORAGE	4
#define IN_LIVE		5

/* display mode */
#define DISP_NONE   0
#define DISP_LEFT   1
#define DISP_RIGHT  2
#define DISP_BOTH   3
#define DISP_MOSAIC 4
#define DISP_HIST   5
#define DISP_IMAGE  6
#define DISP_X1     7
#define DISP_X2     8

/* acquisition mode */
#define AQ_SINGLE  1
#define AQ_CONTIN  2
#define AQ_FREEZE  11
#define AQ_CALIBRATE 10

/* GM:acquisition type */
#define AQT_SVM   3
#define AQT_BUFFER 4
#define AQT_FRAMEGRAB 5
#define AQT_NONE 6

/* storage modes */
#define STO_ON  1
#define STO_OFF 2
#define STO_CLEAR 3

/* capture formats */
#define CAP_INTERLACE   0
#define CAP_RGCOMPONENT 1
#define CAP_DUAL        2
#define CAP_TRIPLE      3
#define CAP_CHANNELS    4

#define BI_YVU9  0x39555659		// YVU9, planar
#define BI_YV12  0x32315659		// YVU12 (4:1:1), planar
#define BI_I420  0x30323449		// YUV12 (4:2:0), planar 
#define BI_I422  0x32323449		// YUV12 (4:1:1), planar
#define BI_MRG2  0x3247524d		// YUV16 (4:2:2) ??

/*
 * SVS video capture main functions
 */

IMPORT extern char *svsVideoIdent;  // identifier string for this capture interface

IMPORT extern int svsDevice1;	// first video capture device
IMPORT extern int svsDevice2;	// second video capture device
IMPORT extern int svsDeviceName; // name for the default device

IMPORT extern int svsDevice1;	 /* first video capture device */
IMPORT extern int svsDevice2;	 /* second video capture device */
IMPORT extern char *svsDevName;  /* name of first device */
IMPORT extern char *svsDevName2; /* name of second device */
IMPORT extern int  svsIsPXC_PC104; /* set to 1 for PXC PC104+ board under LINUX */

IMPORT extern int svsMeteorMotion; /* set if capturing motion images */
IMPORT extern int svsDualCaptureFormat; /* set if line interlace stereo */
IMPORT extern int svsSwapInterlace; /* 1 to swap L/R interlaced images */
IMPORT extern int svsVideoFormat;	/* fourcc code for video stream type */
IMPORT extern int svsVideoBPP;	/* number of bits per pixel, only for RGB */
/* opens capture for business: */
IMPORT int  svsVideoOpen(svsSP *sp);/* returns -1 if error, 0 if ok */
IMPORT void svsVideoClose(void); /* closes capture */
/* waits for next continuous image from meteors: */
IMPORT int  svsVideoWaitImage(int ms); 
IMPORT int  svsVideoCheckImage(void); /* Returns 1 if an image is ready */
IMPORT void svsVideoGetImage(unsigned char **buf, unsigned char **buf2, 
			     unsigned char **color);
IMPORT void svsVideoGetRectImage(unsigned char **left, unsigned char **right, 
				 unsigned char **color, svsSP *sp);
IMPORT int  svsVideoCheckParameters(svsSP *sp);	/* returns -1 on error, 0 if ok */
IMPORT int  svsVideoStartContinuous(svsSP *sp);	/* ditto */
IMPORT void svsVideoHaltContinuous(void);
IMPORT int  svsVideoGrabSingle(svsSP *sp, int ms); /* returns -1 on error, 0 if ok ??? */

// Calls to set parameters while streaming video
IMPORT int svsVideoSetDigParams(svsSP *sp);/* -1  on error, 0 if ok */
IMPORT int svsVideoSetOffsets(svsSP *sp); /* ditto */

// Imager name (not implemented on analog stereo heads)
IMPORT int svsVideoGetName(char *name);
IMPORT int svsVideoSetName(char *name);

// Debugging callbacks
IMPORT extern void (*capDebug)(char *str);
IMPORT extern void (*svsDebug)(char *str);
IMPORT void svsDebugMessage(char *str, ...);


/* 
 * Sometimes we need to say which image we want
 */

#define svsLEFT  1
#define svsRIGHT 2
#define svsBOTH  3
#define svsLEFTCOLOR 100
#define svsLEFTBAYER 102
#define svsRIGHTBAYER 103

// added for CAP_TRIPLE
//
#define svsLEFT1  4
#define svsLEFT2  5
#define svsLEFT3  6
#define svsRIGHT1 7
#define svsRIGHT2 8
#define svsRIGHT3 9

/*
 * display output
 */
#define svsNONE       0
#define svsSTEREO     1
#define svsDISPARITY  1
#define svsCONFIDENCE 2
#define svsEDGES      3
#define svsMOTION     4
#define svsCOLOR      5
#define svsRGB24      5
#define svsMONOCHROME 6

/*
 * Calculations
 */

/* MMX definition */
#ifdef IMPORT_HASMMX
REALLYIMPORT
#else
IMPORT 
#endif
extern int svsHasMMX;	/* 1 for MMX, 0 for none, -1 for not checked. 
				   Can be set or read */
IMPORT int svsCheckMMX(void); /* Checks for MMX and sets svsHasMMX */

/* 
 * Puts stereo results in dest, using il and ir images.
 * Uses stereo params structure; if null, defaults to svsParams.
 */
IMPORT void svsWarpImage(unsigned char *dest, unsigned char *src, int which, svsSP *sp);
IMPORT void svsWarpImageBayer(unsigned char *dest, unsigned char *dest2, unsigned char *src, 
			      int which, svsSP *sp);
IMPORT void svsWarpImageMono (unsigned char *dest, unsigned char *src, 
			      int which, svsSP *sp);
IMPORT void svsCalcStereo(short *dest, unsigned char *il, 
			  unsigned char *ir, svsSP *sp);
IMPORT void svsCalcEdges(signed char *ld, signed char *rd, unsigned char *il, 
			 unsigned char *ir, svsSP *sp);
IMPORT void svsCalcConfidence(char *dest, unsigned char *il, svsSP *sp);
IMPORT void svsCalcMotion(unsigned char *dest, unsigned char *il, 
			  unsigned char *ir, int xs, int ys, svsSP *sp);


// 3D reconstruction functions
//
IMPORT void     svsReconstruct3D (float *X, float *Y, float *Z, float x, float y, float disp, svsSP *sp);
IMPORT void svsReconstruct3DFast (float *X, float *Y, float *Z, float x, float y, float disp, svsSP *sp);

#define LRVAL     0xfe		/* failed LR check */
#define THRESHVAL 0xff		/* failed texture threshold */

/*
 * Display functions
 */

IMPORT extern int svsStereoOffsetX, svsStereoOffsetY; /* right image offsets */
IMPORT extern float svsRangeImageValue(short *im, svsSP *sp);
IMPORT extern void svsSleep(int ms); /* wait this long in milliseconds */
IMPORT extern int svsSharedMemory;	/* if we can use shared memory, really only Unix */

#ifdef UNIX
#endif

#ifdef WIN32
IMPORT extern HWND svsMainWin;		/* main window handle */
#endif

/*
 * image file I/O
 */

#define HEADER_STRING "srisunim"

#define HEADER_TYPE 0
#define DATA_INT 0
#define DATA_FLOAT 1
#define HEADER_SIZE 256

/* whole structure should take 256 bytes */
struct  srisunim_header
{ 
    char srisunim[8];
    char header_type;
    char data_type;
    char bits_per_pixel;
    char n_dimensions;
    unsigned char x_dimension_h;
    unsigned char x_dimension_l;
    unsigned char y_dimension_h;
    unsigned char y_dimension_l;
    unsigned char z_dimension_h;
    unsigned char z_dimension_l;
    unsigned char n_images;	/* number of images */
    char padding[237];
  };
    
/* Reads an .ssi file and put contents into buffer.  Creates
   buffer if null. */
IMPORT unsigned char *svsReadFile(char *name, unsigned char *buffer, 
			   int *height, int *width, int *num);
IMPORT int svsWriteImage(char *name, unsigned char *buf, int width, 
			 int height, char *type, int numframes);
IMPORT int svsReadFileSizeBMP(char *name, int *width, int *height);
IMPORT int svsReadFileBMP(char *name, unsigned char *buffer, 
			   int *width, int *height, int interlace);
IMPORT unsigned char *svsReadFile2BMP(char *name, unsigned char *buffer, 
			   int *width, int *height);
IMPORT int svsWriteImageBMP(char *name, unsigned char *im, int width, int height, 
			     int isDisparity);
IMPORT int svsWriteImageColorBMP(char *name, unsigned char *im, int width, int height, 
				 int isDisparity);
IMPORT int
svsWrite3DArray(char *fname, float *X, float *Y, float *Z, unsigned char *image, 
		int type, svsSP *sp);
IMPORT int
svsWrite3DCloud(char *fname, float *X, float *Y, float *Z, unsigned char *image, 
		int type, svsSP *sp, int nump);


IMPORT extern char *svsError;
IMPORT int svsWriteParamFile(char *fname, svsSP *sp);	/* writes out stereo setting */
IMPORT int svsWriteRectFile(char *fname, svsSP *sp);	/* writes out stereo rectification */
IMPORT int svsReadParamFile(char *fname, svsSP *sp);	/* reads in stereo setting */



/* version of the software */
#define engine_version 21
IMPORT extern int svsEngineVersion;

IMPORT extern int svsPreFilterTime, svsPostFilterTime, svsCorrTime, svsMaxTime;

//
// new FLTK-based drawing fns
//

IMPORT void svsDrawImage(void *w, unsigned char *im, svsSP *p, int type);
IMPORT void *svsGetWindow(int w, int h);
IMPORT int  fltk_check(void);	/* process fltk events */

#ifdef __cplusplus
}
#endif

#endif    /* ifdef _svs_h_ */
