/*                                                                -*-c++-*-
    Copyright (C) 1991-1997 Gregory D. Hager (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//-----------------------------------------------------------------------------
//
//  Video.hh
//
//  Video, MonoVideo, and ColorVideo classes declared.
//
//  06/06/93 Sidd Puri
//  07/06/94 Greg Hager  (Complete revision)
//  11/11/95 Greg Hager  Added direct video mapping capabilities
//  02/29/96 Kentaro Toyama  Added dynamic RGB grab function modifiability
//  07/17/96 Greg Hager
//    Some more cleanup of RGB capabilities and added explicit color 
//    video support.  Moved to Image class-based grabs.   Put more
//    default functionality in the class so that most standard
//    framegrabbers are more easily support.
//    Added an argument which is the number of types needed for a pixel.
//  02/05/97 Kentaro Toyama  Comments added
//  09/04/98 Greg Hager added a little infrastructure to allow
//  for synchronization with framegrabber
//
//-----------------------------------------------------------------------------

#ifndef Video_h
#define Video_h

#include "Tracker.hh"
#include "Tools.hh"
#include "Image.hh"
#include "XVColors.hh"
#include "XVColorSchemes.hh"


#include <math.h>

enum Device_output_type {

  // Monochrome image types

  PIX_INTENSITY  = 0,
  PIX_HUE        = 1,
  PIX_SATURATION = 2,
  PIX_RED        = 3,
  PIX_GREEN      = 4,
  PIX_BLUE       = 5,
  PIX_LUMINANCE  = 6,

  // Color image types and others

  PIX_PACKED = 7,
  PIX_COLOR = 8,
  PIX_OVERLAY = 9,
  PIX_OTHER1 = 10,
  PIX_OTHER2 = 11,
  PIX_OTHER3 = 12,
  PIX_HSI_PACKED = 13,
  };

int const device_noutput_types = 14;
int const device_noutput_mono_types = 7;

//-----------------------------------------------------------------------------
//  Class : Video
//
//* The Video class is an abstract base class for video device interfacing.
//* It provides an interface between tracking software and video hardware.
//*
//* Video-derived classes fulfill the following broad functions:
//*   Initialization and closing of hardware devices.
//*   Transfers (grabbing) of data from hardware device into memory,
//*     or, if the device can be memory mapped, access to pixel data.
//*   Displays of lines, etc., on the device monitor (in some instances,
//*     the user terminal may function as the device monitor).  
//*
//* For purposes of porting XVision to a new hardware device, a class that
//* instantiates these functions must be defined.  In many cases, the default
//* Video class methods will be sufficient, although code at this level should
//* be optimized for the specific device for best performance.  The easiest
//* way to create a new class is to model it on one of the many classes
//* in the Devices directory.  Most non-virtual member functions will 
//* probably not need to be rewritten in derived classes.
//*
//* The classes MonoVideo and ColorVideo are derived from the Video class,
//* and most instantiations should be derived from one of these (see comments
//* for MonoVideo and ColorVideo classes). 
//*
//*
//* Video class objects have an output type associated with them, which are
//* one of the Device_output_type's enumerated in Video.hh.  This allows
//* one to switch (even during execution) between returning RGB data or 
//* monochrome intensity data, for example.
//*
//* Many video devices are memory mapped in some fashion.  If so, it's
//* sometimes advantageous to bypass the usual memory acquisition routines
//* and perform inline processing or dump memory straight to the display
//* device without copying.
//*
//* Video classes grab windows, which are rectangular, and have the following
//* parameters: position (x,y) in pixels, orientation in radians, and 
//* subsampling rates (sampw,samph) in pixels.  The conventions are that the
//* position coordinates indicate the center of the window, and that the 
//* 0 orientation has window sides aligned with the camera image.  Orientation
//* is also measured to coincide with image coordinates (i.e., (0,0) at the 
//* top left and increasing Y values going down).  Thus, a window that is 
//* tilted counterclockwise on the monitor (or in the image), would have
//* NEGATIVE angular orientation.  This is the same convention that is used
//* in the rest of the XVision system. 
//-----------------------------------------------------------------------------

class Video {

protected:
  
  // These are various values that must be filled in to characterize
  // the video device
  
  int _bytesperpixel; // How many bytes in a single pixel (no default)
  int _is_mappable;   // If the user is allowed to directly access frame memory
                      // (default is no)
  
  int ncols;          // image width in pixels
  int nrows;          // image height in pixels

  int fg_ncols;       // framebuffer width (defaults to ncols)
  int fg_nrows;       // framebuffer height (defaults to nrows)



  int compute_envelope; // turns envelope computation on or off


  int synchronous;      /* Whether to try to make the device
			   synchronize */
  int images;           /* Number of images acquired from device
			   used why trying to synchronize */

  //* This is a pure virtual function.  Instantiate this to deal with 
  //* mapping default colors listed in Tracker.hh as appropriate
  //* for the monitor display.

  virtual int map_color(Color colin) = 0; 

  //* In some cases, you can't tell the size until you open the device
  //* This allows you to set the size later in the constructor.
  void set_size(int nrows_in,int ncols_in);

  //* Set the size of the framegrabber, which may not be the same as the
  //* size of the camera image.
  void set_fg_size(int nrows_in,int ncols_in);

  // Input and output types.  For monochrome cameras, these are fixed.

  Device_output_type output_type;   // What the user sees
  Device_output_type input_type;    // What the device is doing

  //* Allows setting the input type.  By default, this is monochrome
  //* for a monochrome device, but in some cases (e.g. the IndyCam),
  //* you have to convert from color to monochrome in software, so
  //* you need to actually set this input type.
  Device_output_type set_input_type(Device_output_type ot);
  
  //* Change the data presentation as requested.  Unfortunately, for 
  //* upwards compatibility we need two versions.  
  void map_image(int *x, int npixels);
  void map_image(Image &x);

  //* Used to compute the location of the first pixel in an aligned grab.
  int adjust_to_first_coord(float &x,int width);
  int adjust_to_first_coord(int x,int width);



  
public:
  int frames;           /* number frames grabed (used for communication 
			   with XWindow class */
  int envelope_x;     
  int envelope_y;       // top left corner of envelope
  int envelope_width;
  int envelope_height;  // these will hold the dimensions of the envelope 
  int x1;
  int y1;               // left upper corner of window 
  int x2;
  int y2;                // right upper corner of window 
  int x3;
  int y3; 
  int x4;
  int y4; /* hold the geometry of the actual grab window.
	     if the window is aligned, computation is spared by setting x1 < 0 */


  virtual int compute_envelope_parameters_on_grab()
  {
    int bla = compute_envelope;
    compute_envelope = 1;
    return bla;
  };

  virtual int set_synchronous(int val = 1)
  {
    int bla = synchronous;
    synchronous = val;
    return bla;
  };
  
  src_color_scheme color_scheme;
  
  //* Basic bookeeping and info transfer.  The only required
  //* argument is the number of bytes per pixel in this device (usually 1).
  //* The other arguments indicate image size: width and height in pixels.  
  //*
  //* The constructor should either initialize the video device or call
  //* a function which does the initialization.  Initialization should 
  //* include hardware initialization, hardware parameter settings, 
  //* memory mapping (if available), and so on.  Example constructors
  //* in specific Device classes in the Devices directory may help, but
  //* generally, this function will be unique to the hardware and its drivers.
  Video(int nbytes, int nc_in = 640, int nr_in = 480); 

  //* Default is simply a call to Video::close().  If any memory was allocated
  //* in the constructor, it should also be deleted here.  
  inline virtual ~Video();

  //* Sometimes, it's nice to be able to dump a description of the device.  
  //* Define this function with a descriptive string when instantiating.
  virtual char *device_info();

  //* Allow switching of type of data delivered by the device.
  //* By default, calling this generates an error, hence for 
  //* a monochrome device, trying to set the data type will cause
  //* an error.  
  virtual void set_grab_type(Device_output_type ot);

  //* For determining the current output data type.
  Device_output_type output_data_type() const; 

  //* This functions are here to handle any special-purpose buffer
  //* management or whatever needed before acquiring images.
  //* The return value can be used to, for example, determine
  //* whether a DMA buffer was successfully attached.
  virtual int pre_grab();

  //* Same as pre_grab, but for processing afterwards.
  virtual int post_grab();

  //* Clean up function for Video class.  By default it does nothing.
  inline virtual void close();

  //* Return image width.  
  int width();

  //* Return image height.  
  int height();

  //* This is the function which gives the pointer to the current
  //* base of memory.  Up to the instantiator to fill it in.
  virtual void* current_frame_ptr();

  // The following routines allow direct framegrabber access, through
  // memory mapping.  Anything that uses them should check direct_memptr 
  // for a NULL return, indicating direct mapping is not enabled.
  // ONLY INSTANTIATE IF DATA BUFFERS ARE CONTIGUOUS AND CAN BE LOCKED!
  
  //* Returns the value of _is_mappable, which should be TRUE for devices
  //* that have memory mapping capability.
  virtual int is_mappable() const;

  //* Returns the pointer to framebuffer memory if device is memory mappable,
  //* otherwise returns NULL.
  virtual void* direct_memptr();

  //* Returns bytes per pixel supplied by device.
  virtual int bytesperpixel() const;

  //* Forces the sequence forward by one step.
  //* Should is called explicitly in pre_grab and is used
  //* by synchronize

  virtual int forward() {
    cerr << "Forward not defined for synchronous operation on " <<endl;
    return 0;
  }

  int check_sequence(int framenr) {
    if (synchronous) {
      if (framenr > images) {
	forward();
	images ++;
      }
    }
    return images;
  }

  // Various ways of checking if a window is within the acquirable range.

  //* Checks if a window is completely within the boundaries of the full image.
  //* This function is overloaded to allow several argument formats:
  //*   (centerx, centery, width, height),
  //*   (centerx, centery, width, height, orientation),
  //*   (Image class object, centerx, centery),
  //*   (Image class object, centerx, centery, orientation),
  //*   in_view2 takes (leftx, topy, rightx, bottomy).
  int in_view(int cx, int cy, int w, int h);
  int in_view(int cx, int cy, int w, int h, float angle);
  int in_view(const Image &x,int cx, int cy);
  int in_view(const Image &x,int cx, int cy, float angle);
  int in_view2(int x1, int y1, int x2, int y2);

  //* General purpose grab functions, allowing grabbing of windows at
  //* any orientation and any subsampling rate.
  //* Acquires a rectangle of data at an angle, and inserts it in row-major
  //* format into an array or an Image class object.
  //* Defaults are 8 bit mode oriented with coordinate axis.
  //* This version does a pixel-to-pixel mapping between framegrabber
  //* and buffer, even if this causes the area covered by the window
  //* to grow and shrink, depending on the orientation.  
  //* x and y are the center of the window.
  virtual int grab(int *image, int x, int y, int width, int height,
		   float angle = 0.0, int sampw =1 , int samph =1,
		   short int mask = 0xff);

  //  virtual int grab(Image &image, int x, int y, float angle = 0.0, 
  virtual int grab(XVImage<int> &image, int x, int y, float angle = 0.0, 
		   int sampw =1 , int samph =1, short int mask = 0xff); 

  //* Acquire a rectangle of the image oriented with the coordinate
  //* axes.  By default, this just calls grab, but in many cases its
  //* possible to optimize acquisition for this case, so we have it here.
  //* Also, note the use of floats for the center.  This makes it possible
  //* to have even-sized windows and still get what you'd expect.  The
  //* actual coordinates that are acquired are returned in the arguments.

  virtual int grab_aligned(int *image, float &x, float &y,
			   int width, int height,
			   int sampw =1, int samph =1,
			   short int mask = 0xff);
  virtual int grab_aligned(int *image, int x, int y,
			   int width, int height,
			   int sampw =1, int samph =1,
			   short int mask = 0xff);


  //  virtual int grab_aligned(Image &image, float &x, float &y,
  virtual int grab_aligned(XVImage<int> &image, float &x, float &y,
			   int sampw =1, int samph =1,
			   short int mask = 0xff);
  //  virtual int grab_aligned(Image &image, int x, int y,
  virtual int grab_aligned(XVImage<int> &image, int x, int y,
			   int sampw =1, int samph =1,
			   short int mask = 0xff);


  //* These functions are for devices such as VCR's or image sequences 
  //* where, particularly during initialization, it makes sense not to 
  //* forward the sequence.  By default, they act the same as the normal 
  //* grab functions.
  virtual int grab_static(int *image, int x, int y, int width, int height,
			  float angle = 0.0, int sampw =1 , int samph =1,
			  short int mask = 0xff);

  //  virtual int grab_static(Image &image, int x, int y, float angle = 0.0, 
  virtual int grab_static(XVImage<int> &image, int x, int y, float angle = 0.0, 
		   int sampw =1 , int samph =1, short int mask = 0xff); 

  //* Grab a region using upper-left/lower-right parameters.
  virtual int grab_region(int *image, int lx, int ly, int ux, int uy,
			  int sampw =1, int samph = 1);

  // Changed to XVImage<int> to make more generic --- GDH  virtual int grab_region(Image &image, int lx, int ly,
  virtual int grab_region(XVImage<int> &image, int lx, int ly,
			  int sampw =1, int samph = 1);

  // Display functions ---------------------------------------------------

  //* This is a pure virtual function.  Derived class methods should 
  //* show an image on the video screen at the specified location
  //* with the specified height and width.
  //* Note that (x,y) are the coordinates of the upper left, rather
  //* than the center (though the latter would be more consistent
  //* with the grab functions
  virtual int show(int x, int y, int width, int height,	
		    int *image) = 0;

  virtual int show(Image &im, int x=100, int y=100);

  //* Video::line is a pure virtual function.  Derived class methods should
  //* draw a line on the monitor with specified center (x,y), length, and 
  //* angle.
  virtual void line(float x, float y, int length, float angle,
		     Color color = default_color,int samp = 1) = 0;

  //* Erases a line by drawing a transparent line.  For some monitor devices,
  //* (particularly those without an overlay plane) it may not be necessary 
  //* to actively erase anything, since the screen may refresh often, 
  //* effectively erasing drawn lines.
  virtual void clearline(float x, float y, int length, float angle, 
			  int samp = 1);

  //* Same as line, but inputs are endpoint coordinates.  Default does a 
  //* conversion and call to line.
  virtual void line2(float x1, float y1, float x2, float y2, 
		      Color color = default_color);

  //* Default calls line2 with transparent color.
  virtual void clearline2(float x1, float y1, float x2, float y2);

  // Draw and clear a circle on the video screen

  //* Default draws a crosshair with input center and radius.
  virtual void circle(float x, float y, int r, 
		       Color color = default_color);
  //* Default calls circle with transparent color.
  virtual void clearcircle(float x, float y,  int r);

  // Draw and clear a cubic-polynomial segment

  //* Default draws nothing.  Implemented version should draw a cubic
  //* curve segment with starting point (x,y), at orientation "angle",
  //* length "length," and coefficents a, b, and c, where:
  //*   y = a x^3 + b x^2 + c x + d.
  //* d is implicit in the fact that the starting point is y.  The angular
  //* orientation should indicate the orientation of the x-axis.
  virtual void cubic(float x, float y, int length, float angle,
		      float a, float b, float c, 
		      Color color = default_color);

  //* Default does nothing.
  virtual void clearcubic(float x, float y, int length, float angle,
			  float a, float b, float c);

};


//  Class : MonoVideo
//
//* MonoVideo is derived from Video.
//*
//* The MonoVideo class supports acquisition of eight bit monochrome images.
//* If the underlying device is monochrome, then the only possible output
//* is what the device supplies.  If the underlying device is color, it is
//* also possible to get other projections (hue, saturation, r, g, b) of the
//* color image.  By default luminance (band averaged) data is returned.

class MonoVideo : public Video {

public:

  //* Input and output data types set to PIX_LUMINANCE by default.
  MonoVideo(int nbytes = 1, int nc_in = 640, int nr_in = 480);
  ~MonoVideo() {}

  //* Allow switching of type of data delivered by the device, provided it is
  //* a legal monochrome type (See Device_output_type enumeration in Video.hh).
  virtual void set_grab_type(Device_output_type ot);

};

//  Class : ColorVideo
//  
//* ColorVideo is derived from Video.
//* 
//* The ColorVideo class adds full colormapping support to the
//* video class.  In general, for hardware with color, it is better to
//* instantiate both a color class and a monochrome class which
//* puts the device on mono mode and keeps it there.  This allows
//* for better static type checking (instead of dynamic errors).
class ColorVideo : public Video {

public:

  //* Input and output data types set to PIX_PACKED by default.
  ColorVideo(int nbytes = 4, int nc_in = 640, int nr_in = 480);
  ~ColorVideo() {;}
  
  //* Allow switching of type of data delivered by the device.
  virtual void set_grab_type(Device_output_type ot);
  
};

// Include inline member functions.
#include "Video.icc"    

//-----------------------------------------------------------------------------
#endif


