/*                                                                -*-c++-*-
    Copyright (C) 1996 Gregory D. Hager (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//---------------------------------------------------------------------------
//
//  Video.cc
//
//  Definition of Video, MonoVideo, and ColorVideo member functions.
//
//  07/17/96 Gregory Hager
//
//---------------------------------------------------------------------------

#include "Video.hh"
#include "Acquire.hh"

// include this to get all of the acquisition templates instantiated

#include "Acquire.cc"
#include "Drawing.cc"

// inline functions

#include "RGB2int.cc"

//-----------------------------------------------------------------------
// Video class methods.
//-----------------------------------------------------------------------

// protected: -----------------------------------------------------------

void
Video::map_image(int *image, int npixels)
{
  int *endimage = image + npixels;
  
  // If there's nothing to do, get out of here.
  
  if (input_type == output_type) return;

  // Right now, we really only know how to convert packed data to
  // other representations.

  switch (input_type) {

  case PIX_INTENSITY:
    panic("Cannot change intensity data to anything else.");
    break;
    
  case PIX_PACKED:
    switch(output_type) {
      
    case PIX_RED:
      for(;image<endimage;image++) *image = compute_red(*image);
      break;

    case PIX_GREEN:
      for(;image<endimage;image++) *image = compute_green(*image);
      break;

    case PIX_BLUE:
      for(;image<endimage;image++) *image = compute_blue(*image);
      break;
    
    case PIX_LUMINANCE: 
      for(;image<endimage;image++) *image = compute_luminance(*image);
      break;

    case PIX_HSI_PACKED: 
      for(;image<endimage;image++) *image = compute_hsi_packed(*image);
      break;

    case PIX_PACKED:
      break;

    case PIX_SATURATION: 
      for(;image<endimage;image++) *image = compute_saturation(*image);
      break;

    case PIX_HUE: 
      for(;image<endimage;image++) *image = compute_hue(*image);
    break;

    case PIX_INTENSITY:
      for(;image<endimage;image++) *image = compute_intensity(*image);
    break;

    default:
      panic("Don't know how to create desired output");
    }
    break;

  default:
    panic("Don't know how to process desired input");
  }
}

// public: ---------------------------------------------------------------

// Constructor -----------------------------------------------------------

Video::Video (int nbytes, int nc_in, int nr_in) : 
_bytesperpixel(nbytes),
_is_mappable(0), 
ncols(nc_in), nrows(nr_in),
fg_ncols(nc_in), 
fg_nrows(nr_in),
compute_envelope(0),
synchronous(0),
images(0),
// for now lets make this the default -- its not that expensive
output_type(PIX_INTENSITY),
input_type(PIX_INTENSITY)
{
  // must be redefined!
  color_scheme = undefined_src_color_scheme;
};

// Functions needed for grabbing -----------------------------------------

template <class T>
inline void
internal_grab(int *image, T *mem,
     int x, int y, int width, int height,
     float angle,
     int sampw , int samph,
     short int mask,int ncols)
{
  if (sampw == 1) {
    if (samph == 1)
      bham_rectangle(mem,image,x,y,angle,height,width,mask,ncols);
    else
      bham_rectangle_msh(mem,image,x,y,angle,height,width,samph,mask,ncols);
  }
  else
    if (samph == 1)
      bham_rectangle_msw(mem,image,x,y,angle,height,width,sampw,mask,ncols);
    else
      bham_rectangle_ms(mem,image,x,y,angle,height,width,
			samph,sampw,mask,ncols);

}

template <class T>
inline void
internal_grab_aligned(register int *image, T *mem, int fx, int fy,
		      int width, int height,
		      int sampw, int samph, int ncols) 
{
  int i,rskip;
  register T *ldata, *endldata, *finalldata;

  if ((sizeof(T) == sizeof(int)) && (sampw == 1)){
    // printf("doing memcopy\n\n");
    for (i=0;i<height;i++)
      memcpy(image+i*width ,mem+(fy+i*samph)*ncols+fx,
	     width*(sizeof(int)));
  }
  else {
    width *=sampw;
    rskip  = samph*ncols;
    // printf("doing standard copy at %p \n", mem);
    // printf("to %p \n\n", image);

    ldata = mem + fy*ncols+fx;
    finalldata = ldata + rskip*(height-1) + width;
    rskip -= width;

    for(;ldata<finalldata;ldata += rskip) {
      // printf("%d..%d, %d %d", ldata, finalldata, width, sampw); fflush(stdout);
      for(endldata = ldata+width;ldata< endldata;ldata+=sampw)
	*image++ = *ldata;
    }
  }
}

// Grab Functions --------------------------------------------------------

int Video::grab(int *image, int x, int y, int width, int height, float angle,
	    int sampw, int samph, short int mask)
{

  if (angle == 0.0) 
     return grab_aligned(image,x,y,width,height,sampw,samph,mask);

  if (!pre_grab())
    return 0;

  if (!(in_view(x,y,width*sampw,height*samph,angle)))
    return 0;

  // Set up constants and map the desired window into the framebuffer
  // memory access window.  This scheme assumes that windows are
  // relatively square.

  angle = -angle;            

  switch(bytesperpixel()) {
  case  1: 
    internal_grab(image,(unsigned char *)current_frame_ptr(),
		  x,y,width,height,angle,sampw,samph,mask,ncols);
    break;

  case  2: 
    internal_grab(image,(unsigned short *)current_frame_ptr(),
		  x,y,width,height,angle,sampw,samph,mask,ncols);
    break;

  case  4: 
    internal_grab(image,(unsigned int *)current_frame_ptr(),
		  x,y,width,height,angle,sampw,samph,mask,ncols);
    break;
  default:
    panic("Unknown image size in grab_aligned");
  }

  map_image(image,width*height);

  if (compute_envelope)
    {
      // ToDo: doesn't this call sin and cos a lot? ADR
      envelope_height =  int(abs(height * cos(angle)));
      envelope_height += (x2 = int(abs(width * sin(angle))));
      envelope_width  =  int(abs(height * sin(angle)));
      envelope_width  += (y1 = int(abs(width * cos(angle))));
      envelope_width  *= sampw;
      envelope_height *= samph;
      
      envelope_x = x;      
      envelope_y = y;
      
      /* not working yet... 
	 x1 = envelope_x - envelope_width/2
	 x3 = x1         + envelope_width;
	 y2 = envelope_y - envelope_height/2
	 y4 = y2 + envelope_height;
	 y3 = y4 - y1;
	 y1 += y2;
	 x4 = x3 - x1;
	 x2 += x1;      
      */
      frames++;
    };      
  return post_grab();
}

//* The reason for the references here is that grab aligned
//* must choose an integer coordinate to grab this thing
//* at; the reference is used to communicate that back to the
//* caller.  There are several cases where the call should 
//* know and account for this change.  

int Video::grab_aligned (int *image, float &x, float  &y,
			 int width, int height,
			 int sampw, int samph, short int mask) 
{
  int i,j;
  int fx, fy;

  // adjust_to_first_coord should adjust x and y to the
  // "real" coordinates of acquired image.  fx and fy
  // are the left and uppermost coordinates of region, respectively.

  fx = adjust_to_first_coord(x,width*sampw);
  fy = adjust_to_first_coord(y,height*samph);

  if (!(in_view2(fx,fy,fx+width*sampw-1,fy+height*samph-1)))
    return 0;

  if (!pre_grab())
    return 0;

  switch(bytesperpixel()) {
  case  1:
    // cout << "grab_aligned 1 byteperpixel" << endl;
    internal_grab_aligned(image,(unsigned char *)current_frame_ptr(),
			 fx,fy,width,height,sampw,samph,ncols);
    break;

  case  2: 
    // cout << "grab_aligned 2 byteperpixel" << endl;
    internal_grab_aligned(image,(unsigned short *)current_frame_ptr(),
			 fx,fy,width,height,sampw,samph,ncols);
    break;

  case  4: 
    // cout << "grab_aligned 4 byteperpixel" << endl;
    internal_grab_aligned(image,(unsigned int *)current_frame_ptr(),
			 fx,fy,width,height,sampw,samph,ncols);
    break;
  default:
    panic("Unknown image size in grab_aligned");
  }

  map_image(image,width*height);
  
  if (compute_envelope)
    {
      x1 = -1;
      envelope_x = fx;
      envelope_y = fy;
      envelope_width = width*sampw;
      envelope_height = height*samph;
      frames++;
    };

  return post_grab();
}

//* This is an "I don't want to know" version of the
//* former --- question is, should be have this 
//* ambiguity at all?

int Video::grab_aligned (int *image, int x, int y,
			 int width, int height,
			 int sampw, int samph, short int mask) 

{

  float rx = x;
  float ry = y;
  return grab_aligned (image, rx,ry,
		       width,  height,  sampw,  samph,  mask);
}



//-----------------------------------------------------------------------
// MonoVideo and ColorVideo class methods.
//-----------------------------------------------------------------------

MonoVideo::MonoVideo (int nbytes, int nc_in, int nr_in) : 
Video(nbytes,nc_in,nr_in) 
{
      // Defaults
      output_type = PIX_LUMINANCE;
      input_type  = PIX_LUMINANCE;
}

void
MonoVideo::set_grab_type(Device_output_type ot) 
{
  if ((ot == input_type) || 
      ((input_type >= device_noutput_mono_types) &&
       (ot <  device_noutput_mono_types)))
    output_type = ot;
  else
    panic ("Attempt to set monochrome device to output which cannot be supported.\n");
}

//-----------------------------------------------------------------------

ColorVideo::ColorVideo (int nbytes, int nc_in, int nr_in) : 
Video(nbytes,nc_in,nr_in) 
{
      // Defaults
      output_type = PIX_PACKED;
      input_type  = PIX_PACKED;
}

void
ColorVideo::set_grab_type(Device_output_type ot) {
  output_type = ot;
}

//-----------------------------------------------------------------------
