/*                                                                -*-c++-*-
    Copyright (C) 1995-1997 Gregory D. Hager, Kentaro Toyama (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//-----------------------------------------------------------------------------
//
//  K2T_V300_Mono.cc
//
//  Definition of Video class member functions.
//
//  03/15/95 Kentaro Toyama, after code for IT_FG101 by Sidd Puri.
//
//-----------------------------------------------------------------------------

#include <site_config.h>
#ifdef  USE_K2T_MONO

#include <fcntl.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <stdlib.h>
#include <sys/file.h>
#include <unistd.h>
#ifdef __GNUC__
#include <string.h>
#else
#include <strings.h>
#endif

#include "Acquire.hh"
#include "K2T_V300_Mono.hh"

// This seems to require some special handling

extern "C" int gethostname(char *, int);

//-----------------------------------------------------------------------------
//  Constructor and destructor
//-----------------------------------------------------------------------------

// Creating a K2T_V300_Mono device corresponds to initializing it
// and mapping the framegrabber memory.

K2T_V300_Mono::K2T_V300_Mono (int channel,int camera) : 
MonoVideo(1,KTV_DEFAULT_COLS,KTV_DEFAULT_ROWS)
{

  char buff[80];
  gethostname(buff,80);

  // Do this first to make sure there is a framebuffer attached.
  //  if (!strcmp(buff,_K2TV300_HOST_)) 
  {
    printf ("K2T_V300 framegrabber and camera running on %s.\n",buff);

    vimage = new int[VIMAGE_SIZE*KTV_DEFAULT_ROWS];
  
    // Check to make sure parameters are legal
  
    if ((channel < 0) || (channel > 2))
      panic("Channel number out of range \n",false);
    if ((camera < 0) || (camera > 0))
      panic("Camera number out of range \n",false);
  
    ktv_init_mono(0,KTV_SYNCIN_SYNC);         // framegrabber syncs output
    switch (channel) {
    case 0:
      mem_map = (unsigned int *) KTV_RED;
      break;
    case 1:
      mem_map = (unsigned int *) KTV_GREEN;
      break;
    case 2:
      mem_map = (unsigned int *) KTV_BLUE;
      break;
    }
    
    ktv_clear_ram(); 		              // Clear the display
    ktv_clear_overlay(); 	              // Clear the overlay planes
    
    // Put board into live mode
    ktv_live_mode();
    
    tracker_cleanup (this);
  }
}

void K2T_V300_Mono::close() 
{
}

//-----------------------------------------------------------------------------
//  Grab functions
//-----------------------------------------------------------------------------

// !!! This code really has to be replaced.
int K2T_V300_Mono::grab (int *image, int x, int y, int width, int height, 
		  float angle, 
		  int sampw, int samph,
		  short int mask) 
{

  if (angle == 0.0)
    if ((sampw == 1) && (samph == 1)) 
      return grab_aligned(x,y,width,height,image,mask);
    else
      return grab_aligned_s(x,y,width,height,image,sampw,samph,mask);
  
  int new_width,new_height;

  aligned_envelope(width*sampw,height*samph,angle,&new_width,&new_height);
  if (compute_envelope)
    {
      envelope_height =  new_height;
      envelope_width  =  new_width;      
      envelope_x = x;      
      envelope_y = y;
      
      /* not working yet... 
         x1 = envelope_x - envelope_width/2
         x3 = x1         + envelope_width;
         y2 = envelope_y - envelope_height/2
         y4 = y2 + envelope_height;
         y3 = y4 - y1;
         y1 += y2;
         x4 = x3 - x1;
         x2 += x1;      
      */
      frames++;
    };      

  if (!(in_view(x,y,new_width,new_height)))
    return 0;

  grab_aligned2(x,y,new_width,new_height,vimage,mask);

  // Set up constants and map the desired window into the framebuffer
  // memory access window.  This scheme assumes that windows are
  // relatively square.

  angle = -angle;            
  x += KVM_xspin_offset;			     // set up constants

  if (sampw == 1) {
    if (samph == 1)
      bham_rectangle(vimage,image,x,y,angle,height,width,mask,1024);
    else
      bham_rectangle_msh(vimage,image,x,y,angle,height,width,samph,mask,1024);
  }
  else
    if (samph == 1)
      bham_rectangle_msw(vimage,image,x,y,angle,height,width,sampw,mask,1024);
    else
      bham_rectangle_ms(vimage,image,x,y,angle,height,width,samph,sampw,mask,1024);
  
  return 1;
}

int K2T_V300_Mono::grab_aligned (int x, int y,
				 int width, int height, 
				 int *image, short int mask) 
{
  if (!in_view(x,y,width,height))
    return 0;
  
  x += KVM_xspin_offset;	// transform coordinates to corner
  int ix = x - half (width);
  int iy = y - half (height);

  if (compute_envelope)
    {
      x1 = -1;
      envelope_x = x;
      envelope_y = y;
      envelope_width = width;
      envelope_height = height;
      frames++;
    };

  register int row, col;
  register int word_offset, byte_offset;
  register unsigned int *impixel;
  unsigned int nextg;
  register unsigned int *ktv_pixel;

  impixel = (unsigned int *) image;
  for (row = iy ; row < iy+height ; row++) {           // Handle each row
    col = ix;
    // These are our starting points on the board - pointers to 32 bit words
    ktv_pixel = (unsigned int *) mem_map + col/4 + row*256;
    impixel = (unsigned int *) image + col - ix + (row - iy) * width;
    // First few bytes may be done by hand 
    if (col%4) {
      byte_offset = ix%4;
      nextg = *ktv_pixel++;
      while ( (byte_offset < 4) && (col < ix+width) ) {
        *impixel++ = ((unsigned char *) (&nextg))[byte_offset];
	byte_offset++;   col++;
      }
    }
    // Now we're on the next word boundary
    while (col < (ix+width)-4) {
      nextg = *ktv_pixel++;
      *impixel++ = ((unsigned char *) (&nextg))[0];
      *impixel++ = ((unsigned char *) (&nextg))[1];
      *impixel++ = ((unsigned char *) (&nextg))[2];
      *impixel++ = ((unsigned char *) (&nextg))[3];
      col += 4;
    }
    // Last word is also done "by hand" 
    if (col < ix+width) {
      byte_offset = 0;
      nextg = *ktv_pixel++;
      while ( (byte_offset < 4) && (col < ix+width) ) {
	*impixel++ = ((unsigned char *) (&nextg))[byte_offset];
	byte_offset++;   col++;
      }
    }
  }
  
  return 1;
}

// !!! Are grab_aligned and grab_aligned2 the same?
int K2T_V300_Mono::grab_aligned2 (int x, int y,
				  int width, int height, 
				  int *image, short int mask) 
{
  if (!in_view(x,y,width,height))
    return 0;
  
  x += KVM_xspin_offset;        // transform coordinates to corner
  int ix = x - half (width);
  int iy = y - half (height);

  if (compute_envelope)
    {
      x1 = -1;
      envelope_x = x;
      envelope_y = y;
      envelope_width = width;
      envelope_height = height;
      frames++;
    };

  register int row, col;
  register int word_offset, byte_offset;
  register unsigned int *impixel;
  unsigned int nextg;
  register unsigned int *ktv_pixel;
  
  impixel = (unsigned int *) image;
  for (row = iy ; row < iy+height ; row++) {           // Handle each row 
    col = ix;
    // These are our starting points on the board - pointers to 32 bit words 
    ktv_pixel = (unsigned int *) mem_map + col/4 + row*256;
    impixel = (unsigned int *) image + col + (row) * VIMAGE_SIZE;
    // First few bytes may be done by hand 
    if (col%4) {
      byte_offset = ix%4;
      nextg = *ktv_pixel++;
      while ( (byte_offset < 4) && (col < ix+width) ) {
        *impixel++ = ((unsigned char *) (&nextg))[byte_offset];
	byte_offset++;   col++;
      }
    }
    // Now we're on the next word boundary 
    while (col < (ix+width)-4) {
      nextg = *ktv_pixel++;
      *impixel++ = ((unsigned char *) (&nextg))[0];
      *impixel++ = ((unsigned char *) (&nextg))[1];
      *impixel++ = ((unsigned char *) (&nextg))[2];
      *impixel++ = ((unsigned char *) (&nextg))[3];
      col += 4;
    }
    // Last word is also done "by hand" 
    if (col < ix+width) {
      byte_offset = 0;
      nextg = *ktv_pixel++;
      while ( (byte_offset < 4) && (col < ix+width) ) {
	*impixel++ = ((unsigned char *) (&nextg))[byte_offset];
	byte_offset++;   col++;
      }
    }
  }
  
  return 1;
}

// !!! What's the actual optimum?
int K2T_V300_Mono::grab_aligned_s (int x, int y,
			  int width, int height, 
			  int *image, int wsamp, int lsamp, short int mask) 
{
  if (!in_view(x,y,width,height))
    return 0;

  x += KVM_xspin_offset;	// transform coordinates to corner
  int ix = x - half (width*wsamp);
  int iy = y - half (height*lsamp);

  if (compute_envelope)
    {
      x1 = -1;
      envelope_x = x;
      envelope_y = y;
      envelope_width = width*wsamp;
      envelope_height = height*lsamp;
      frames++;
    };

  register int row, col;
  register int byte_offset;
  register unsigned int *impixel;
  register unsigned int *ktv_pixel;
  unsigned int nextp;

  impixel = (unsigned int *) image;
  // Handle each row 
  for (row = iy ; row < iy+height*lsamp ; row += lsamp) {     
    byte_offset = row%4;
    for (col = ix; col < ix+width*wsamp ; col += wsamp) {
      // All bytes done by hand 
      ktv_pixel = (unsigned int *) mem_map + col/4 + row*256;
      nextp = *ktv_pixel;                      // copy word to local
      *impixel++ = ((unsigned char *) (&nextp))[byte_offset];
    }
  }
  
  return 1;
}

// !!! Can we yank this?
void 
K2T_V300_Mono::grab_constant_size (int *image, float x, float y, 
				   int width, int height, 
				   float angle, short int mask) 
{
  if (angle == 0) {
    grab_aligned(round(x),round(y),width, height, image, mask);
    return;
  } else {
    grab(image, round(x), round(y), width, height, angle, 1, 1, mask);
    return;
  }
}

// !!! Implement or kill.
void K2T_V300_Mono::linegrab (int x, int y, int length,
		      int *image,float angle, short int mask) 
{
  x += KVM_xspin_offset;			     // set up constants
//  bham_line(mem_map,image, x,y,angle,length, mask, 1024);
}

//-----------------------------------------------------------------------------
//  Display functions
//-----------------------------------------------------------------------------

// !!! Needs to be fixed or eliminated.
int K2T_V300_Mono::show (int x, int y,int width, int height, int *image) 
{
  if (!(in_view(x,y,width,height)))
    return 0;
  
  int nextwd;
  
  ktv_digitize_and_wait();

  for (int row = y+height-1; row >= y; row--) {
     for (int col = x+width-1; col >= x; col--) {
       nextwd = *(mem_map + row*256 + col/4);
       ((unsigned char *) (&nextwd))[col%4] = image[(row-y)*width+(col-x)];
       *(mem_map + row*256 + col/4) = nextwd;
     }
   }

  ktv_live_mode();

  return 1;
}

//--------------------- For Drawing Pretty Pictures ------------

// !!! Shouldn't there be a non-pure virtual one of these in the Video class?
void K2T_V300_Mono::point (int x, int y, Color color, int size) 
{

  color = map_color(color);
  
  x += KVM_xspin_offset;

  extern int *KTV_OV;			// Overlay band 

  int nextwd;

  int  ktv_pp_wd_off = y*256 + x/4;
  int  ktv_pp_bt_off = x%4;
  nextwd = *(KTV_OV + ktv_pp_wd_off);
  ((unsigned char *) (&nextwd))[ktv_pp_bt_off] = (color);
  *(KTV_OV + ktv_pp_wd_off) = nextwd;

}

// !!! We need to decide which version to use and stick with it.
void K2T_V300_Mono::line (float x, float y, int length, float angle,
		     Color color, int samp) 
{
  color = map_color(color);
  
  if (in_view(x,y,(int)ceil(fabs(cos(angle))*length),
	          (int)ceil(fabs(sin(angle))*length))) 
    {
      x += KVM_xspin_offset;

      // K2Tdraw_line(int x,int y,float angle,int length,short int color) {
      //   a separate line-drawing function which can be removed if
      //   the more general template function works
      //   doesn't support sampling

      if ((color > 15) || (color < 0)) {color = 2;}  // green is the default

      unsigned char c = (unsigned char)color;
      int nypixels = (int)(fabs(((double)length)*sin(angle)) + 0.5);
      int nxpixels = (int)(fabs(((double)length)*cos(angle)) + 0.5);
      int low,high,alt,dir; double step,total=0.0;
      
      if (nxpixels>nypixels) {           // x-dominant
	dir = ( (tan(angle) >= 0) ? 1 : 0 );
	if (dir) {alt=y-nypixels/2;} else {alt=y+nypixels/2;}
	step = ((double)nypixels)/((double)nxpixels);
	low = x-nxpixels/2;  high = low+nxpixels;
	
	for (int pix=low ; pix<high ; pix++) {
	  ktv_put_overlay_pixel(&c,alt,pix);
	  total += step;
	  if (total>=1.0) {
	    total -= 1.0;
	    if (dir) {alt++;} else {alt--;}
	  }
	}
      } else {                           // y-dominant
	dir = ( (tan(angle) >= 0) ? 1 : 0 );
	if (dir) {alt=x-nxpixels/2;} else {alt=x+nxpixels/2;}
	step = ((double)nxpixels)/((double)nypixels);
	low = y-nypixels/2;  high = low+nypixels;
	
	for (int pix=low ; pix<high ; pix++) {
	  ktv_put_overlay_pixel(&c,pix,alt);
	  total += step;
	  if (total>=1.0) {
	    total -= 1.0;
	    if (dir) {alt++;} else {alt--;}
	  }
	}
      }
  
      // }  end of the ad hoc line-drawing function
      //
      //    This is the old line drawing function -- the code is in
      //      Drawing.cc.template
      //
      //    UNIQNAM(draw_line)((unsigned int *)mem_map,(int)x,(int)y,
      //		       angle,length,(short int)color,samp,1024);
 
  }   // end of   if (in_view ... )
}     // end of   K2T_V300_MONO::line(...)

void K2T_V300_Mono::line2 (float x1, float y1, float x2, float y2, Color color){
  double angle = atan2(y2-y1,x2-x1);
  double length = sqrt (sqr (y2-y1) + sqr (x2-x1));
  line ((x2+x1)/2.0, (y2+y1)/2.0, length, angle, color, 1);
}

void K2T_V300_Mono::circle (float x, float y, int r, Color color) 
{
  line (x,y-r,r*2,0.0,color);
  line (x,y+r,r*2,0.0,color);
  line (x-r,y,r*2,M_PI_2,color);
  line (x+r,y,r*2,M_PI_2,color);
}

void K2T_V300_Mono::cubic (float x, float y, int length, float angle,
		   float a, float b, float c, Color color)
{
}

//-----------------------------------------------------------------------------

#endif // USE_K2T_MONO
