//-----------------------------------------------------------------------------
//
//  Ieee1394.cc
//
//  Definition of Ieee1394 member functions.
//
//  07/06/94 Greg Hager
//  ??/??/96 Greg Hager various improvements.
//
//-----------------------------------------------------------------------------

#include <site_config.h>
#ifdef USE_IEEE1394

#include <stdlib.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/mman.h>

#include "Video.hh"
#include "Acquire.hh"
#include "Ieee1394.hh"

int Ieee1394::fd = -1;
int Ieee1394::scheduled=0;
int Ieee1394::gotfirst = 0;
int Ieee1394::node_id = -1;
int Ieee1394::buffer_id = 0;
int Ieee1394::n_buffers = 0;
int Ieee1394::raw_rgb;
int Ieee1394::brightness=128;
int Ieee1394::scale;
raw1394handle_t Ieee1394::handle;
u_char *Ieee1394::mm_buf[DIG_DEF_NUMFRAMES];
u_char *Ieee1394::image_buffers[DIG_DEF_NUMFRAMES];
int Ieee1394::ninstances = 0;
long win;
ulong *Ieee1394::dataPtr=NULL;
dc1394_cameracapture Ieee1394::camera; 
nodeid_t * Ieee1394::camera_nodes;


#define YUV2RGB(y, u, v, r, g, b)\
  r = y + ((v * 1434) / 2048);\
  g = y - ((u * 406) / 2048) - ((v * 595) / 2048);\
  b = y + ((u * 2078) / 2048);\
  r = r < 0 ? 0 : r;\
  g = g < 0 ? 0 : g;\
  b = b < 0 ? 0 : b;\
  r = r > 255 ? 255 : r;\
  g = g > 255 ? 255 : g;\
  b = b > 255 ? 255 : b

static unsigned char t_r[1<<16];
static unsigned char t_b[1<<16];
static unsigned char t_g1[1<<16];
static unsigned char t_g2[1<<16];

static void
init_tab(void)
{
   int r,b,g1,g2;

   for(int i=0;i<256;i++)
     for(int j=0;j<256;j++)
     {
        r=i + (((j-128) * 1434) / 2048);
        b=i + (((j-128) * 2078) / 2048);
	g1=(((i-128)*406)/2048)+(((j-128)*595)/2048);
	g2=i-j;
	r = r < 0 ? 0 : r;
	g1 = g1 < 0 ? 0 : g1;
	g2 = g2 < 0 ? 0 : g2;
	b = b < 0 ? 0 : b;
	r = r > 255 ? 255 : r;
	g1 = g1 > 255 ? 255 : g1;
	g2 = g2 > 255 ? 255 : g2;
	b = b > 255 ? 255 : b;
	t_r[(i<<8)|j]=r;
	t_b[(i<<8)|j]=b;
	t_g1[(i<<8)|j]=g1;
	t_g2[(i<<8)|j]=g2;
     }
}

/*routine to convert an array of YUV data to RGB format
  based on routine from Bart Nabbe
*/
int Ieee1394::yuv422torgb (unsigned char *YUV, unsigned char *targ, 
				    int NumPixels) {
  int i, j;

    for (i = 0, j = 0; i < 2 * NumPixels; i += 4, j += 6)
    {
      *targ++=t_r[(YUV[i+1]<<8)|YUV[i+2]];
      *targ++=t_g2[(YUV[i+1]<<8)|t_g1[(YUV[i]<<8)|YUV[i+2]]];
      *targ++=t_b[(YUV[i+1]<<8)|YUV[i]];
      *targ++;
      *targ++=t_r[(YUV[i+3]<<8)|YUV[i+2]];
      *targ++=t_g2[(YUV[i+3]<<8)|t_g1[(YUV[i]<<8)|YUV[i+2]]];
      *targ++=t_b[(YUV[i+3]<<8)|YUV[i]];
      *targ++;
    }
  return 0;
}

int Ieee1394::rawtorgb (unsigned char *RGB, u_char *targ, 
				 int NumPixels) {
  int i, j;

    for (i = 0; i < NumPixels; i ++)
    {
      *targ++=*RGB++;
      *targ=*RGB++;
      *targ++=*RGB++;
      *targ;
      
    }
  return 0;
}


int Ieee1394::open(const char *dev_name) {

  if(fd>=0) close(); // close old stuff first ???
  if((fd=::open(dev_name,O_RDWR))<0)
  {
    perror("XVision:");
    exit(1);
  }
  ioctl(fd,VIDEO1394_LISTEN_CHANNEL,&v_mmap);
  mm_buf[0]=(unsigned char *)mmap((void *)0,v_mmap.buf_size*v_mmap.nb_buffers,
                        PROT_READ|PROT_WRITE,MAP_SHARED,fd,0);
  if(mm_buf[0]==(unsigned char *)~0)
  {
        cerr << "couldn't mmap buffers" << endl;
        exit(1);
  }
  //mm_buf[0]+=12;
  for(int i=1;i<v_mmap.nb_buffers;i++)
      mm_buf[i]=mm_buf[0]+i*v_mmap.buf_size;
  for(int i=0;i<v_mmap.nb_buffers;i++)
     image_buffers[i]=new (unsigned char)[width()*height()*4];
  return 1;
}



int Ieee1394::parse_param(char * &paramstring, XVParser &result) {
   char *s_ptr;
   int j;

   switch(sscanf(paramstring,"%c%d%n",&result.c,&result.val,&j))
   {
     case 2:
       paramstring+=j; // set pointer to next parameter
       return 1;
     case 1:	            // no parameters to character
       cerr << "No value parsed for " << result.c << endl;
       return -1;
     default:               // end of string
       break;
   }
   return 0;  // signal end of string
}

int Ieee1394::set_params(char *paramstring) {

   int 		num_frames=DIG_DEF_NUMFRAMES;
   XVParser     parse_result;  

   scale=1;  // full-size
   n_buffers=DIG_DEF_NUMFRAMES;
   raw_rgb=0;
   while(parse_param(paramstring,parse_result)>0) 
     switch(parse_result.c)
     {
       case 'b':
	 brightness=parse_result.val;
	 break;
       case 'B':
	 n_buffers=parse_result.val;
	 break;
       case 'R':
	 raw_rgb=parse_result.val;
	 break;
       default:
	 cerr << parse_result.c << "=" << parse_result.val 
	      << " is not supported by Ieee1394 (skipping)" << endl;
     }

   if(raw_rgb && scale>1)
   {
	   cerr << "RGB supports just 640x480" << endl;
	   scale=1;
   }
   return 1;
}

Ieee1394::Ieee1394(char *parmstring) : 
ColorVideo(4,640,480) 
{
  int numCameras;
  ninstances++;
  
  // There is supposed to be a color mode, but I've never seen it work
  
  set_input_type(PIX_PACKED);
  
  // The first instance opens up the camera and gets things set.
  
  if (ninstances == 1) {
     init_tab();
     scale=1;
     if(!(handle = dc1394_create_handle(0)))
     {
        cerr << "Couldn't get raw1394 handle" << endl;
        exit(1);
     }
     camera_nodes = dc1394_get_camera_nodes(handle,&numCameras,1);
     if(numCameras<1)
     {
        cerr << "I don't think you connected the camera?"<<endl;
        raw1394_destroy_handle(handle);
        exit(1);
     }
     node_id=camera_nodes[0];
     if(node_id!=-1) 
        cout << "Assigned to node "<<node_id<< endl;
     else
     {
        cerr << "No cameras found!" << endl;
        exit(1);
     }
     node_id=camera_nodes[0];
    if(parmstring) set_params((char *)parmstring);
    dc1394_init_camera(handle,node_id);
    if (dc1394_setup_camera(handle,camera_nodes[0],0,
		 FORMAT_VGA_NONCOMPRESSED,
		 raw_rgb? MODE_640x480_RGB:MODE_640x480_YUV422,
		 SPEED_400,raw_rgb?FRAMERATE_15:FRAMERATE_30,&camera)!=
		         DC1394_SUCCESS)
    {
       cerr << "unable to setup camera" << endl;
       dc1394_release_camera(handle,&camera);
       raw1394_destroy_handle(handle);
       exit(1);
    }
    v_mmap.channel=0;
    cout << "Listening on channel " << v_mmap.channel << endl;
    v_mmap.sync_tag=1;
    v_mmap.nb_buffers=n_buffers;
    v_mmap.packet_size=raw_rgb?640*3:640*4;
    v_mmap.buf_size=raw_rgb?640*480*3:640*480*2;
    v_mmap.flags=VIDEO1394_SYNC_FRAMES;
    open(DC_DEVICE_NAME);
    dc1394_start_iso_transmission(handle,node_id);
    struct video1394_wait w_descr;
    w_descr.channel=v_mmap.channel;
    w_descr.buffer=buffer_id=0;
    ioctl(fd,VIDEO1394_LISTEN_QUEUE_BUFFER,&w_descr); // buffer free?
    cerr << "number of buffers " << n_buffers << endl;
  }
  
  ncols = width();
  nrows = height();
   
  dataPtr = NULL;
}



void*
Ieee1394::current_frame_ptr() 
{
  pre_grab();
  return dataPtr;
}

void
Ieee1394::close() 
{
  
  ninstances --;
  if (ninstances == 0) {
    dc1394_stop_iso_transmission(handle,node_id);
    raw1394_destroy_handle(handle);
    if(fd>0) close();

  }
}


int
Ieee1394::checknew()
{
  struct video1394_wait w_descr;

  w_descr.channel=v_mmap.channel;
  // check if next scheduled
  if(!scheduled)
  {
     w_descr.buffer=(buffer_id+1)%n_buffers;
     if(ioctl(fd,VIDEO1394_LISTEN_QUEUE_BUFFER,&w_descr))
     {
        perror("XVision:");
        exit(1);
     }
     scheduled=1;
  }
  w_descr.channel=v_mmap.channel;
  w_descr.buffer=buffer_id;
  if(!ioctl(fd,VIDEO1394_LISTEN_POLL_BUFFER,&w_descr)) //buffer free?
  {
     scheduled=0;
     if(raw_rgb)
       rawtorgb(mm_buf[buffer_id],image_buffers[buffer_id],
                                   width()*height());
     else
       yuv422torgb(mm_buf[buffer_id],image_buffers[buffer_id],
                                      width()*height());

     dataPtr = (u_long *)image_buffers[buffer_id];
     buffer_id=(buffer_id+1)%n_buffers;
     return 1;
  }

  return 0;
}


int
Ieee1394::pre_grab() 
{
  
  checknew();
  
  if (!gotfirst) {
    while (dataPtr == NULL)
      checknew();
    gotfirst = 1;
  }
  
  if (dataPtr == NULL)
    return 0;
  else
    return 1;
}

void
Ieee1394::set_color(Color x) 
{
  int y = map_color(x);
  char *xx = (char *)(&y);
}

// !!! Major hack.
int
Ieee1394::map_color(Color x) 
{
  int temp = 0;
  char *rptr = ((char *)&temp);
  char *gptr = ((char *)&temp)+1;
  char *bptr = ((char *)&temp)+2;
  
  //  cout << "Color is " << x << endl;
  
  switch (x) {
  case no_color:
    return 0;
    break;
    
  case green_color:
    *gptr = 255;
    break;
    
  case red_color:     
    *rptr = 255;
    break;
    
  case blue_color:    
    *bptr = 255;
    break;
    
  case yellow_color:
    *gptr = 100;
    *bptr = 100;
    break;
    
  case purple_color:
    *rptr = 100;
    *bptr = 100;
    break;
    
  case cyan_color:    
    *gptr = 100;
    *rptr = 100;
    break;
    
  case white_color:   
    *rptr = *gptr = *bptr = 255;
    break;
    
  case black_color:
    break;
    
  default:
    temp = (int )x;
    break;
    
  }
  
  return temp;
}

#endif 
