/*                                                                -*-c++-*-
    Copyright (C) 1993-1997 Gregory D. Hager and Sidd Puri (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//-----------------------------------------------------------------------------
//
//  IT_FG101.sunos.cc
//
//  Definitions of member functions of the IT_FG101 class, for SUNOS.
//
//  06/06/93 Sidd Puri
//
//-----------------------------------------------------------------------------

#include <fcntl.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <sys/file.h>
#include <strings.h>
#include <unistd.h>

#include "Video.hh"
#include "Acquire.hh"
#include "IT_FG101.hh"

extern "C" int gethostname(char *, int);

//-----------------------------------------------------------------------------
//  Constructor and destructor
//-----------------------------------------------------------------------------

// A private function to memory map the framegrabber

unsigned short*
memory_map(unsigned int address)
{

  int fd;

  if ((fd = open("/dev/sfvme24D16",O_RDWR)) == -1) {
    perror("File open");
    printf("ferrno = %d: Probably no device on this host. \n",errno);
    exit(1);
  }

   if ((int)(address = (unsigned int)mmap(0,FG_WIDTH*FG_HEIGHT*sizeof(short),
				     PROT_READ|PROT_WRITE,
				     MAP_SHARED,fd,address)) == -1)  {
     perror("mmap failed");
     exit(1);
   }
  return (unsigned short *)address;
}



// Creating a IT_FG101 device corresponds to initializing it
// and mapping the framegrabber memory.
// !!! This is an older version! Not compatible with XVision 1.2+.
IT_FG101::IT_FG101 (int channel,int camera) : 
Video(FG_101_NCOLS,FG_101_NROWS)
{
  char buff[80];
  gethostname(buff,80);

  if (!strcmp(buff,_ITFG101_HOST_)) {
    printf ("IT_FG101 framegrabber and camera running on %s.\n",buff);
    int 
        regaddress = REGBASE + camera*32,
        memaddress = MEMBASE + camera*0x80000;
  
    // Check to make sure parameters are legal
  
    if ((channel < 0) || (channel > 2))
      panic("Channel number out of range \n",false);
  
    if ((camera < 0) || (camera > 1))
      panic("Camera number out of range \n",false);
  
    // Do this first to make sure there is a framebuffer attached.
  
    mem_map = memory_map(memaddress);
  
    // Do the sethdw
  
    ::sethdw (regaddress, memaddress, FG_101, MEMORY);
  
    // Go off and do the usual stuff
  
    setdim (FG_WIDTH, FG_HEIGHT, TWELVE_BITS); // 1024 * 1024 frame memory
  
    // Get hold of the other memory mapped address
  
    reg_map = ::reg_map;
  
  
    initialize();			     // initialize registers and LUTs
    setreg (X_SPIN, 0);		     // disable spin compensation
    sclear (0);			     // clear frame memory
    setreg (PAN, ncols);
    sclear (0);
    setreg (PAN, 0);
    setvmask (0xf00);		     // don't let IT_FG101 change overlay
    setpmask (0x0ff);		     // processor only affects overlay
    setcamera (channel);		     // select which input to use
    dynamic_luts();		     // use top four bits for overlays
  
    ::grab (0);			     // start digitizing
  
    // Enable host memory addressing.  Anything that changes this
    // should change it back when it is done.
  
    setreg (POINTER_CONTROL, 0x0044);	     // reset pointer control
    setreg (CPU_ADR_CONTROL, 0x0407);	     // allow CPU memory mapping
  
    tracker_cleanup (this);
  } else {
    printf ("Host %s does not have IT_FG101 framegrabber.  Exiting...\n",buff);
    exit (1);
  }
}

void 
IT_FG101::close() 
{
  sethdw();
  stopgrab (NO_WAIT);
}

//-----------------------------------------------------------------------------
//  Refresh functions
//-----------------------------------------------------------------------------

// The following variable is here only to allow probing of the data
// port register without the compiler optimizing it away.

static int tmp;
inline 
void 
fillpixbuffer (unsigned short *dport) 
{   // read the data port register to
#ifdef __GNUC__
  asm volatile ("lduh [%1], %0": "r=" (tmp): "r" (dport): "memory");
#else
  tmp = *dport;
#endif
}

int IT_FG101::grab (int *image, int x, int y, int width, int height, 
		  float angle, 
		  int sampw, int samph,
		  short int mask) {

  if ((angle == 0.0) && (sampw == 1) && (samph == 1))
    return grab_aligned(x,y,width,height,image,mask);

  if (!(in_view(x,y,width*sampw,height*samph,angle)))
    return 0;

  // Set up constants and map the desired window into the framebuffer
  // memory access window.  This scheme assumes that windows are
  // relatively square.

  angle = -angle;            
  x += IT_xspin_offset;			     // set up constants
  int maxside = ((sampw*width + samph*height) >> 1) + 1;
  int xoff = max(0,x-maxside);
  int yoff = max(0,y-maxside);
					     // optimized using CPU addressing
					     // handle with care
  setreg (X_POINTER, xoff);
  setreg (Y_POINTER, yoff);
  x-=xoff;
  y-=yoff;

  if (sampw == 1) {
    if (samph == 1)
      bham_rectangle(mem_map,image,x,y,angle,height,width,mask,512);
    else
      bham_rectangle_msh(mem_map,image,x,y,angle,height,width,samph,mask,512);
  }
  else
    if (samph == 1)
      bham_rectangle_msw(mem_map,image,x,y,angle,height,width,sampw,
			 mask,512);
    else
      bham_rectangle_ms(mem_map,image,x,y,angle,height,width,samph,sampw,
			mask,512);

  return 1;
}

int
IT_FG101::grab_aligned (int x, int y, int width, int height, 
			int *image, short int mask) 
{
  if (!in_view(x,y,width,height))
    return 0;

  x += IT_xspin_offset;			     // transform coordinates to corner
  int ix = x - half (width);
  int iy = y - half (height);

/*\
| for (int row = 0; row < height; row++)     // transfer data
|   rhline (ix, iy + row, width, &image[row * width]);
|					     // mask overlays
| for (int i = width * height - 1; i >= 0; i--)
|   image[i] &= mask;
\*/

  setreg (CPU_ADR_CONTROL, 0x0406);	     // disallow CPU memory mapping
					     // optimized using pixel buffer
					     // handle with care
  setreg (POINTER_CONTROL, 0x3074);	     // auto-increment x pointer by 8
  int *imagerow = &image[(height - 1) * width];
  unsigned short *xpoint = &reg_map[X_POINTER    >> 1];
  unsigned short *ypoint = &reg_map[Y_POINTER    >> 1];
  unsigned short *dport  = &reg_map[DATA_PORT    >> 1];
  unsigned short *pixbuf = &reg_map[PIXEL_BUFFER >> 1];
  for (int row = height - 1; row >= 0; row--) {
    *xpoint = ix;			     // reset pointer, since it changes
    *ypoint = iy + row;
    int loopend = width & ~7;		     // loop over full pixel buffers
    for (int col = 0; col < loopend;) {
      fillpixbuffer (dport);
      for (int pixbufend = col + 8; col < pixbufend; col++)
	imagerow[col] = *pixbuf & mask;
    }
    fillpixbuffer (dport);
    for (; col < width; col++)
      imagerow[col] = *pixbuf & mask;
    imagerow -= width;
  }
  setreg (POINTER_CONTROL, 0x0044);	     // restore register
  setreg (CPU_ADR_CONTROL, 0x0407);	     // reallow CPU memory mapping

  return 1;
}

void IT_FG101::grab_constant_size (int *image, float x, float y, 
				   int width, int height, 
				   float angle, short int mask) 
{
  if (angle == 0) {
    grab_aligned(round(x),round(y),width, height, image, mask);
    return;
  }

  x += IT_xspin_offset;			     // set up constants
  float s = sin (angle), c = cos (angle);
  int maxside = round (sqrt (width * width + height * height)) + 1;

/*\
| for (int row = 0; row < height; row++)     // transfer data
|   for (int col = 0; col < width; col++)
|     image[row * width + col] =
|	wrpixel (x + col * c + row * s,
|		 y + row * c - col * s) & 0xff;
\*/
					     // optimized using CPU addressing
					     // handle with care

  int xoff = int (x - half (maxside));	     // set x and y pointer offsets
  int yoff = int (y - half (maxside));
  setreg (X_POINTER, xoff);
  setreg (Y_POINTER, yoff);

  int *imagerow = &image[(height-1) * width];
  float xr = x - xoff + half (width -1) * c + half (height-1) * s + .5;
  float yr = y - yoff + half (height-1) * c - half (width -1) * s + .5;
  unsigned short *mem = mem_map;
  for (int row = height-1; row >= 0; row--) {
    float xc = xr, yc = yr;
    for (int col = width-1; col >= 0; col--) {
      imagerow[col] = mem[(int (yc) << 9) + int (xc)] & 0xff;
      xc -= c;
      yc += s;
    }
    imagerow -= width;
    xr -= s;
    yr -= c;
  }
}


void IT_FG101::linegrab (int x, int y, int length,
		      int *image,float angle, short int mask) {
  x += IT_xspin_offset;			     // set up constants
					     // optimized using CPU addressing
					     // handle with care
  setreg (X_POINTER, 0);
  setreg (Y_POINTER, 0);

  bham_line(mem_map,image, x,y,angle,length, mask, 512);
    
}

//-----------------------------------------------------------------------------
//  Display functions
//-----------------------------------------------------------------------------

int toggle=0;
int
IT_FG101::show (int x, int y,int width, int height, int *image) 
{
  unsigned int pmask = getpmask();

  if (!(in_view(x,y,width,height)))
    return 0;

  // Permit host write to framebuffer

  setpmask (0x000);

/*\
| for (int row = 0; row < height; row++)     // transfer data
|   for (int col = 0; col < width; col++)
|     image[row * width + col] =
|	wrpixel (x + col * c + row * s,
|		 y + row * c - col * s) & 0xff;
\*/
					     // optimized using CPU addressing
					     // handle with care
  setreg (X_POINTER, x);
  setreg (Y_POINTER, y);

  for (int row = height-1; row >= 0; row--) {
    for (int col = width-1; col >= 0; col--) {
      mem_map[(row << 9) + col] = image[row*width+col];
    }
  }
  
  // restore processor mask

  setpmask (pmask);

  return 1;
}

//--------------------- For Drawing Pretty Pictures ------------

void
IT_FG101::line (float x, float y, int length, float angle,
		Color color, int samp) 
{

  color = map_color(color);

  if (in_view(x,y,
	      (int)ceil(cos(angle)*length),
	      (int)ceil(sin(angle)*length))) {
  x += IT_xspin_offset;
  short int colr = color << 8;		     // turn color into overlay

  int maxside = ((samp*length) >> 1) + 1;

  int xoff = max(0,round(x)-maxside);
  int yoff = max(0,round(y)-maxside);
					     // optimized using CPU addressing
					     // handle with care
  setreg (X_POINTER, xoff);
  setreg (Y_POINTER, yoff);
  x-=xoff;
  y-=yoff;

  draw_line(mem_map,(int)x,(int)y,angle,length,colr,samp);
}
}

void
IT_FG101::line2 (float x1, float y1, float x2, float y2, Color color) 
{
  int xi1=round(x1), yi1=round(y1), xi2=round(x2), yi2=round(y2);
  
  color = map_color(color);

  if (in_view2(x1,y1,x2,y2)) {
    xi1 += IT_xspin_offset;
    xi2 += IT_xspin_offset;
    int colr = color << 8;                   // turn color into overlay
    
    ::line(xi1,yi1,xi2,yi2,colr);
    
    setreg (POINTER_CONTROL, 0x0044);	     // reset pointer control, since it
					     // is changed in ::line
    setreg (CPU_ADR_CONTROL, 0x0407);	     // allow CPU memory mapping
  }
}

void
IT_FG101::square (float x, float y, int r, Color color) 
{
  // draws a square with (x,y) as cender, 2r as width
  line( x-r, y, 2*r, M_PI_2, color);
  line( x+r, y, 2*r, M_PI_2, color);
  line( x, y-r, 2*r, 0, color);
  line( x, y+r, 2*r, 0, color);

}

void
IT_FG101::circle (float x, float y, int r, Color color)
{
  //draws a square instead, since IT library is gone
  square(x,y,r,color);
}

void
IT_FG101::cubic (float x, float y, int length, float angle,
		 float a, float b, float c, Color color)
{
  sethdw();
  color = map_color(color);
  int colr = color << 8;		     // turn color into overlay
    
  int i;
  float ca = cos(angle), sa = sin(angle);

  float xt[2], yt[2];
  int toggle = 0;
  
  // Compute and draw linear segments piece by piece.
  //   Parts out of boundaries of image are squashed to boundary.

  xt[toggle] = x + (ca * 0 - sa * (((a*0+b)*0+c)*0));
  yt[toggle] = y + (sa * 0 + ca * (((a*0+b)*0+c)*0));
  xt[toggle] = max(0,(min(ncols-1,round(xt[toggle]))));
  yt[toggle] = max(0,(min(nrows-1,round(yt[toggle]))));

  toggle = 1 - toggle;

  for (i=1;i<length;i++) {
    xt[toggle] = x + (ca * i - sa * (((a*i+b)*i+c)*i));
    yt[toggle] = y + (sa * i + ca * (((a*i+b)*i+c)*i));
    xt[toggle] = max(0,(min(ncols-1,round(xt[toggle]))));
    yt[toggle] = max(0,(min(nrows-1,round(yt[toggle]))));

    ::line (round(xt[1-toggle]),round(yt[1-toggle]),
	    round(xt[toggle]),round(yt[toggle]),colr);
    toggle = 1 - toggle;
  }  

  // Restore these registers to their expected state

  setreg (POINTER_CONTROL, 0x0044);	     // reset pointer control, since it
					     // is changed in ::line
  setreg (CPU_ADR_CONTROL, 0x0407);	     // allow CPU memory mapping
}

//-----------------------------------------------------------------------------
