/*                                                                -*-c++-*-
    Copyright (C) 1993-1997 Gregory D. Hager and Sidd Puri (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//-----------------------------------------------------------------------------
//
//  Galileo.cc
//
//  Definitions of Galileo class functions.
//
//  06/06/93 Sidd Puri
//  07/06/94 Greg Hager
//
//-----------------------------------------------------------------------------

#include <site_config.h>
#ifdef  GALILEO_MONO

#include <Tracker.hh>
#include <stdlib.h>
#include "Video.hh"
#include "Acquire.hh"
#include "Galileo.hh"

extern "C" {
#include <gl/gl.h>
}

int default_color = 0;

VLServer Galileo::svr = NULL;

int swval[] = {3,5};
int Galileo::switchvar = 0;
int Galileo::ninstances = 0;
VLPath Galileo::path;
long Galileo::win[2];
VLBuffer Galileo::buffer;
VLNode Galileo::src, Galileo::drn;
ulong *Galileo::lbuf[2];

static int showrow = 0;

// !!! Should be relegated to panic(), or to some other universal error fcn.
void 
error_exit(char *mess = NULL) 
{
  if (mess != NULL) 
    cerr << mess << endl;
  else
    cerr << "Some problem with the Galileo setup" << endl;
  exit(1);
}

Galileo::Galileo(int source_channel, int zoom_in) : Video(640,480) 
{
  VLControlValue val;
  
  ninstances++;
  
  if (ninstances > 2) 
    error_exit("Too many Galileo channels");
  
  if (source_channel > 1) 
    error_exit("Only channels 0/1");
  
  foreground();
  
  // Connect to the daemon 
  if (svr == NULL)
    if (!(svr = vlOpenVideo(""))) {
      cerr << "Unable to open camera server" << endl;
      exit(1);
    }
  
  source = source_channel;
  lbuf[source] = NULL;
  
  if (ninstances < 2) {
    // Set up a drain node in memory 
    drn = vlGetNode(svr, VL_DRN, VL_MEM, VL_ANY);
    
    // Set up a source node on any video source  
    src = vlGetNode(svr, VL_SRC, VL_VIDEO, VL_ANY);
    
    // Create a path using the Galileo 
    path = vlCreatePath(svr, GALILEO_DEVICE_NUMBER, src, drn); 
    
    
    // Set up the hardware for and define the usage of the path 
    if ((vlSetupPaths(svr, (VLPathList)&(path), 
		      1, VL_SHARE, VL_SHARE)) < 0)
      error_exit();
    
    switchvar = source;
    
    val.intVal = swval[source];
    if (vlSetControl(svr,path,src,
		     VL_MUXSWITCH,&val) < 0)
      error_exit("Setting Channel Error\n");
    
    // Set the packing to RGB 
    val.intVal = VL_PACKING_RGB_8;
    vlSetControl(svr, path, drn, VL_PACKING, &val);
    
    // Experimental Zoom 
    val.fractVal.numerator = 1;
    val.fractVal.denominator = zoom_in;
    if(vlSetControl(svr, path, drn, VL_ZOOM, &val))
      error_exit("Can't set zoom");
    
    // Get the video size 
    vlGetControl(svr, path, drn, VL_SIZE, &val);
    
    ncols = val.xyVal.x;
    nrows = val.xyVal.y;
    
    cout << "Picture size is: " << nrows << " by " << ncols << endl;
  }    
  else {
    
    // Get the video size 
    vlGetControl(svr, path, drn, VL_SIZE, &val);
    
    ncols = val.xyVal.x;
    nrows = val.xyVal.y;
    
    // Build a local buffer in case we are using two channels
    //   and have to copy things. 
    
    lbuf[0] = new ulong[ncols*nrows];
    if (lbuf[0] == NULL)
      error_exit("Unable to allocate internal buffer");
    memset(lbuf[0],0,sizeof(ulong)*ncols*nrows);
    
    lbuf[1] = new ulong[ncols*nrows];
    if (lbuf[1] == NULL)
      error_exit("Unable to allocate internal buffer");
    memset(lbuf[1],0,sizeof(ulong)*ncols*nrows);
  }
  
  source = source_channel;
  
  
  // Set up and open a GL window to display the data 
  prefsize(width(),height());
  win[source] = winopen("Tracking Window");
  RGBmode();
  RGBcolor(0,100,200);
  pixmode(PM_TTOB, 1);
  gconfig();
  
  // Create and register a buffer for 1 frame 
  if (ninstances < 2) {
    
    buffer = vlCreateBuffer(svr, path, drn, 2);
    if (buffer == NULL)
      error_exit("Buffer Allocation");	
    
    vlRegisterBuffer(svr, path, drn, buffer);
    
    // Begin the data transfer 
    
    if (vlBeginTransfer(svr, path, 0, NULL))
      error_exit();
  }
  
}

VLInfoPtr 
Galileo::checknew()
{
  VLInfoPtr info;
  ulong *dataPtr;
  
  info = vlGetNextValid(svr, buffer);
  
  if (info != NULL) {
    
    //    cout << ".";
    // Get a pointer to the frame 
    dataPtr = (ulong *)vlGetActiveRegion(svr, buffer, info);
    
    // Write the data to the proper screen 
    
    winset(win[switchvar]);
    lrectwrite(0,0, width()-1, height()-1, (ulong *)dataPtr);
    
    //    lrectwrite(0, showrow, width()-1, min(showrow+MYJUMP,height()-1), 
    //	((ulong *)dataPtr) + width()*showrow);
    
    //    show(0,showrow,width(),MYJUMP,
    //	((int *)dataPtr) + width()*showrow);
    
    //    showrow += MYJUMP;
    //    if (showrow > height())
    //      showrow = 0;
    
    if (ninstances > 1) {
      
      // Copy data to save area 
      
      //      cout << "sw  " << switchvar << endl;
      memcpy(lbuf[switchvar],dataPtr,sizeof(ulong)*width()*height());
      
      // Switch channels 
      
      vlEndTransfer(svr,path);
      vlPutFree(svr, buffer);
      
      switchvar  = 1-switchvar;
      if (vlSetControl(svr,path,src,VL_MUXSWITCH,
		       (VLControlValue*)&(swval[switchvar])) < 0)
	cerr <<"Set Control Error " << vlErrno << endl;
      
      // Begin the data transfer 
      if (vlBeginTransfer(svr, path, 0, NULL)) 
	cerr <<"Transfer Error " << vlErrno << endl;
      
    }
    else {
      lbuf[source] = dataPtr;
      vlPutFree(svr,buffer);
    }
  }
  
  return info;
}

int Galileo::grab(int *image, int x, int y, int width, int height,
		  float angle, int sampw, int samph, short int mask)
{
  checknew();
  
  if (lbuf[source] == NULL)
    return 0;
  
  angle = -angle;
  
  if (!(in_view(x,y,width*sampw,height*samph,angle)))
    return 0;
  
  if (sampw == 1) {
    if (samph == 1)
      bham_rectangle(lbuf[source],image,x,y,angle,height,width,mask,ncols);
    else
      bham_rectangle_msh(lbuf[source],image,x,y,angle,height,width,samph,	
			 mask,ncols);
  }
  else
    if (samph == 1)
      bham_rectangle_msw(lbuf[source],image,x,y,angle,height,width,sampw,mask,
			 ncols);
    else
      bham_rectangle_ms(lbuf[source],image,x,y,angle,height,width,samph,sampw,mask,
			ncols);
  
  
  return 1;
}

int 
Galileo::show (int x, int y, int width, int height, int *image) 
{
  
  winset(win[source]);
  lrectwrite(x,this->height()-y-height,
	     x+width-1,this->height()-y-1,
	     (ulong *)image);
  return 1;
}

void 
Galileo::line (float x, float y, int length, float angle,
		    int color,int samp){
  
  if (in_view(x,y,
	      ceil(fabs(cos(angle))*length),
	      ceil(fabs(sin(angle))*length))) {
    
    length/=2.0;
    float vec[2];
    
    winset(win[source]);
    RGBcolor(0,255,0);
    bgnline();
    vec[0] = x-cos(angle)*length;
    vec[1] = height()- (y-sin(angle)*length);
    v2f(vec);
    vec[0] = x+cos(angle)*length;
    vec[1] = height() - (y+sin(angle)*length);
    v2f(vec);
    endline();
  }
}

#endif // GALILEO_MONO
