/*
    Copyright (C) 1994 Kentaro Toyama and Gregory D. Hager (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    We provide this software "as is" without express or implied warranty.

*/
//-----------------------------------------------------------------------------
//
//  Contour.cc
//
//  Snake-like contour tracker.
//
//  July 19, 1994
//  Kentaro Toyama
//  Sept. 5, 1994 -- updated for 0.95 release of tracking code.
//  Feb. 28, 1995 -- updated for 0.99 release of tracking code.
//  Nov. 7, 1995  -- updated for 1.02 release, adapted to Component class,
//                     simplified and debugged.
//
//-----------------------------------------------------------------------------

#include <math.h>
#include "Contour.hh"
#include "Line.hh"

// Utility functions (should be in Tools.hh) ----------------------------

float dist (float x1, float y1, float x2, float y2) {
  return sqrt ((x1-x2) * (x1-x2) + (y1-y2) * (y1-y2));
}

float dista (Component *a, Component *b) {
  return sqrt ((a->x() - b->x())*(a->x() - b->x()) + 
	       (a->y() - b->y())*(a->y() - b->y()));
}

float absanglea (Component *a0, Component *a1) {
  float ang;
  
  ang = atan2(a1->y() - a0->y(),a1->x() - a0->x()) ; // - 0.15;
  return ang;
}


//----------------------------------------------------------------------
// Contour class member functions
//----------------------------------------------------------------------

Contour::Contour (Video *v_in, int st_in, 
		  int compwidth_in) :
BasicFeature (2, v_in, "Contour", default_color) {

  compwidth = compwidth_in;
  number = 0;
  show_type = st_in;

  feature_array = new (Component *);
}

Contour::Contour (Contour &p): BasicFeature (2, p.source(), p.name,
					     p.color) {
  *this = p;
  compwidth = p.compwidth;
  number = p.number;
  show_type = p.show_type;
   
  features = p.features;
  feature_array = p.feature_array;
}

Contour::~Contour() {
  delete [] feature_array;
  features.kill();
}

Contour &Contour::operator+= (BasicFeature &t) {
  BasicFeature *temp = t.dup();
  features += temp;
  number++;

  return *this;
}

Contour &Contour::operator+= (BasicFeature *t) {
  features += t;
  number++;

  return *this;
}


int Contour::array_init () {
  int i;
  for (i=0;i<number;i++) {
    feature_array[i] = ((Component *)child(i));
  }
  return 1;
}

// Manual initialization.  Click near control points.
int Contour::interactive_init (CWindow &c, Video *v, 
			       int width, int comps) {
  int i;
  Component *a;
  for (i=0;i<comps;i++) {
    a = new Component(v, width);
    (*this) += a;
  }

  features.front();
  for (i=0; i<comps; ++i, ++features)
    (*features)->interactive_init(c);

  coords co[contour_max_comps];
 
  for (i=0;i<comps;i++) 
    co[i] = spinterpolate_nocheck(i,comps);
  for (i=0;i<comps;i++)
    ((Component *)child(i))->state_init(&co[i]);

  number = comps;
  array_init();

  c.close();

  printf ("%d components initialized.\n", number);
  return 1;
}

// Automatic initialization by following strongest edge.  Doesn't work
//   too well now, especially for contours with corners.
int Contour::interactive_init (CWindow &c) {
  Edge e1;
  Line ls(&e1,v,SEARCH_LINE_LEN,SEARCH_LINE_WID);
  Component *lt = new Component(v,compwidth);
  
  int i;
  float xinit, yinit;
  coords co;

  ls.interactive_init(c);
  ls.update();
  xinit = ls.x();
  yinit = ls.y();

  co.x = ls.x();
  co.y = ls.y();
  co.o = ls.orientation();
  co.v = 1;
  lt->state_init(&co);
  (*this) += (*lt);

  int sharp;
  i=1;
  while ((i<5) || 
         ((i<contour_max_comps) && (dist(xinit,yinit,ls.x(),ls.y())>2*INCH))) {
    ls.findnext(INCH,0);
    sharp = ls.findnext(INCH,0);

    lt = new Component(v,compwidth);
    co.x = ls.x();
    co.y = ls.y();
    co.o = ls.orientation();
    co.v = 1;
    
    lt->state_init(&co);
    (*this) += (*lt);
    lt->show();
    i++;
  }
  printf ("%d components initialized.\n", number);
  c.close();

  array_init();

  return 1;
}


int Contour::update() {
  int i;
  coords spn[contour_max_comps];

  for (i=0;i<number;i++) {
    spn[i] = spinterpolate_nocheck(i,number);
  }

  for (i=0;i<number;i++) {
    Component *a = feature_array[i];
    statedata *s = a->look(&spn[i]);
      
    coords newpos;
    newpos = addoffset(&spn[i],s->bestguess-compwidth/2);
    if (a->in_view(int(newpos.x),int(newpos.y),newpos.o))
      a->set(&newpos);
  }

// Let components update themselves, no adjustment by Contour.
//   for (i=0;i<number;i++) {
//     feature_array[i]->update();
//   }

  return 1;
}


//-------------------------------------------------------------------
// interpolation, extrapolation functions
//-------------------------------------------------------------------

coords Contour::spinterpolate_nocheck (int num, int tot) {
  coords pt;
  
  int l1 = (num + tot - 1)%tot;
  int l2 = (num + tot + 1)%tot;

  Component *a1 = (Component*)child(l1);
  Component *a2 = (Component*)child(l2);

  pt.x = (a1->x() + a2->x()) / 2.0;
  pt.y = (a1->y() + a2->y()) / 2.0;
  pt.o = atan2(a2->y() - a1->y(), a2->x() - a1->x());
  pt.v = 1;

  return pt;
}


coords Contour::addoffset (coords *a, int offset) {
  coords pt;

  float angle = a->o;
  double ss, cc;

  //sincos(angle,&ss,&cc); // sgi doesn't have sincos
  ss=sin(angle); cc=cos(angle);

  float off = offset;         // -?
  float wangle = -(angle)-M_PI_2;
  
  pt.x = a->x - signum (off*ss) * fabs(off * cos (wangle));
  if (pt.x <     0) pt.x = 0;
  if (pt.x > (source()->width())) pt.x = (source()->width());
  
  pt.y = a->y + signum (off*cc) * fabs(off * sin (wangle));
  if (pt.y <     0) pt.y = 0;
  if (pt.y > (source()->height())) pt.y = (source()->height());

  pt.o = angle;
  pt.v = 1;

  return pt;
}

//-------------------------------------------------------------------
// functions that wouldn't have had to be here if I did this right
//    with CompFeatures.
//-------------------------------------------------------------------

void Contour::forward(int n) {
  int i;
  for (i=0;i<number;i++) {
    feature_array[i]->back();
  }
}

void Contour::back(int n) {
  int i;
  for (i=0;i<number;i++) {
    feature_array[i]->forward();
  }
}

void Contour::erase_last() {
  back(); clear(); forward();
}

void Contour::replicate_forward() {
  int i;
  for (i=0;i<number;i++) {
    feature_array[i]->replicate_forward();
  }
}  

//-------------------------------------------------------------------
// display functions
//-------------------------------------------------------------------

int Contour::show(Color color_in) {
  switch (show_type) {
  case SEARCH_WINDOWS:
    return show1();
    break;
  case CUBIC_SEGS:
    return show3();
    break;
  default:
  case LINE_SEGS:
    return show2();
    break;
  }
}

int Contour::clear() {
  switch (show_type) {
  case SEARCH_WINDOWS:
    return clear1();
    break;
  case CUBIC_SEGS:
    return clear3();
    break;
  default:
  case LINE_SEGS:
    return clear2();
    break;
  }
}

// Show search windows.
int Contour::show1() {
  int res = 1;
  for (int i=0; i<number;i++) {
    res &= feature_array[i]->show();
  }
  return res;
}

int Contour::clear1() {
  int res = 1;
  for (int i=0; i<number;i++) {
    res &= feature_array[i]->clear();
  }
  return res;
}

// Show components connected by straight lines.
int Contour::show2 () {
  int i,j;

  for (i=0;i<number;i++) {
    j = (i+1)%number;
    v->line2(feature_array[i]->x(),
             feature_array[i]->y(),
             feature_array[j]->x(),
             feature_array[j]->y(),
             feature_array[i]->colorof());
  }  
  return 1;
}

int Contour::clear2() {
  int i,j;

  for (i=0;i<number;i++) {
    j = (i+1)%number;
    v->clearline2(feature_array[i]->x(),
		  feature_array[i]->y(),
		  feature_array[j]->x(),
		  feature_array[j]->y());
  }  
  return 1;
}

// Show components connected by cubic segments.
// These operations do something only if cubic is defined for the
//   video class.  
int Contour::show3 () {
  int i;

  float ang,len,aa,bb,cc, slope1, slope2;
  for (i=0;i<number;i++) {
    Component *a = feature_array[i];
    Component *b = feature_array[(i+1)%number];
    ang = absanglea(a,b);
    slope1 = tan(a->o() - ang);
    slope2 = tan(b->o() - ang);
    len = dista (a,b);
    aa = (slope2 + slope1)/(len*len);
    bb = -(2*slope1 + slope2)/len;
    cc = slope1;
    v->cubic (a->x()+16,a->y(),round(len)+1,ang,aa,bb,cc,
	      feature_array[i]->colorof());
  }  
  return 1;
}   

int Contour::clear3 () {
  int i;
  // Inefficient to recompute these values.
  float ang, len,aa,bb,cc,slope1, slope2;
  for (i=0;i<number;i++) {
    Component *a = feature_array[i];
    Component *b = feature_array[(i+1)%number];
    ang = absanglea(a,b);
    slope1 = tan(a->o() - ang);
    slope2 = tan(b->o() - ang);
    len = dista (a,b);
    aa = (slope2 + slope1)/(len*len);
    bb = -(2*slope1 + slope2)/len;
    cc = slope1;
    v->clearcubic (a->x()+16,a->y(),round(len)+1,ang,aa,bb,cc);
  }  
  return 1;
}
