/*                                                                -*-c++-*-
    Copyright (C) 1995 Kentaro Toyama and Gregory D. Hager (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    We provide this software "as is" without express or implied warranty.

*/
//-----------------------------------------------------------------------------
//
//  Component.hh
//
//  Declaration of Component class.
//
//  July 19, 1994
//  Kentaro Toyama
//  Sept. 5, 1994 -- updated for 0.95 release of tracking code.
//  Feb. 28, 1995 -- updated for 0.99 release of tracking code.
//  Nov.  6, 1995 -- updated for 1.02 release of tracking code.  Bugs fixed.
//
//-----------------------------------------------------------------------------
#include <math.h>
#include "Tracker.hh"
#include "FTypes.hh"
#include "Pattern.hh"
#include "Image.hh"
#include "BasicFeature.hh"
#include "Line.hh"

//-----------------------------------------------------------------------------
// Constants, etc.
//-----------------------------------------------------------------------------

int const comp_dft_width = 30;          // max width of line.
int const comp_dft_length = 1;          // default length of line.
int const comp_dft_dp = 0;          // distraction proofing?
int const comp_dft_grow = 0;        // expanding search window?
int const comp_dft_os = 1;              // original width sampling rate.
int const comp_statevars = 6;           // x, y, angle, str, width, and samp

struct statedata {                  // Struct used to hold Automaton state.
  int state;                        
  int certainty;                    // Certainty of bestguess.
  int direction;                    // Direction to which edge could lie.
  int bestpk;                       // Edge number of best guess.
  int inside;                       // Intensity values immediately inside
  int outside;                      //   and outside contour edge.
  int numedges;                     // Number of edges in window.
  int bestguess;                    // Loc of best guess.
  int secguess;                     // Loc of next guess.
};

struct coords {
  float x;
  float y;
  float o;
  int v;
};

//-----------------------------------------------------------------------------
//  Component class declaration
//-----------------------------------------------------------------------------

class Component : 
public BasicFeature, public Orientable, public Positionable {

  // Line related stuff --

  int distraction_proof;
  int grow;

  int length, max_width;
  int orig_samp;

  Shortedge *pat;
  Image *im;

  float window_angle() { return -orientation() - M_PI_2; }

  // Unique to Component --

  SE_out lowout;                    // Low level output (peaks, modes).
  statedata cur, last;              // New and recent state information.
  int *lastimval;                   // Last image seen.

 public:

  Component(Video *v, int width_in = comp_dft_width, 
	    int dp_in = comp_dft_dp, int grow_in = comp_dft_grow, 
	    int os_in = comp_dft_os);

  Component (const Component &x);
  
  ~Component() {}

  BasicFeature *dup() { return new Component (*this); }

  int state_init (float *s) {
    return state_init (s[0],s[1],s[2]);
  }
  int state_init (coords *co);
  int state_init (float xx, float yy, float oo);
  int interactive_init (CWindow &w);

  int sub_init (SE_out *s);

  // state variable accessors

  float x() const        { return state[0]; }        
  float y() const        { return state[1]; }
  float o() const        { return state[2]; }
  float orientation() const { return state[2]; }
  int width() const      { return int(state[3]); }
  int samp() const       { return int(state[4]); } 
  float strength() const { return state[5]; }

  // state variable modifiers

  float set_x(float newx)             { return state[0] = newx; }            
  float set_y(float newy)             { return state[1] = newy; }
  float set_o(float newo)             { return state[2] = newo; }
  float set_orientation(float newo)   { return state[2] = newo; }
  int set_width(int neww)             { return int((state[3] = float(neww))); }
  int set_samp(int news)              { return int((state[4] = float(news))); }

  int showwid(Color color_in = default_color, Color color_in2 = red_color);
  int showwid() { return showwid(color,red_color); }
  int clearwid();

  int show (Color color_in = default_color) 
    { return(showwid(color,red_color)); }
  int clear() { return (clearwid()); }

  int in_view (int x, int y, float o);
  int in_view() { return in_view(round(x()),round(y()),orientation()); }

  statedata *look(float x, float y, float o) {
    coords tmp;
    tmp.x = x; tmp.y = y; tmp.o = o;
    return look(&tmp);
  }
  statedata *look(coords *co);
  statedata *look() {
    return look(x(),y(),o());
  }
  void set (int offset);
  void set (coords *final);

  void rf() {replicate_forward();}
  void clear_last();

  Color colorof() {return color;}

  int update() {
    statedata *tmp;
    replicate_forward();
    tmp = look();
    set(tmp->bestguess - width()/2 );
    return 1;
  }

};



