/*                                                                -*-c++-*-
    Copyright (C) 1991 Gregory D. Hager and Sidd Puri (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.
*/

//-----------------------------------------------------------------------------
//  XWindowPlus.cc implementation of XWindowPlus class
// 
//  08/01/94 Gregory D. Hager
//  07/05/95 Jonathan Wang
//  07/25/96 Gregory D. Hager
//  Added color support.  Now a console can be opened as color or
//  monochrome.  At the moment, however, all of the attached windows
//  inherit this from the console, so mixed windows are not possible.
//  Also if you try to display a monochrome image on a color window,
//  you'll get a very blue looking image.........
//  Added 16 bit color support.
//
//  11/25/96 Loring Holden (lsh@cs.brown.edu)
//  Added support for 24 bit color when 24 bit color is not default visual
//  Fixed shared memory support so shared mem segments won't hang around
//
//-----------------------------------------------------------------------------

#include <X11/Xlib.h>
#include <X11/extensions/XShm.h>
#include <X11/cursorfont.h>
#include <X11/Xutil.h>
#include <string.h>	
#include <stdlib.h>	
#include <stdio.h>	
#include "XWindowPlus.hh"
#include <assert.h>

inline int bits_per_pixel_mask(unsigned long mask)
{
  int bpp = 0;
  if (mask & 0x01)      bpp++;  if (mask & 0x02)      bpp++;
  if (mask & 0x04)      bpp++;  if (mask & 0x08)      bpp++;
  if (mask & 0x10)      bpp++;  if (mask & 0x20)      bpp++;
  if (mask & 0x40)      bpp++;  if (mask & 0x80)      bpp++;
  if (mask & 0x0100)    bpp++;  if (mask & 0x0200)    bpp++;
  if (mask & 0x0400)    bpp++;  if (mask & 0x0800)    bpp++;
  if (mask & 0x1000)    bpp++;  if (mask & 0x2000)    bpp++;
  if (mask & 0x4000)    bpp++;  if (mask & 0x8000)    bpp++;
  if (mask & 0x010000)  bpp++;  if (mask & 0x020000)  bpp++;
  if (mask & 0x040000)  bpp++;  if (mask & 0x080000)  bpp++;
  if (mask & 0x100000)  bpp++;  if (mask & 0x200000)  bpp++;
  if (mask & 0x400000)  bpp++;  if (mask & 0x800000)  bpp++;
  if (mask & 0x01000000)bpp++;  if (mask & 0x02000000)bpp++;
  if (mask & 0x04000000)bpp++;  if (mask & 0x08000000)bpp++;
  if (mask & 0x10000000)bpp++;  if (mask & 0x20000000)bpp++;
  if (mask & 0x40000000)bpp++;  if (mask & 0x80000000)bpp++;
  return bpp;
};

// finds the multiplier to get to the least significant bit of mask
inline  int get_multiplier(unsigned long mask)
{
  unsigned long mult = 0x01;
  assert(mask); // there has to be a mask for this to work
  while (!(mult & mask))
    mult = mult << 1;
  return (int)mult;
};

inline int get_shift_for_division(unsigned int new_max, unsigned int old_max)
{
  int n_sb = 1, o_sb = 1;
  while (!(0X80000000 & new_max)) 
    {
      n_sb++;
      new_max = new_max << 1;
    };
  while (!(0X80000000 & old_max)) 
    {
      o_sb++;
      old_max = old_max << 1;
    };  
  // cout << "need to shift value by " << n_sb - o_sb << endl;
  return n_sb - o_sb;
};

static unsigned long GetColor(Display *disp, Colormap cmap, 
			      char* c_n, int& errcount)
{
  // Colormap cmap;
  XColor best, exact;
  /* Get default colormap of window*/
  // cmap = DefaultColormapOfScreen(DefaultScreenOfDisplay(disp));

  if (c_n != NULL) 
    if (XAllocNamedColor(disp, cmap, c_n, &best, &exact)) {
      /*fprintf(stderr, "XWindow allocated pixel value %#X \
	for pixel name %s\n", best.pixel, c_n);*/
      return (best.pixel);
    }
  
  fprintf(stderr, "XWindow could not allocate a pixel value \
for %s, returned WhitePixel instead\n", c_n); 
    errcount += 1;
    return WhitePixel (disp, DefaultScreen(disp));  
};

void XWindowPlus::allocate_color_pixels() {
  //*** Get color pixels for graphics **********

  int errcount = 0;

  if (color_scheme.is_color) {
    // cerr << "Initializing Color Pixels" << endl;
    red_pixel    = GetColor(display, colormap, "Red", errcount);
    blue_pixel   = GetColor(display, colormap, "Blue", errcount);
    green_pixel  = GetColor(display, colormap, "Green", errcount);
    yellow_pixel = GetColor(display, colormap, "Yellow", errcount);
    purple_pixel = GetColor(display, colormap, "Purple", errcount);
    cyan_pixel   = GetColor(display, colormap, "Cyan", errcount);
    white_pixel  = GetColor(display, colormap, "White", errcount);
    black_pixel  = GetColor(display, colormap, "Black", errcount);
  } else {
    // cerr << "Initializing B/W Highlighter Pixels" << endl;
    red_pixel    = GetColor(display, colormap, "White", errcount);
    blue_pixel   = GetColor(display, colormap, "Black", errcount);
    green_pixel  = GetColor(display, colormap, "White", errcount);
    yellow_pixel = GetColor(display, colormap, "White", errcount);
    purple_pixel = GetColor(display, colormap, "Black", errcount);
    cyan_pixel   = GetColor(display, colormap,  "White", errcount);
    white_pixel  = GetColor(display, colormap, "White", errcount);
    black_pixel  = GetColor(display, colormap, "Black", errcount);
  };
  if (errcount)
    cerr << "Warning: XWindow could not allocate " 
	 << errcount << " color pixels" << endl;
};


inline unsigned long XWindowPlus::xv_get_color(Color pixel_name)
{
  switch (pixel_name) {     
  case green_color:
    return green_pixel;
  case blue_color:  
    return blue_pixel;
  case red_color:    
    return red_pixel;
  case yellow_color:
    return yellow_pixel;
  case purple_color:
    return purple_pixel;
  case cyan_color:
    return cyan_pixel;
  case white_color:
    return white_pixel;
  case black_color:
    return black_pixel;
  default:
    cerr << "Warning, asked for unknown Color pixel, returning white_pixel" 
	 << endl;
    return white_pixel;
  };
};

// XWindowPlus constructor 
XWindowPlus::XWindowPlus(XConsole& x) :
  CWindow(),
  width(-1),
  height(-1)  
{
  c = &x;
  if (c->color())
    color_scheme.is_color = 1;
  else
    color_scheme.is_color = 0;
  visual = NULL;
};

// XWindowPlus constructor 
XWindowPlus::XWindowPlus(int iscolor) :
  CWindow(),
  width(-1),
  height(-1)  
{
  color_scheme.is_color = iscolor;
  visual = NULL;
};

XWindowPlus::~XWindowPlus()
{  
  close();
};

void XWindowPlus::Set_Name(char *name)
{
  if (name != NULL) 
    {
      XTextProperty nametext;
      XStringListToTextProperty (&name, 1, &nametext);
      XSetWMName (display, winid, &nametext);
      XSetWMIconName (display, winid, &nametext);
    }
};

static void printvisual(const XVisualInfo& vi)
{
  cerr << "VisualID = " << vi.visualid << endl
       << "screen = " << vi.screen << endl
       << "depth = " << vi.depth << endl
       << "red_mask = " << vi.red_mask << endl 
       << "green_mask = " << vi.green_mask << endl 
       << "blue_mask = " << vi.blue_mask << endl
       << "colormap_size = " << vi.colormap_size << endl
       << "bits_per_rgb = " << vi.bits_per_rgb << endl;
};

void
XWindowPlus::open(int width_in, int height_in, 
		  char* window_title, char* display_name)
{ 
  if (!_is_open) {       
    //************ open default display ************
    if ((display = XOpenDisplay(display_name)) == 0) {
      if (!display_name)
	cerr << ("Can't open default X display")<< endl;
      else
	cerr << ("Can't open X display at ") << display_name << endl;
      exit(1);
    }
    else
      if (display_name)
	cerr << "Opened display " << display_name << endl;
      else
	cerr << "Opened default display " << endl;
    //********************************************

    //*** get screen, root window for display ****
    screen = XDefaultScreen (display);
    root = XDefaultRootWindow (display);
    //********************************************    

    // if you want to play with different visuals do that here
    // but you have to make sure there is an appropriate colormap !!!
    // this is where you would try to match gray scale visual 
    // or create a RBG color map and make sure color pixels allocate 
    // right (ie not from default colormap)

    // for now we support for color only:  
    // 24 bit true color/direct color, 
    // 16 bit true color/direct color,
    // 8  bit true color/direct color 
    //  (depth is decided based on what you set your display to so to
    //   get 24 bit mode you have to run X in 24 bit mode)
    // 
    // for monochrome 
    // 8 bit GreyScale
    
    // if you ask for monochrome and we cant give it to you in 
    // 8 bit color, we will just use a color visual and copy 
    // the value into all three bands
   
    // the first step is to check, if the user wants monochroome, and
    // if so, to try to get the gray scale visual


    int found = 0, status =0;
    if (!color_scheme.is_color) {
      // If I knew of a 16/24/32 bit Gray Scale visual I might 
      // try to grab one here.  It would be nice to have  
      cerr << "Trying to get 8 bit StaticGray Visual ... ";
      status = XMatchVisualInfo(display,screen,8,StaticGray,visual_info);
      cerr << "found " << status << endl;
      if (status) {
	color_scheme.bitsperpixel = 8;
	color_scheme.redmax = 255;
	//cerr << "printing StaticGray visual" << endl; 
	//printvisual(*visual_info);
	found = 1;
      } 
      if (!found) { 
        cerr << "Trying to get 8 bit GreyScale Visual ... ";
        status = XMatchVisualInfo(display,screen,8,GrayScale,visual_info);
	cerr << "found " << status << endl;
	if (status) {
	  color_scheme.bitsperpixel = 8;
	  color_scheme.redmax = 255;
	  //cerr << "printing GrayScale visual" << endl; 
	  //printvisual(*visual_info);
	  found = 1;
	};
      };
    };
    
    if (!found) {
      depth = XDefaultDepth(display, screen);
    } else {
      depth = 8;
    };

    // cerr << "XWindow depth: " << depth << endl;

    // setup color display
    if (!found) {
      color_scheme.is_color = 1;
      cerr << "Trying to get " << depth << " bit TrueColor Visual ... ";
      status = XMatchVisualInfo(display,screen,depth,TrueColor,visual_info);
      cerr << "found " << status << endl;      
      if (status) {
	assert (visual_info[0].red_mask); // gotta make sure there are masks
	//cerr << "printing TrueColor visual" << endl;
	//printvisual(visual_info[0]);
	found = 1;
      };
    };
    if (!found) {
      color_scheme.is_color = 1;
      cerr << "Trying to get " << depth << " bit DirectColor Visual ... ";
      status = XMatchVisualInfo(display,screen,depth,DirectColor,visual_info);
      cerr << "found " << status << endl;      
      if (status) {
	assert (visual_info[0].red_mask); // gotta make sure there are masks
	//cerr << "printing DirectColor visual" << endl;
	//printvisual(visual_info[0]);
	found = 1;
      };
    };
   
    if (found) {
      visual = visual_info->visual;
      //cout << "visual is " << visual << endl;
    } else {
      cerr << "Xwindow could obtain proper visual" << endl;
      exit(1);
    };

    // visual info here needs to be matched more carefully!!
    /***************FIX ME**********************************/
 
    if (color_scheme.is_color) {
     
      color_scheme.redmask =    visual_info->red_mask;
      color_scheme.greenmask =  visual_info->green_mask;
      color_scheme.bluemask =   visual_info->blue_mask;
      
      /* fprintf(stderr, "visual info -- bits_per_rgb equals %u?\n", visual_info->bits_per_rgb);
	 have to be explicit about bits per pixel */
      unsigned int bpp_red, bpp_green, bpp_blue;
      /*         cerr << "bpp_red = " 
	 << (bpp_red =  bits_per_pixel_mask(visual_info->red_mask))
	 << "\tbpp_green = " 
	 << (bpp_green =  bits_per_pixel_mask(visual_info->green_mask))
	 << "\tbpp_blue = " 
	 << (bpp_blue =  bits_per_pixel_mask(visual_info->blue_mask))
	 << endl;
	 fflush(stderr);
	 fprintf(stderr, "redmask %#X \tgreenmask %#X \tbluemask %#X \n",
	 visual_info->red_mask, 
	 visual_info->green_mask, 
	 visual_info->blue_mask);
	 fflush(stderr); 
	 cerr << "redmax = " <<      (color_scheme.redmax   = ((int)1 << bpp_red) - 1); 
	 cerr << "\tgreenmax = " <<  (color_scheme.greenmax = ((int)1 << bpp_green) - 1); 
	 cerr << "\tbluemax = " <<   (color_scheme.bluemax  = ((int)1 << bpp_blue) - 1); 
	 cerr << "\nredmult = " <<   (color_scheme.redmult  = get_multiplier(visual_info->red_mask));
	 cerr << "\tgreenmult = " << (color_scheme.greenmult= get_multiplier(visual_info->green_mask));
	 cerr << "\tbluemult = " <<  (color_scheme.bluemult = get_multiplier(visual_info->blue_mask)) << endl;      
      */
      
      bpp_red =  bits_per_pixel_mask(visual_info->red_mask);
      bpp_green =  bits_per_pixel_mask(visual_info->green_mask);
      bpp_blue =  bits_per_pixel_mask(visual_info->blue_mask);
      color_scheme.redmax   = ((int)1 << bpp_red) - 1 ;
      color_scheme.greenmax = ((int)1 << bpp_green) - 1 ;
      color_scheme.bluemax  = ((int)1 << bpp_blue) - 1 ;
      color_scheme.redmult  = get_multiplier(visual_info->red_mask);
      color_scheme.greenmult= get_multiplier(visual_info->green_mask);
      color_scheme.bluemult = get_multiplier(visual_info->blue_mask);     
      if (depth == 24) color_scheme.bitsperpixel = 32;
      else color_scheme.bitsperpixel = depth;
      //************************************************
    };

    /*
    cout << "\n**********************" << endl;
    cout << " FOR YOUR INFORMATION:" << endl;
    int num;
    Colormap* cmaps =
      XListInstalledColormaps (display, root, &num);
    cout << "  " << num << " instaled colormaps" << endl;

    int* depths = XListDepths(display, screen, &num);
    for (int iii = 0; iii < num; iii++)
      cout << "  " << "found depth of " << depths[iii] << endl; 
    cout << "**********************" << endl;
       */    



    colormap = XCreateColormap(display, root, visual, AllocNone);
    // colormap = XDefaultColormap (display, screen);
    allocate_color_pixels(); // initializes color pixels for graphics

    //**************  create window ***************
    width = width_in;  // window dimensions
    height = height_in;   
    XSetWindowAttributes attributes;
    attributes.cursor = XCreateFontCursor(display, XC_crosshair);    
    attributes.colormap = colormap;
    attributes.background_pixel = black_pixel;
    attributes.border_pixel = white_pixel;

    unsigned long mask = CWBackPixel | CWBorderPixel |CWCursor | CWColormap; 
    winid = XCreateWindow (display, root,
			   0, 0,		     // position (not used)
			   width, height,
			   1,			     // border width
			   depth,	             // depth
			   InputOutput,	             // class
			   visual,
			   mask,		     // value mask
			   &attributes);
    if (winid == 0) 
      { 
	cerr << ("Could not create X window"); 
	exit(1);
      };
    //*********************************************




    //*** create buffer pixmap, Ximage ************
    buffer_pixmap = XCreatePixmap(display, winid, width,height, depth);
    Xim = create_image(width,height);    
    //*********************************************


    //*** creates the graphics context for drawing into our window ***/
    XGCValues gcvalues;
    gcvalues.function = GXcopy; // overwrite graphics
    gc = XCreateGC (display, buffer_pixmap, 
		    GCFunction | GCForeground | GCBackground,
		    &gcvalues);

    XTextProperty nametext;
    XStringListToTextProperty (&window_title, 1, &nametext);
    XSizeHints sizehints;
    sizehints.flags = PSize | PMinSize | PMaxSize;
    sizehints.width  = sizehints.min_width  = sizehints.max_width  = width;
    sizehints.height = sizehints.min_height = sizehints.max_height = height;
    XSetWMProperties (display, winid,
		      &nametext, &nametext,    // window and icon name
		      0, 0,		       // argv and argc (not used)
		      &sizehints,
		      0, 0);		       // WM and class hints (not used)
    XSelectInput (display, winid, ExposureMask);
    XEvent event;
    XEvent saveEvent;
    Bool eventSaved = False;
    event.type = NoExpose;
    XMapWindow (display, winid);
    while (event.type != Expose) {
      XNextEvent (display, &event);
#ifndef noSHM
      // Hack to save the CompletionType event for the other window
      if (event.type > LASTEvent) {
	memcpy((void *) &saveEvent, (void *) &event, sizeof(XEvent));
	eventSaved = True;
      }
#endif
    }
    XSelectInput(display, winid, NoEventMask);
    if (eventSaved) XPutBackEvent(display, &saveEvent);    
    _is_open = 1;
  };  
  // return (int)winid;
}

void
XWindowPlus::close ()
{
  if (_is_open) {
    destroy_image(Xim);
    XDestroyWindow (display, winid);
    XFreePixmap(display, buffer_pixmap);
    XFlush (display);
    XCloseDisplay(display);
  }
  delete [] data;
  _is_open = 0;
}

void
XWindowPlus::resize(int NewHoriz, int NewVert, char *name)
{ 
  // Update the associated XImage
  if ((NewHoriz > width) || (NewVert > height)) {
    // cerr << "destroying Ximage" << endl;
    destroy_image(Xim);
    // cerr << "creating new Ximage" << endl;
    Xim = create_image(NewHoriz,NewVert);
    // cerr << "destroying pixmap" << endl;
    XFreePixmap(display, buffer_pixmap);
    // cerr << "creating new pixmap" << endl;
    buffer_pixmap = XCreatePixmap(display, 
				  winid, 
				  NewHoriz, 
				  NewVert, 
				  depth);   
    // sleep(1); // give the XServer time to make the stuff
  }
  
  Set_Name(name); // change window title
  // cerr << "resizing window through WM" << endl;    
  XResizeWindow(display,winid,NewHoriz,NewVert);
  width = NewHoriz;
  height = NewVert;
}

XImage*
XWindowPlus::create_image(int width, int height)
{
  XImage *image;

#ifndef noSHM
  int ma, mi, px;
  candoshm = XShmQueryVersion(display,&ma,&mi,&px);
  image = TrySharedImage();
  candoshm = (image != 0);
  if (candoshm) {
    cout << "Using Shared Memory connection to server" << endl << flush;
  } else 
#endif    
  {
    int n = width*height;
    data = new char[(depth != 24) ? (n*(depth/8)) : (n*4)];
    cout << "width = " << width << " height = " << height << endl;
    image = XCreateImage (display, visual,
			  depth,		     // depth
			  ZPixmap,		     // format
			  0,		             // offset
			  (char *)data,
			  width, height,
			  (depth == 24) ? 32 : 8,    // bitmap_pad
			  0);		             // bytes_per_line (not used)
  }
  return image;
}
  
// putimage function
#include "putimage.icc"


int
XWindowPlus::destroy_image(XImage *im)
{
#ifndef noSHM
  if (candoshm) {
    XShmDetach(display,&IF);
    shmdt(IF.shmaddr);
    shmctl(IF.shmid, IPC_RMID, 0);
  }
#endif
  return  XDestroyImage (im);
}

int
XWindowPlus::ready_to_send_image()
{
#ifndef noSHM
  XEvent E;

//   int pending = XPending(display);
//   if (pending == 1) {
// 	  XNextEvent(display, &E);
// 	  if (E.xany.window != winid)
// 		  printf("event type = %d (%d)\n", E.type, E.xany.window);
// 	  XPutBackEvent(display, &E);
//   } else
//   if (pending) printf("(%d) Pending: %d\n", winid, pending);

  return ((!candoshm) ||
	  (!sent_image) || 
	  XCheckTypedWindowEvent(display, buffer_pixmap, CompletionType, &E));
#else
  return 1;
#endif
}

void
XWindowPlus::show(const Image &imag, int x, int y, 
		  int no_buffer, char *namein)
{
    // Make sure the window is open 
  if (!_is_open) 
    open(imag.width(),imag.height(),namein);

    // Resize the window to fit if its too small
  if ((width < imag.width()) || (height < imag.height()))
    resize(imag.width(),imag.height());

  Set_Name(namein);
  put_image(imag.data(), imag.color_scheme, x,y,imag.width(),imag.height(), 1, no_buffer);
}

void
XWindowPlus::show(int* imag, const src_color_scheme& color_scheme, 
		  int widim, int width_in, int height_in, 
		  int no_buffer, char *namein)
{
  // Make sure the window is open 
  if (!_is_open) 
    open(width_in,height_in,NULL);
  
  // Resize the window to fit if its too small
  if ((width < width_in) || (height < height_in))
    resize(width_in,height_in);
  
  Set_Name(namein);
  put_image(imag,color_scheme,0,0,width_in,height_in, 1, no_buffer);
}

void
XWindowPlus::show(void* imag, const src_color_scheme& color_scheme, 
		  int widim, int width_in, int height_in, 
		  int no_buffer, char *namein)
{
  // Make sure the window is open 
  if (!_is_open) 
    open(width_in,height_in,NULL);
  
  // Resize the window to fit if its too small
  if ((width < width_in) || (height < height_in))
    resize(width_in,height_in);
  
  Set_Name(namein);
  put_image(imag,color_scheme,0,0,width_in,height_in, 1, no_buffer);
}

void
XWindowPlus::clear(int pixel, int flushrequest)
{  		  
  if (is_open())
    {
      // color conversion happens here!      
      unsigned long draw_color = xv_get_color(pixel);
      XSetForeground (display, gc, draw_color);

      XFillRectangle(display,buffer_pixmap, gc,0,0,width,height);
      if (flushrequest) flush(flushrequest); 
    };
}

void
XWindowPlus::flush(int flushrequest)
{ 
  if (is_open())
    {
      XCopyArea(display, buffer_pixmap, winid, gc, 
		0, 0, width, height, 0, 0);            
      if (flushrequest) XFlush(display); 
    }
};

void
XWindowPlus::segment(int x1, int y1, int x2, int y2, int pixel)
{  	
  if (is_open())
    {	    
      XSetForeground (display, gc, pixel);
      XDrawLine (display, buffer_pixmap, gc, x1,y1,x2,y2);
    }
};


void
XWindowPlus::line(int x, int y, int length, float angle, int pixel)
{  		    
  if (is_open())
    {
      // color conversion happens here!      
      unsigned long draw_color = xv_get_color(pixel);
      XSetForeground (display, gc, draw_color);
      
      if (length > 0) {
	float rs = half (length) * sin (angle), rc = half (length) * cos (angle);
	XDrawLine (display, buffer_pixmap, gc,
		   round (x-rc), round (y-rs), round (x+rc), round (y+rs));
      } else {
	XDrawLine (display, buffer_pixmap, gc,
		   x + xwpcrlength, y - xwpcrlength, 
		   x - xwpcrlength, y + xwpcrlength);
	XDrawLine (display, buffer_pixmap, gc,
		   x - xwpcrlength, y - xwpcrlength, 
		   x + xwpcrlength, y + xwpcrlength);
      }
    }
};

void
XWindowPlus::cline(int x, int y, int length1, float lam1, float angle1,
	       int length2, float lam2, float angle2, int pixel)
{
  if (is_open())
    {
      // color conversion happens here!      
      unsigned long draw_color = xv_get_color(pixel);
      XSetForeground (display, gc, draw_color);
      
      float tx = x - lam1*cos(angle1);
      float ty = y - lam1*sin(angle1);
      
      line((int)tx,(int)ty,length1,angle1,pixel);
      
      tx = x - lam2*cos(angle2);
      ty = y - lam2*sin(angle2);
      
      line((int)tx,(int)ty,length2,angle2,pixel);
    }
};

//-----------------------------------------------------------------------------
//  Input functions
//-----------------------------------------------------------------------------

// getpos() checks whether the video type is IMG_SEQ
// if so, grab_static() will be called instead of grab()
// consuming image sequence in getpos() is not only wasteful
// but will also mess up the console since getpos() grabs gp_rows at a time
// -- J.W.

//* Method : XWindowPlus::getpos
//* Gets a point from a mouse click on an XWindowPlus
//* getpos() checks whether the video type is IMG_SEQ
//* if so, grab_static() will be called instead of grab()
//* consuming image sequence in getpos() is not only wasteful
//* but will also mess up the console since getpos() grabs gp_rows at a time


position
XWindowPlus::getpos(Video &v, char *name, int length)
{
  if (!is_open())
    open(v.width(),v.height(),name);
  else
    resize(v.width(),v.height(),name);

 if (is_undefined_color_scheme(v.color_scheme))
    {
      cout << "undefined color scheme in video class, please remedy this problem" << endl;
      exit(1);
    };

  int mmapped = (v.is_mappable() && (v.bytesperpixel() == 1));
  int *imdata = NULL;
  char *data;
  XEvent event;

  if (mmapped != 1) 
    imdata = new int[v.width() * v.height()];
  else
    cout << "Using memory mapping for image acquistion" << endl;

  position pos;
  pos.x = v.width() / 2;
  pos.y = v.height() / 2;
  pos.angle = 0;

  v.line (pos.x, pos.y, length, pos.angle);
  
  XSelectInput (display, winid,
                ButtonPressMask | ButtonReleaseMask | PointerMotionMask);

  while (1) {                                // main loop
      // First, check if there is any new data to display
    if (!mmapped) {
      v.grab_static (imdata, (v.width()/2),(v.height()/2), 
		     v.width(), v.height());
      put_image(imdata,v.color_scheme,0,0,v.width(),v.height(),1);
    } else {
      put_image((u_char *)(v.direct_memptr()),v.color_scheme,0,0,v.width(),v.height(),1);
    }

    line (pos.x, pos.y, length, pos.angle);

    flush();

    while (XCheckMaskEvent (display, ~0, &event)) {
      switch (event.type) {
      case ButtonPress:                           // left button exits
        if (event.xbutton.button == Button1) {
	  if (!mmapped)
	    delete imdata;
          XSelectInput (display, winid, NoEventMask);
          return pos;
        }
        break;
      case ButtonRelease:
        if (event.xbutton.button == Button2)
          XWarpPointer (display, 0, winid, 0, 0, 0, 0, pos.x, pos.y);
        break;
      case MotionNotify:
        clearline (pos.x, pos.y, length, pos.angle);
        v.clearline (pos.x, pos.y, length, pos.angle);
        if ((event.xmotion.state & Button2Mask) == 0) {
          pos.x = event.xmotion.x;
          pos.y = event.xmotion.y;
        } else {                             // middle button changes angle
          float diffx = event.xmotion.x - pos.x;
          float diffy = event.xmotion.y - pos.y;
          if (diffx != 0 || diffy != 0)
            pos.angle = atan2 (diffy, diffx);
        }
        line (pos.x, pos.y, length, pos.angle);
        v.line (pos.x, pos.y, length, pos.angle);
        break;
      default:
        break;
      }    // end switch
    }   // end while(XCheckEvent...)
  }  // end while(1)
  if (!mmapped)
    delete imdata;
  flush();
}

//* Method : XWindowPlus::get_corner_pos
//* Same as get_pos except drawing a corner cursor instead of a short line 
position
XWindowPlus::get_corner_pos(Video &v, char *name, int length)
{
  if (!is_open())
    open(v.width(),v.height(),name);
  else
    resize(v.width(),v.height(),name);

  if (is_undefined_color_scheme(v.color_scheme))
    {
      cout << "undefined color scheme in video class, please remedy this problem" << endl;
      exit(1);
    };


  position pos;
  pos.x = v.width()/2 ;
  pos.y = v.height()/2 ;
  pos.angle = 0;

  int cc = (int)(length*cos(pos.angle)/2);
  int ss = (int)(length*sin(pos.angle)/2);
  v.line (pos.x-cc, pos.y-ss, length, pos.angle);
  v.line (pos.x+ss, pos.y-cc, length, pos.angle-M_PI_2);
  
  int mmapped = (v.is_mappable() && (v.bytesperpixel() == 1));
  int *imdata = NULL;
  char *data;
  XEvent event;

  if (mmapped != 1) 
    imdata = new int[v.width() * v.height()];  //.display initial image
  else
    cout << "Using memory mapping for image acquiistion" << endl;

  XSelectInput (display, winid,
		ButtonPressMask | ButtonReleaseMask | PointerMotionMask);

  while (1) {				     // main loop

      // First, check if there is any new data to display
    if (!mmapped) {
      v.grab_static (imdata, (v.width()/2), (v.height()/2), 
	      v.width(), v.height());
      put_image(imdata,v.color_scheme,0,0,v.width(),v.height(),1);
    }
    else {
      put_image((u_char *)(v.direct_memptr()),v.color_scheme,0,0,v.width(),v.height(),1);
    }

    cc = (int)(length*cos(pos.angle)/2);
    ss = (int)(length*sin(pos.angle)/2);
    line (pos.x-cc, pos.y-ss, length, pos.angle);
    line (pos.x+ss, pos.y-cc, length, pos.angle-M_PI_2);

    flush();

    while (XCheckMaskEvent (display, ~0, &event)) {
      switch (event.type) {
      case ButtonPress:			     // left button exits
	if (event.xbutton.button == Button1) {
	  cc=(int)(length*cos(pos.angle)/2);
	  ss=(int)(length*sin(pos.angle)/2);
	  v.clearline (pos.x-cc, pos.y-ss, length, pos.angle);
	  v.clearline (pos.x+ss, pos.y-cc, length, pos.angle-M_PI_2);
	  if (!mmapped)
	    delete imdata;
	  //  XSelectInput (display, winid, NoEventMask);
	  //  return pos;
	}
	break;
      case ButtonRelease:
	if (event.xbutton.button == Button2)
	  XWarpPointer (display, 0, winid, 0, 0, 0, 0, pos.x, pos.y);
	if (event.xbutton.button == Button1) {
          XSelectInput (display, winid, NoEventMask);
	  return pos;
	}
	break;
      case MotionNotify:
        cc = (int)(length*cos(pos.angle)/2);
	ss = (int)(length*sin(pos.angle)/2);
	clearline (pos.x-cc, pos.y-ss, length, pos.angle);
	clearline (pos.x+ss, pos.y-cc, length, pos.angle-M_PI_2);
	v.clearline (pos.x-cc, pos.y-ss, length, pos.angle);
	v.clearline (pos.x+ss, pos.y-cc, length, pos.angle-M_PI_2);
	if ((event.xmotion.state & Button2Mask) == 0) {
	  pos.x = event.xmotion.x;
	  pos.y = event.xmotion.y;
	} else {			     // middle button changes angle
	  float diffx = event.xmotion.x - pos.x;
	  float diffy = event.xmotion.y - pos.y;
	  if (diffx != 0 || diffy != 0)
	    pos.angle = atan2 (diffy, diffx);
	}
	cc=(int)(length*cos(pos.angle)/2);
	ss=(int)(length*sin(pos.angle)/2);
	line (pos.x-cc, pos.y-ss, length, pos.angle);
	line (pos.x+ss, pos.y-cc, length, pos.angle-M_PI_2);
	v.line (pos.x-cc, pos.y-ss, length, pos.angle);
	v.line (pos.x+ss, pos.y-cc, length, pos.angle-M_PI_2);
	break;
      default:
	break;
      }  // end switch
    }  // end while(XCheckEvent...)
  }  // end while(1)
  if (!mmapped)
    delete imdata;
  flush();
}

//* Method : XWindowPlus::rectangle
//* An internally used function for drawing rectangles given
//* the two opposite corners of the rectangle
void
XWindowPlus::rectangle(int x1, int y1, int x2, int y2, Video &v, int color)
{
  if (is_open())
    {
      //Video::line() doesn't support negative length
      // 8.98: shouldn't Video::line() do this then???? sheesh. -cc
      if (x1>x2) 
  	{ int tmp=x2; x2=x1; x1=tmp; }
      if (y1>y2)
  	{ int tmp=y2; y2=y1; y1=tmp; }

      if (color==1) 
	{
	  // draw the rectangle on video
	  v.line( (x1+x2)/2, y1, x2-x1, 0); 
	  v.line( (x1+x2)/2, y2, x2-x1, 0);
	  v.line( x1, (y1+y2)/2, y2-y1, M_PI_2);
	  v.line( x2, (y1+y2)/2, y2-y1, M_PI_2);
	  
	  // draw the rectangle on screen
	  rectangle(x1, y1, x2 - x1, y2 - y1);
	} 
      else 
	{
	  // erase rectangle on video
	  v.clearline( (x1+x2)/2, y1, x2-x1, 0); 
	  v.clearline( (x1+x2)/2, y2, x2-x1, 0);
	  v.clearline( x1, (y1+y2)/2, y2-y1, M_PI_2);
	  v.clearline( x2, (y1+y2)/2, y2-y1, M_PI_2);
	  
	  // erase the rectangle on screen
	  rectangle(x1, y1, x2 - x1, y2 - y1);
	}
    }
};

void
XWindowPlus::rectangle(int x1, int y1, int width, int height, int color, bool filled)
{
  if (is_open())
    {
      // color conversion happens here!      
      unsigned long draw_color = xv_get_color(color);
      XSetForeground (display, gc, draw_color);
      if (!filled)
	{
	  XDrawRectangle(display,buffer_pixmap,gc,
			 x1, y1, width, height);    
	}
      else
	{
	  XFillRectangle(display,buffer_pixmap,gc,
			 x1, y1, width, height);
	}    	
    } 
};

//* Method : XWindowPlus::circle    
void
XWindowPlus::circle(int x, int y, int radius, int color, bool filled)
{
  if (is_open())
    {
      // color conversion happens here!      
      unsigned long draw_color = xv_get_color(color);
      XSetForeground (display, gc, draw_color);

      if (!filled)
	XDrawArc(display,buffer_pixmap,gc,x-radius,y-radius,2*radius,2*radius, 0,64*360);
      else
	XFillArc(display,buffer_pixmap,gc,x-radius,y-radius,2*radius,2*radius, 0,64*360);
    }
};

//* Method : XWindowPlus::get_region
//* Returns two positions: upper left and lower right in a position*
//* Should be implemented to return a region class, in order that
//* the caller not have to "delete" the pointer returned
//* I believe some of the calls to this function currently do not
//* delete it. The interface is point and drag
//* It is based on a modified version of getpos()
position*
XWindowPlus::get_region(Video &v, char *name)
{
  int length=1; //length of the cursor

  position* pos = new position[2]; // upper left and lower right
  pos[0].x = pos[1].x = v.width() / 2;
  pos[0].y = pos[1].y = v.height() / 2;
  pos[0].angle = pos[1].angle = 0;
  
  if (!is_open())
    open(v.width(),v.height(),name);
  else
    resize(v.width(),v.height(),name);

 if (is_undefined_color_scheme(v.color_scheme))
    {
      cout << "undefined color scheme in video class, please remedy this problem" << endl;
      exit(1);
    };


  int mmapped = (v.is_mappable() && (v.bytesperpixel() == 1));
  int *imdata = NULL;
  char *data;
  XEvent event;

  if (mmapped != 1) 
    imdata = new int[v.width() * v.height()]; 
  else
    cout << "Using memory mapping for image acquisstion" << endl;

  XSelectInput (display, winid,
                ButtonPressMask | ButtonReleaseMask | PointerMotionMask);


  while (1) {                                // main loop
    // First, check if there is any new data to display

    if (!mmapped) {
      v.grab_static (imdata, (v.width()/2),(v.height()/2), 
	      v.width(), v.height());
      put_image(imdata,v.color_scheme,0,0,v.width(),v.height(),1);
    }
    else {
      put_image((u_char *)(v.direct_memptr()),v.color_scheme,0,0,v.width(),v.height(),1);
    }
    
    rectangle( pos[0].x, pos[0].y, pos[1].x, pos[1].y, v, 1);
          					 //draw new rect

    flush();

    while (XCheckMaskEvent (display, ~0, &event)) {
      switch (event.type) {
      case ButtonPress:                      
        if (event.xbutton.button == Button1) ;
        break;
      case ButtonRelease:
        if (event.xbutton.button == Button1) { //exit
	  if (!mmapped)
	    delete imdata;
	  rectangle( pos[0].x, pos[0].y, pos[1].x, pos[1].y, v, 0);
	                                         //erase old rect
          XSelectInput (display, winid, NoEventMask);
          return pos;
        }
        break;
      case MotionNotify:
        if ( (event.xmotion.state & Button1Mask) == 0 )  {
          	v.clearline (pos[0].x, pos[0].y, length, pos[0].angle);
          	pos[0].x = event.xmotion.x;
          	pos[0].y = event.xmotion.y;
          	v.line(pos[0].x, pos[0].y, length, pos[0].angle);
          } else {
          	rectangle( pos[0].x, pos[0].y, pos[1].x, pos[1].y, v, 0);
          					 //erase old rect
          	pos[1].x = event.xmotion.x;
          	pos[1].y = event.xmotion.y;
          	rectangle( pos[0].x, pos[0].y, pos[1].x, pos[1].y, v, 1);
          					 //draw new rect
          }
        break;
      default:
        break;
      }
    }
  }
  if (!mmapped)
    delete imdata;
  flush();
}


//* Method : XWindowPlus::get_region
//* Returns two positions: upper left and lower right
//* Gets a region of constant size -- size_x * size_y
//* click once at the center to grab region and exit
position*
XWindowPlus::get_region(Video &v, int size_x, int size_y, char *name)
{
  int length=1; // length of cursor

  position *pos = new position[2]; // upper left and lower right
  pos[0].x =0; pos[1].x=size_x;
  pos[0].y =0; pos[1].y=size_y;
  pos[0].angle = pos[1].angle = 0;
  
  if (!is_open())
    open(v.width(),v.height(),name);
  else
    resize(v.width(),v.height(),name);

 if (is_undefined_color_scheme(v.color_scheme))
    {
      cout << "undefined color scheme in video class, please remedy this problem" << endl;
      exit(1);
    };

  int mmapped = (v.is_mappable() && (v.bytesperpixel() == 1));
  int *imdata = NULL;
  char *data;
  XEvent event;

  if (mmapped != 1) 
    imdata = new int[v.width() * v.height()];  //.display initial image
  else
    cout << "Using memory mapping for image acquiistion" << endl;


  XSelectInput (display, winid,
                ButtonPressMask | ButtonReleaseMask | PointerMotionMask);


  while (1) {                                // main loop

    // First, check if there is any new data to display

    if (!mmapped) {
      v.grab_static (imdata, (v.width()/2), (v.height()/2), 
	      v.width(), v.height());
      put_image(imdata,v.color_scheme,0,0,v.width(),v.height(),1);
    }
    else {
      put_image((u_char *)(v.direct_memptr()),v.color_scheme,0,0,v.width(),v.height(),1);
    }

    rectangle( pos[0].x, pos[0].y, pos[1].x, pos[1].y, v, 1);
          					 //draw new rect


    flush();
    // Now, handle the X stuff

    while (XCheckMaskEvent (display, ~0, &event)) {
      switch (event.type) {
      case ButtonPress:                      
        if (event.xbutton.button == Button1) ;
        break;
      case ButtonRelease:
        if (event.xbutton.button == Button1) {
	  if (!mmapped)
	    delete imdata;
	  rectangle( pos[0].x, pos[0].y, pos[1].x, pos[1].y, v, 0);
	                                         //erase old rect
          XSelectInput (display, winid, NoEventMask);
          return pos;
        }
        break;
      case MotionNotify:
        if ( event.xmotion.state== 0 )  {
          	rectangle( pos[0].x, pos[0].y, pos[1].x, pos[1].y, v, 0);
          					 //erase old rect
		pos[0].x = event.xmotion.x-size_x/2;
		pos[0].y = event.xmotion.y-size_y/2;
          	pos[1].x = event.xmotion.x+size_x/2;
          	pos[1].y = event.xmotion.y+size_y/2;
          	rectangle( pos[0].x, pos[0].y, pos[1].x, pos[1].y, v, 1);
          					 //draw new rect
          }
        break;
      default:
      // cerr << "Event type = " << event.type << " ( " << CompletionType << ")" << endl;
        break;
      }
    }
  }
  if (!mmapped) 
    delete imdata;
  flush();
}


//* Method : XWindowPlus::get_region_with_sampling
//* Returns two positions: upper left and lower right
//* Gets a region of constant size -- size_x * size_y
//* click once at the center to grab region and exit
//* subsamples the display window for more efficient display 
//* rates
position*
XWindowPlus::get_region_with_sampling(Video &v, int size_x, int size_y, 
				  int hs, int ws, char *name)
{
  int length=1; // length of cursor
  int window_width = v.width()/ws;
  int window_height = v.height()/hs;
  size_x /= ws;
  size_y /= hs;


  position *pos = new position[2]; // upper left and lower right
  pos[0].x =0; pos[1].x=size_x;
  pos[0].y =0; pos[1].y=size_y;
  pos[0].angle = pos[1].angle = 0;
  
  if (!is_open())
    open(window_width,window_height,name);
  else
    resize(window_width,window_height,name);

 if (is_undefined_color_scheme(v.color_scheme))
    {
      cout << "undefined color scheme in video class, please remedy this problem" << endl;
      exit(1);
    };

  int *imdata = NULL;
  char *data;
  XEvent event;

  imdata = new int[window_width * window_height];  //.display initial image
  
  XSelectInput (display, winid,
                ButtonPressMask | ButtonReleaseMask | PointerMotionMask);


  while (1) {                                // main loop

    v.grab_static (imdata, (v.width()/2), (v.height()/2), 
		   window_width, window_height, 0, ws, hs);
    
    put_image(imdata,v.color_scheme,0,0,window_width,window_height,1);


    rectangle( pos[0].x, pos[0].y, pos[1].x, pos[1].y, v, 1);
          					 //draw new rect

    flush();

    // Now, handle the X stuff

    while (XCheckMaskEvent (display, ~0, &event)) {
      switch (event.type) {
      case ButtonPress:                      
        if (event.xbutton.button == Button1) ;
        break;
      case ButtonRelease:
        if (event.xbutton.button == Button1) {
	  delete imdata;
	  rectangle( pos[0].x, pos[0].y, pos[1].x, pos[1].y, v, 0);
	                                         //erase old rect
          XSelectInput (display, winid, NoEventMask);
	  pos[0].x *= ws;
	  pos[1].x *= ws;
	  pos[0].y *= hs;
	  pos[1].y *= hs;
	  /* printf("get region returning (%d %d) (%d, %d)\n", 
		 pos[0].x, pos[0].y, 
		 pos[1].x, pos[1].y); */ 
          return pos;
        }
        break;
      case MotionNotify:
        if ( 1) {// event.xmotion.state== 0 )  {
          	rectangle( pos[0].x, pos[0].y, pos[1].x, pos[1].y, v, 0);
          					 //erase old rect
		pos[0].x = event.xmotion.x-size_x/2;
		pos[0].y = event.xmotion.y-size_y/2;
          	pos[1].x = event.xmotion.x+size_x/2;
          	pos[1].y = event.xmotion.y+size_y/2;
		/* printf("moving to (%d %d) (%d, %d)\n", 
		   pos[0].x, pos[0].y,
		   pos[1].x, pos[1].y); */
          	rectangle( pos[0].x, pos[0].y, pos[1].x, pos[1].y, v, 1);
          					 //draw new rect
          }
	else
	  printf("not moving to (%d %d) (%d, %d)\n", 
		 pos[0].x, pos[0].y,
		 pos[1].x, pos[1].y);
        break;
      default:
      // cerr << "Event type = " << event.type << " ( " << CompletionType << ")" << endl;
        break;
      }
    }
  }
  delete imdata;
  flush();
}


//* Method : XWindowPlus::get_multi_pos
//* Returns a list of positions, with each click defining a position
//* As with other list-returners, there really should be a class which
//* will automatically destruct itself so the caller does not have to
//* remember to delete the pointer returned
position*
XWindowPlus::get_multi_pos(Video &v, int* count,char *name, int length)
{
  (*count)=0;
  
  position* pos=(position *)calloc(1, sizeof(position) );
  pos[0].x = v.width() / 2;
  pos[0].y = v.height() / 2;
  pos[0].angle  = 0;

  // window stuff
    
  if (!is_open())
    open(v.width(),v.height(),name);
  else
    resize(v.width(),v.height(),name);

 if (is_undefined_color_scheme(v.color_scheme))
    {
      cout << "undefined color scheme in video class, please remedy this problem" << endl;
      exit(1);
    };


  int mmapped = (v.is_mappable() && (v.bytesperpixel() == 1));
  int *imdata = NULL;
  char *data;
  XEvent event;

  if (mmapped != 1) 
    imdata = new int[v.width() * v.height()];  //.display initial image
  else
    cout << "Using memory mapping for image acquiistion" << endl;


  XSelectInput (display, winid,
                ButtonPressMask | ButtonReleaseMask | PointerMotionMask);

  while (1) {                                // main loop

    // First, check if there is any new data to display

    if (!mmapped) {
      v.grab_static (imdata, (v.width()/2), (v.height()/2), 
	      v.width(), v.height());
      put_image(imdata,v.color_scheme,0,0,v.width(),v.height(),1);
    }
    else {
      put_image((u_char *)(v.direct_memptr()),v.color_scheme,0,0,v.width(),v.height(),1);
    }

    flush();

    while (XCheckMaskEvent (display, ~0, &event)) {
      switch (event.type) {
      case ButtonPress:                      
        if (event.xbutton.button == Button1) {
          (*count)++;
          pos=(position *) realloc( pos, sizeof(position)*((*count)+1) );
          	//start next click/position. the old one is remembered
        }
        break;
      case ButtonRelease:
        if (event.xbutton.button == Button3) { //exit
	  if (!mmapped)
	    delete imdata;
          XSelectInput (display, winid, NoEventMask);
          return pos;
        }
        break;
      case MotionNotify:
        if ( event.xmotion.state == 0 )  {
          	v.clearline (pos[*count].x, pos[*count].y, 
          			length, pos[*count].angle);
          	pos[(*count)].x = event.xmotion.x;
          	pos[(*count)].y = event.xmotion.y;
          	v.line(pos[*count].x, pos[*count].y, 
          			length, pos[*count].angle);
          } 
        break;
      default:
        break;
      }
    }
  }
  if (!mmapped)
    delete imdata;
  flush();
}


//* Method : XWindowPlus::TrySharedImage
//*
//* This code originally from the mesa (public domain OpenGL) source...
//* 
//* Not only does this source gracefully exit when a shared memory image
//* can't be created, but it also doesn't leave any shared memory segments
//* hanging around
//*
//* Also defined is a static int and static function for the case
//* where there is no Shared Memory
#ifndef noSHM
static int S_Xerror;
static int S_handleXerror(Display*,XErrorEvent*)
{
  S_Xerror = 1;
  return 0;
}
#endif

XImage*
XWindowPlus::TrySharedImage()
{
#ifndef noSHM
   XImage *image;
   Display *display = display;

   image = XShmCreateImage(display,
                           visual,
			   depth,
                           ZPixmap,
			   NULL,
			   &IF,
			   width, height);
   if (image == NULL) {
      cerr << "XShmCreateImage problem" << endl;
      return 0;
   }

   // create the shared memory segment
  IF.shmid = shmget(IPC_PRIVATE,
                          image->bytes_per_line * height,
                          IPC_CREAT | 0777);
  IF.readOnly = False;
  if (IF.shmid < 0) {
     cerr << "shmget() problem" << endl;
     return 0;
  }

  data = (char *) (IF.shmaddr = image->data = (char *) shmat(IF.shmid, 0, 0));

  if (data  == (char *) -1) {
    cerr << "shmat() problem" << endl;
    return 0;
  }

  XSync(display, False);

  // set the x error handler 
  int (*old_handler)(Display *, XErrorEvent *);
  old_handler = XSetErrorHandler(S_handleXerror);

  // we need
  //   to be especially careful with this call, because it may fail
  //   if X is being used remotely

  // have the server attach to the shared memory segment 
  S_Xerror = 0;
  XShmAttach(display, &IF);
  XSync(display, False);

  if (S_Xerror) {
    // there was an error; can't use shared memory images
    XSetErrorHandler(old_handler);

    // detach from the segment
    shmdt(IF.shmaddr);

    // remove the shmid of the shared memory segment
    shmctl(IF.shmid, IPC_RMID, 0);

    return 0;
  }

  // remove the shmid of the shared memory segment; no one else
  //   needs to attach to it, so it will disappear once the client
  //   quits
  shmctl(IF.shmid, IPC_RMID, 0);

  // reset the X error handler
  XSetErrorHandler(old_handler);

  CompletionType = XShmGetEventBase(display) +  ShmCompletion;
  sent_image = 0;

  return image;
#else
  // We aren't doing shared memory, so return 0
  return 0;
#endif
}



















