/*                                                                -*-c++-*-
    Copyright (C) 1991 Gregory D. Hager and Sidd Puri (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//-----------------------------------------------------------------------------
//  XWindow.cc implementation of XWindow class
// 
//  08/01/94 Gregory D. Hager
//  07/05/95 Jonathan Wang
//  07/25/96 Gregory D. Hager
//  Added color support.  Now a console can be opened as color or
//  monochrome.  At the moment, however, all of the attached windows
//  inherit this from the console, so mixed windows are not possible.
//  Also if you try to display a monochrome image on a color window,
//  you'll get a very blue looking image.........
//  Added 16 bit color support.
//
//  11/25/96 Loring Holden (lsh@cs.brown.edu)
//  Added support for 24 bit color when 24 bit color is not default visual
//  Fixed shared memory support so shared mem segments won't hang around
//
//-----------------------------------------------------------------------------

#include <X11/Xlib.h>
#include <X11/extensions/XShm.h>
#include <X11/cursorfont.h>
#include <X11/Xutil.h>
#include <string.h>	
#include <stdlib.h>	
#include "XWindow.hh"
#include "rate.hh"

// GDH added these
#ifndef noSHM
#include <sys/ipc.h>
#include <sys/shm.h>
#endif

inline static
char*
alloc_data(int n, int depth)
{
  if (depth < 24)  {return new char[n*depth>>3];}
  else             {return new char[n*4];}
}

XWindow::XWindow(XConsole& x)          // constructor
{
  c = &x;
  video_device = NULL;
  gc_draw = 0;
}
XWindow::XWindow(XConsole& x, char *name)          // constructor
{
  c = &x;
  video_device = NULL;
  gc_draw = 0;
  Set_Name(name);
}

//XConsole* XWindow::defaultConsole = new XConsole;
//
//XWindow::XWindow()                     // default constructor
//{
//  c = XWindow::defaultConsole;
//}

void 
XWindow::Set_Name(char *name)
{
  if (name != NULL) {
    XTextProperty nametext;
    XStringListToTextProperty (&name, 1, &nametext);
    XSetWMName (c->display, winid, &nametext);
    XSetWMIconName (c->display, winid, &nametext);
  }
}

void
XWindow::open(int width_in, int height_in, char *name,char *displayname)
{
  if (!_is_open) {
    XSetWindowAttributes attributes;
    attributes.cursor = XCreateFontCursor(c->display, XC_crosshair);
    unsigned long mask;
    mask = CWCursor;

    if (c->depth == 24) {
      mask = mask | CWBackPixel | CWBorderPixel | CWColormap;
      attributes.background_pixel = BlackPixel(c->display, c->screen);
      attributes.border_pixel = WhitePixel(c->display, DefaultScreen(c->display));
      attributes.colormap = c->colormap;
    }
    winid = XCreateWindow (c->display, c->root,
			   0, 0,		     // position (not used)
			   width_in, height_in,
			   1,			     // border width
			   c->depth,	             // depth
			   InputOutput,	             // class
			   c->visual,
			   mask,		     // value mask
			   &attributes);
    if (winid == 0) {cerr << ("Could not create X window"); exit(1);}
    if (c->depth == 24) {c->gc = XCreateGC(c->display, winid, 0, 0);}

    XTextProperty nametext;
    XStringListToTextProperty (&name, 1, &nametext);
    XSizeHints sizehints;
    sizehints.flags = PSize | PMinSize | PMaxSize;
    sizehints.width  = sizehints.min_width  = sizehints.max_width  = width_in;
    sizehints.height = sizehints.min_height = sizehints.max_height = height_in;
    XSetWMProperties (c->display, winid,
		      &nametext, &nametext,    // window and icon name
		      0, 0,		       // argv and argc (not used)
		      &sizehints,
		      0, 0);		       // WM and class hints (not used)
    Set_Name(name);
    XSelectInput (c->display, winid, ExposureMask);
    XEvent event;
    XEvent saveEvent;
    Bool eventSaved = False;
    XMapWindow (c->display, winid);
    do
    {
      XNextEvent (c->display, &event);
#ifndef noSHM
        // Hack to save the CompletionType event for the other window
      if (event.type > LASTEvent) {
	memcpy((void *) &saveEvent, (void *) &event, sizeof(XEvent));
	eventSaved = True;
      }
#endif
    }while (event.type != Expose);
    XSelectInput(c->display, winid, NoEventMask);
    if (eventSaved) XPutBackEvent(c->display, &saveEvent);

    XWindowAttributes attributes1;
    XGetWindowAttributes(c->display, winid, &attributes1);
    depth  = attributes1.depth;
    width  = attributes1.width;
    height = attributes1.height;
    _is_open = 1;
  }
  // Since we only use window for display, create an image structure
  // now and keep it around.
  Xim = create_image(width,height);
  //return (int)winid; // I don't think this is needed - ADR
}

void
XWindow::close ()
{
  if (_is_open) {
    destroy_image(Xim);
    XDestroyWindow (c->display, winid);
    XFlush (c->display);
  }
  _is_open = 0;
}

void
XWindow::resize(int NewHoriz, int NewVert, char *name)
{ 
    // Update the associated XImage
  if ((NewHoriz > width) || (NewVert > height)) {
    destroy_image(Xim);
    sleep(1);
    Xim = create_image(NewHoriz,NewVert);
    sleep(1);
  }
  Set_Name(name);
  XResizeWindow(c->display,winid,NewHoriz,NewVert);

  XWindowAttributes attributes1;
  XGetWindowAttributes (c->display, winid, &attributes1);
  depth = attributes1.depth;
  width  = attributes1.width;
  height = attributes1.height;
}

XImage*
XWindow::create_image(int width, int height)
{
  XImage *image;

#ifndef noSHM
  int ma, mi, px;
  candoshm = XShmQueryVersion(c->display,&ma,&mi,&px);
  image = TrySharedImage();
  candoshm = (image != 0);
  if (candoshm) {
    //    cout << "Using Shared Memory connection to server" << endl << flush;
  } else 
#endif    
  {
    data = alloc_data(width * height,depth);
    cout << "width = " << width << " height = " << height << endl;
    image = XCreateImage (c->display, c->visual,
			  depth,		     // depth
			  ZPixmap,		     // format
			  0,		             // offset
			  (char *)data,
			  width, height,
			  (depth == 24) ? 32 : 8,    // bitmap_pad
			  0);		             // bytes_per_line (not used)
  }
  return image;
}

int
XWindow::put_image(int *imdata,int x, int y, int width, int height, int rete)
{
  Xim->width = width;
  Xim->height = height;
  Xim->bytes_per_line = width * ((depth == 24) ? 4 : depth / 8);
#ifndef noSHM
  if (candoshm) {
      if (ready_to_send_image()) {
        c->map(imdata,Xim,width * height,depth);
        sent_image = 1;
        return XShmPutImage (c->display, winid, c->gc, Xim,
			     0, 0, x, y, width, height, rete);
      }
  } else
#endif
    {
      c->map(imdata, Xim, width * height, depth);
      return XPutImage (c->display, winid, c->gc, Xim, 0, 0, x, y, width, height);
    }
}


int
XWindow::put_image(u_char *imdata,int x, int y, int width, int height, int rete)
{
  Xim->width = width;
  Xim->height = height;
  Xim->bytes_per_line = width * ((depth == 24) ? 4 : depth / 8);

#ifndef noSHM
  if (candoshm) {
      if (ready_to_send_image()) {
      c->map(imdata,Xim,width * height,depth);
      sent_image = 1;
      return XShmPutImage (c->display, winid, c->gc, Xim,
			   0, 0, x, y, width, height, rete);
      }
  } else
#endif
    {
      c->map(imdata,Xim,width * height,depth);
      return XPutImage (c->display, winid, c->gc, Xim, 0, 0, x, y, width, height);
    }
}

int
XWindow::destroy_image(XImage *im)
{
#ifndef noSHM
  if (candoshm) {
    XShmDetach(c->display,&IF);
    shmdt(IF.shmaddr);
    shmctl(IF.shmid, IPC_RMID, 0);
  }
#endif
  return  XDestroyImage (im);
}

int
XWindow::ready_to_send_image()
{
#ifndef noSHM
  XEvent E;

//   int pending = XPending(c->display);
//   if (pending == 1) {
// 	  XNextEvent(c->display, &E);
// 	  if (E.xany.window != winid)
// 		  printf("event type = %d (%d)\n", E.type, E.xany.window);
// 	  XPutBackEvent(c->display, &E);
//   } else
//   if (pending) printf("(%d) Pending: %d\n", winid, pending);

  return ((!candoshm) ||
	  (!sent_image) || 
	  XCheckTypedWindowEvent(c->display, winid, CompletionType, &E));
#else
  return 1;
#endif
}

void
XWindow::show(const Image &imag, char *namein, int x, int y)
{
    // Make sure the window is open 
  if (!_is_open) 
    open(imag.width(),imag.height(),namein);

    // Resize the window to fit if its too small
  if ((width < imag.width()) || (height < imag.height()))
    resize(imag.width(),imag.height());

  Set_Name(namein);
  put_image(imag.data(),x,y,imag.width(),imag.height(),1);
  flush();
}

void
XWindow::show(int* imag, int widim, char *namein, int width_in, int height_in)
{
    // Make sure the window is open 
  if (!_is_open) 
    open(width_in,height_in,NULL);
  
    // Resize the window to fit if its too small
  if ((width < width_in) || (height < height_in))
    resize(width_in,height_in);

  Set_Name(namein);
  put_image(imag,0,0,width_in,height_in,1);
}

// HACK HACK HACK for now --- GDH

static int lastxgrab = 0,lastygrab = 0,lastwgrab = 0,lasthgrab = 0;
static TimingInfo Tm(100);

void XWindow::show_grab(Video *v, float interval)
{
  v->frames--;
  if (video_device == NULL)
    {     
      if (v->compute_envelope_parameters_on_grab())
	{
	  show_grab_data = new int[v->width() * v->height()];       
	  video_device = v;
	  // cout << " video device set in XWindow" << endl;
	};
      resize(v->width(), v->height());
      pixmap = XCreatePixmap(c->display, winid, v->width(), v->height(), c->depth);       
      XGCValues gcvalues;
      gcvalues.foreground = BlackPixelOfScreen(DefaultScreenOfDisplay(c->display));
      gcvalues.background = BlackPixelOfScreen(DefaultScreenOfDisplay(c->display));
      gc_clear = XCreateGC(c->display, pixmap, GCForeground | GCBackground, &gcvalues);
    }
  else
    {

      if (lastwgrab == 0) {
	Tm.set_deadline(interval); 
	XFillRectangle(c->display, winid, gc_clear, 
		       0, 0, v->width(),v->height());

	lastxgrab = v->envelope_x - v->envelope_width/2; 
	lastygrab = v->envelope_y - v->envelope_height/2; 
	lastwgrab = v->envelope_width;
	lasthgrab = v->envelope_height;
	  
	v->grab_static (show_grab_data, v->envelope_x, v->envelope_y,
			v->envelope_width, v->envelope_height);
	
	put_image(show_grab_data,
		  lastxgrab, 
		  lastygrab, 
		  lastwgrab, 
		  lasthgrab, 
		  1);
      }
      else {
	if (Tm.alarm(1)) {

	  XFillRectangle(c->display, winid, gc_clear, 
			 lastxgrab, lastygrab, lastwgrab,lasthgrab);

	  lastxgrab = v->envelope_x - v->envelope_width/2;
	  lastygrab = v->envelope_y - v->envelope_height/2;
	  lastwgrab = v->envelope_width;
	  lasthgrab = v->envelope_height;

	    
	  v->grab_static (show_grab_data, v->envelope_x, v->envelope_y,
			  v->envelope_width, v->envelope_height);
	  put_image(show_grab_data,
		    lastxgrab , 
		    lastygrab , 
		    lastwgrab , 
		    lasthgrab , 1);

      
	    //      XSetForeground (c->display, c->gc, c->redpixel);

	    //      XDrawRectangle(c->display,pixmap,c->gc,
	    //		     v->envelope_x - v->envelope_width/2, 
	    //		     v->envelope_y - v->envelope_height/2, 
	    //		     v->envelope_width, v->envelope_height); 
	}
      }
    }
}

void
XWindow::clear(int pixel)
{  		    
    XSetForeground(c->display, c->gc, pixel);
    XFillRectangle(c->display,winid, c->gc,0,0,width,height);
    XFlush(c->display); 
}

void
XWindow::flush(void)
{  		    
    XFlush(c->display); 
}

void
XWindow::segment(int x1, int y1, int x2, int y2, int pixel)
{  		    
    XSetForeground (c->display, c->gc, pixel);
    XDrawLine (c->display, winid, c->gc, x1,y1,x2,y2);
}

void
XWindow::line(int x, int y, int length, float angle, int pixel)
{  		    
    if (pixel == -1) pixel = c->redpixel;
    XSetForeground (c->display, c->gc, pixel);

    if (length > 0) {
      float rs = half (length) * sin (angle), rc = half (length) * cos (angle);
      XDrawLine (c->display, winid, c->gc,
	       round (x-rc), round (y-rs), round (x+rc), round (y+rs));
    } else {
      XDrawLine (c->display, winid, c->gc,
	       x + cr_length, y - cr_length, x - cr_length, y + cr_length);
      XDrawLine (c->display, winid, c->gc,
	       x - cr_length, y - cr_length, x + cr_length, y + cr_length);
    }
}

void
XWindow::cline(int x, int y, int length1, float lam1, float angle1,
	       int length2, float lam2, float angle2, int pixel)
{
  if (pixel == -1) pixel = c->redpixel;
  XSetForeground (c->display, c->gc, pixel);

  int tx = int(x - lam1*cos(angle1));
  int ty = int(y - lam1*sin(angle1));

  line(tx,ty,length1,angle1,pixel);

  tx = int(x - lam2*cos(angle2));
  ty = int(y - lam2*sin(angle2));

  line(tx,ty,length2,angle2,pixel);
}

//-----------------------------------------------------------------------------
//  Input functions
//-----------------------------------------------------------------------------

// getpos() checks whether the video type is IMG_SEQ
// if so, grab_static() will be called instead of grab()
// consuming image sequence in getpos() is not only wasteful
// but will also mess up the console since getpos() grabs gp_rows at a time
// -- J.W.

//* Method : XWindow::getpos
//* Gets a point from a mouse click on an XWindow
//* getpos() checks whether the video type is IMG_SEQ
//* if so, grab_static() will be called instead of grab()
//* consuming image sequence in getpos() is not only wasteful
//* but will also mess up the console since getpos() grabs gp_rows at a time


position
XWindow::getpos(Video &v, char *name, int length,int dovidupdate)
{
  if (!is_open())
    open(v.width(),v.height(),name);
  else
    resize(v.width(),v.height(),name);

  Set_Name(name);

  int mmapped = (v.is_mappable() && (v.bytesperpixel() == 1));
  int *imdata;
  XEvent event;

  if (mmapped != 1) 
    imdata = new int[v.width() * v.height()];
  else
    cout << "Using memory mapping for image acquisition" << endl;

  position pos;
  pos.x = v.width() / 2;
  pos.y = v.height() / 2;
  pos.angle = 0;

  v.line (pos.x, pos.y, length, pos.angle);

  XSelectInput (c->display, winid,
                ButtonPressMask | ButtonReleaseMask | PointerMotionMask);

  while (1) {                                // main loop
      // First, check if there is any new data to display
    if (!mmapped) {
      v.grab_static (imdata, v.width()/2, v.height()/2, 
		     v.width(), v.height());
      put_image(imdata,0,0,v.width(),v.height(),1);
    } else {
      put_image((u_char *)(v.direct_memptr()),0,0,v.width(),v.height(),1);
    }

    line (pos.x, pos.y, length, pos.angle);

    while (XCheckMaskEvent (c->display, ~0, &event)) {
      switch (event.type) {
      case ButtonPress:                           // left button exits
        if (event.xbutton.button == Button1) {
	  if (!mmapped)
	    delete imdata;
          XSelectInput (c->display, winid, NoEventMask);
          return pos;
        }
        break;
      case ButtonRelease:
        if (event.xbutton.button == Button2)
          XWarpPointer (c->display, 0, winid, 0, 0, 0, 0, pos.x, pos.y);
        break;
      case MotionNotify:
        clearline (pos.x, pos.y, length, pos.angle);
        v.clearline (pos.x, pos.y, length, pos.angle);
        if ((event.xmotion.state & Button2Mask) == 0) {
          pos.x = event.xmotion.x;
          pos.y = event.xmotion.y;
        } else {                             // middle button changes angle
          float diffx = event.xmotion.x - pos.x;
          float diffy = event.xmotion.y - pos.y;
          if (diffx != 0 || diffy != 0)
            pos.angle = atan2 (diffy, diffx);
        }
        line (pos.x, pos.y, length, pos.angle);
        v.line (pos.x, pos.y, length, pos.angle);
        break;
      default:
        break;
      }    // end switch
    }   // end while(XCheckEvent...)
  }  // end while(1)
  if (!mmapped)
    delete imdata;
  flush();
}

//* Method : XWindow::get_corner_pos
//* Same as get_pos except drawing a corner cursor instead of a short line 
position
XWindow::get_corner_pos(Video &v, char *name, int length,int dovidupdate)
{
  if (!is_open())
    open(v.width(),v.height(),name);
  else
    resize(v.width(),v.height(),name);

  Set_Name(name);

  position pos;
  pos.x = v.width()/2 ;
  pos.y = v.height()/2 ;
  pos.angle = 0;

  int cc=int(length*cos(pos.angle)/2), ss=int(length*sin(pos.angle)/2);
  v.line (pos.x-cc, pos.y-ss, length, pos.angle);
  v.line (pos.x+ss, pos.y-cc, length, pos.angle-M_PI_2);
  
  int mmapped = (v.is_mappable() && (v.bytesperpixel() == 1));
  int *imdata;
  XEvent event;

  if (mmapped != 1) 
    imdata = new int[v.width() * v.height()];  //.display initial image
  else
    cout << "Using memory mapping for image acquisition" << endl;

  XSelectInput (c->display, winid,
		ButtonPressMask | ButtonReleaseMask | PointerMotionMask);

  while (1) {				     // main loop

      // First, check if there is any new data to display
    if (!mmapped) {
      v.grab_static (imdata, v.width()/2, v.height()/2, 
		     v.width(), v.height());
      put_image(imdata,0,0,v.width(),v.height(),1);
    }
    else {
      put_image((u_char *)(v.direct_memptr()),0,0,v.width(),v.height(),1);
    }

    cc=int(length*cos(pos.angle)/2), ss=int(length*sin(pos.angle)/2);
    line (pos.x-cc, pos.y-ss, length, pos.angle);
    line (pos.x+ss, pos.y-cc, length, pos.angle-M_PI_2);

    while (XCheckMaskEvent (c->display, ~0, &event)) {
      switch (event.type) {
      case ButtonPress:			     // left button exits
	if (event.xbutton.button == Button1) {
	  cc=int(length*cos(pos.angle)/2), ss=int(length*sin(pos.angle)/2);
	  v.clearline (pos.x-cc, pos.y-ss, length, pos.angle);
	  v.clearline (pos.x+ss, pos.y-cc, length, pos.angle-M_PI_2);
	  // This code is completely bogus: killing imdata before its last use
	  // But you don't notice the bug if you click really really fast.
	  //  if (!mmapped)
	  //  	delete imdata;
	  //  XSelectInput (c->display, winid, NoEventMask);
	  //  return pos;
	}
	break;
      case ButtonRelease:
	if (event.xbutton.button == Button2)
	  XWarpPointer (c->display, 0, winid, 0, 0, 0, 0, pos.x, pos.y);
	if (event.xbutton.button == Button1) {
          XSelectInput (c->display, winid, NoEventMask);
	  return pos;
	}
	break;
      case MotionNotify:
        cc=int(length*cos(pos.angle)/2), ss=int(length*sin(pos.angle)/2);
	clearline (pos.x-cc, pos.y-ss, length, pos.angle);
	clearline (pos.x+ss, pos.y-cc, length, pos.angle-M_PI_2);
	v.clearline (pos.x-cc, pos.y-ss, length, pos.angle);
	v.clearline (pos.x+ss, pos.y-cc, length, pos.angle-M_PI_2);
	if ((event.xmotion.state & Button2Mask) == 0) {
	  pos.x = event.xmotion.x;
	  pos.y = event.xmotion.y;
	} else {			     // middle button changes angle
	  float diffx = event.xmotion.x - pos.x;
	  float diffy = event.xmotion.y - pos.y;
	  if (diffx != 0 || diffy != 0)
	    pos.angle = atan2 (diffy, diffx);
	}
	cc=int(length*cos(pos.angle)/2), ss=int(length*sin(pos.angle)/2);
	line (pos.x-cc, pos.y-ss, length, pos.angle);
	line (pos.x+ss, pos.y-cc, length, pos.angle-M_PI_2);
	v.line (pos.x-cc, pos.y-ss, length, pos.angle);
	v.line (pos.x+ss, pos.y-cc, length, pos.angle-M_PI_2);
	break;
      default:
	break;
      }  // end switch
    }  // end while(XCheckEvent...)
  }  // end while(1)
  if (!mmapped)
    delete imdata;
  flush();
}

//* Method : XWindow::rectangle
//* An internally used function for drawing rectangles given
//* the two opposite corners of the rectangle
void
XWindow::rectangle(int x1, int y1, int x2, int y2, Video &v, int color)
{
  
    //Video::line() doesn't support negative length
  if (x1>x2) 
  	{ int tmp=x2; x2=x1; x1=tmp; }
  if (y1>y2)
  	{ int tmp=y2; y2=y1; y1=tmp; }
  
  
  if (color==1) 
    {
      // draw the rectangle on video
      v.line( (x1+x2)/2, y1, x2-x1, 0); 
      v.line( (x1+x2)/2, y2, x2-x1, 0);
      v.line( x1, (y1+y2)/2, y2-y1, M_PI_2);
      v.line( x2, (y1+y2)/2, y2-y1, M_PI_2);
      
      // draw the rectangle on screen
      rectangle(x1, y1, x2 - x1, y2 - y1);
    } 
  else 
    {
      // erase rectangle on video
      v.clearline( (x1+x2)/2, y1, x2-x1, 0); 
      v.clearline( (x1+x2)/2, y2, x2-x1, 0);
      v.clearline( x1, (y1+y2)/2, y2-y1, M_PI_2);
      v.clearline( x2, (y1+y2)/2, y2-y1, M_PI_2);
      
      // erase the rectangle on screen
      rectangle(x1, y1, x2 - x1, y2 - y1);
    }
}

void
XWindow::rectangle(int x1, int y1, int width, int height, int color, bool filled)
{
  if (!gc_draw)
    {
      XGCValues gcvalues;
      XGetGCValues(c->display, c->gc, 
		   GCFunction | GCForeground | GCBackground, 
		   &gcvalues);
      gcvalues.function = GXxor;
      gcvalues.foreground = color;
      gc_draw = XCreateGC (c->display, winid, 
			   GCFunction | GCForeground | GCBackground,
			   &gcvalues);
     // change the graphics context so that drawing is done in Xor mode
     // this should make for much easier erasing
     // grabs the standard graphics context
    }

  if (color == -1) color = c->redpixel;
  XSetForeground (c->display, gc_draw, color);
  if (!filled)
    {
      XDrawRectangle(c->display,winid,gc_draw,
		     x1, y1, width, height);    
    }
  else
    {
      XDrawRectangle(c->display,winid,gc_draw,
		     x1, y1, width, height);
      XFillRectangle(c->display,winid,gc_draw,
		     x1, y1, width, height);
    }    	
} 

//* Method : XWindow::circle    
void
XWindow::circle(int x, int y, int radius, int color, bool filled)
{
  if (!gc_draw)
    {
      XGCValues gcvalues;
      XGetGCValues(c->display, c->gc, 
		   GCFunction | GCForeground | GCBackground, 
		   &gcvalues);
      gcvalues.function = GXxor;
      gc_draw = XCreateGC (c->display, winid, 
			   GCFunction | GCForeground | GCBackground,
			   &gcvalues);
     // change the graphics context so that drawing is done in Xor mode
     // this should make for much easier erasing
     // grabs the standard graphics context
    }

  if (color == -1) color = c->redpixel;
  XSetForeground (c->display, gc_draw, color);
  if (!filled)
    XDrawArc(c->display,winid,gc_draw,x-radius,y-radius,2*radius,2*radius, 0,64*360-1);
  else
    XFillArc(c->display,winid,gc_draw,x-radius,y-radius,2*radius,2*radius, 0,64*360-1);
}

//* Method : XWindow::get_region
//* Returns two positions: upper left and lower right in a position*
//* Should be implemented to return a region class, in order that
//* the caller not have to "delete" the pointer returned
//* I believe some of the calls to this function currently do not
//* delete it. The interface is point and drag
//* It is based on a modified version of getpos()
position*
XWindow::get_region(Video &v, char *name, int dovidupdate)
{
  int length=1; //length of the cursor

  position* pos = new position[2]; // upper left and lower right
  pos[0].x = pos[1].x = v.width() / 2;
  pos[0].y = pos[1].y = v.height() / 2;
  pos[0].angle = pos[1].angle = 0;
  
  if (!is_open())
    open(v.width(),v.height(),name);
  else
    resize(v.width(),v.height(),name);

  Set_Name(name);

  int mmapped = (v.is_mappable() && (v.bytesperpixel() == 1));
  int *imdata;
  XEvent event;

  if (mmapped != 1) 
    imdata = new int[v.width() * v.height()]; 
  else
    cout << "Using memory mapping for image acquisition" << endl;

  XSelectInput (c->display, winid,
                ButtonPressMask | ButtonReleaseMask | PointerMotionMask);


  while (1) {                                // main loop
    // First, check if there is any new data to display

    if (!mmapped) {
      v.grab_static (imdata, v.width()/2,v.height()/2, 
	      v.width(), v.height());
      put_image(imdata,0,0,v.width(),v.height(),1);
    }
    else {
      put_image((u_char *)(v.direct_memptr()),0,0,v.width(),v.height(),1);
    }
    
    rectangle( pos[0].x, pos[0].y, pos[1].x, pos[1].y, v, 1);
          					 //draw new rect

    while (XCheckMaskEvent (c->display, ~0, &event)) {
      switch (event.type) {
      case ButtonPress:                      
        if (event.xbutton.button == Button1) ;
        break;
      case ButtonRelease:
        if (event.xbutton.button == Button1) { //exit
	  if (!mmapped)
	    delete imdata;
	  rectangle( pos[0].x, pos[0].y, pos[1].x, pos[1].y, v, 0);
	                                         //erase old rect
          XSelectInput (c->display, winid, NoEventMask);
          return pos;
        }
        break;
      case MotionNotify:
        if ( (event.xmotion.state & Button1Mask) == 0 )  {
          	v.clearline (pos[0].x, pos[0].y, length, pos[0].angle);
          	pos[0].x = event.xmotion.x;
          	pos[0].y = event.xmotion.y;
          	v.line(pos[0].x, pos[0].y, length, pos[0].angle);
          } else {
          	rectangle( pos[0].x, pos[0].y, pos[1].x, pos[1].y, v, 0);
          					 //erase old rect
          	pos[1].x = event.xmotion.x;
          	pos[1].y = event.xmotion.y;
          	rectangle( pos[0].x, pos[0].y, pos[1].x, pos[1].y, v, 1);
          					 //draw new rect
          }
        break;
      default:
        break;
      }
    }
  }
  if (!mmapped)
    delete imdata;
  flush();
}


//* Method : XWindow::get_region
//* Returns two positions: upper left and lower right
//* Gets a region of constant size -- size_x * size_y
//* click once at the center to grab region and exit
position*
XWindow::get_region(Video &v, int size_x, int size_y, char *name,
		    int dovidupdate)
{
  position *pos = new position[2]; // upper left and lower right
  pos[0].x =0; pos[1].x=size_x;
  pos[0].y =0; pos[1].y=size_y;
  pos[0].angle = pos[1].angle = 0;
  
  if (!is_open())
    open(v.width(),v.height(),name);
  else
    resize(v.width(),v.height(),name);

  Set_Name(name);

  int mmapped = (v.is_mappable() && (v.bytesperpixel() == 1));
  int *imdata;
  XEvent event;

  if (mmapped != 1) 
    imdata = new int[v.width() * v.height()];  //.display initial image
  else
    cout << "Using memory mapping for image acquisition" << endl;


  XSelectInput (c->display, winid,
                ButtonPressMask | ButtonReleaseMask | PointerMotionMask);


  while (1) {                                // main loop

    // First, check if there is any new data to display

    if (!mmapped) {
      v.grab_static (imdata, v.width()/2, v.height()/2, 
		     v.width(), v.height());
      put_image(imdata,0,0,v.width(),v.height(),1);
    }
    else {
      put_image((u_char *)(v.direct_memptr()),0,0,v.width(),v.height(),1);
    }

    rectangle( pos[0].x, pos[0].y, pos[1].x, pos[1].y, v, 1);
          					 //draw new rect

    // Now, handle the X stuff

    while (XCheckMaskEvent (c->display, ~0, &event)) {
      switch (event.type) {
      case ButtonPress:                      
        if (event.xbutton.button == Button1) ;
        break;
      case ButtonRelease:
        if (event.xbutton.button == Button1) {
	  if (!mmapped)
	    delete imdata;
	  rectangle( pos[0].x, pos[0].y, pos[1].x, pos[1].y, v, 0);
	                                         //erase old rect
          XSelectInput (c->display, winid, NoEventMask);
          return pos;
        }
        break;
      case MotionNotify:
        if ( (event.xmotion.state & Button1Mask) == 0 &&
	     (event.xmotion.state & Button3Mask) == 0 &&
	     (event.xmotion.state & Button2Mask) == 0 
	     )  {
          	rectangle( pos[0].x, pos[0].y, pos[1].x, pos[1].y, v, 0);
          					 //erase old rect
		pos[0].x = event.xmotion.x-size_x/2;
		pos[0].y = event.xmotion.y-size_y/2;
          	pos[1].x = event.xmotion.x+size_x/2;
          	pos[1].y = event.xmotion.y+size_y/2;
          	rectangle( pos[0].x, pos[0].y, pos[1].x, pos[1].y, v, 1);
          					 //draw new rect
          }
        break;
      default:
      // cerr << "Event type = " << event.type << " ( " << CompletionType << ")" << endl;
        break;
      }
    }
  }
  if (!mmapped) 
    delete imdata;
  flush();
}

//* Method : XWindow::get_region
//* Returns a list of positions, with each click defining a position
//* As with other list-returners, there really should be a class which
//* will automatically destruct itself so the caller does not have to
//* remember to delete the pointer returned
position*
XWindow::get_multi_pos(Video &v, int* count,char *name, int length,int dovidupdate)
{
  (*count)=0;
  
  position* pos=(position *)calloc(1, sizeof(position) );
  pos[0].x = v.width() / 2;
  pos[0].y = v.height() / 2;
  pos[0].angle  = 0;

  // window stuff
    
  if (!is_open())
    open(v.width(),v.height(),name);
  else
    resize(v.width(),v.height(),name);

  Set_Name(name);

  int mmapped = (v.is_mappable() && (v.bytesperpixel() == 1));
  int *imdata;
  XEvent event;

  if (mmapped != 1) 
    imdata = new int[v.width() * v.height()];  //.display initial image
  else
    cout << "Using memory mapping for image acquisition" << endl;


  XSelectInput (c->display, winid,
                ButtonPressMask | ButtonReleaseMask | PointerMotionMask);

  while (1) {                                // main loop

    // First, check if there is any new data to display

    if (!mmapped) {
      v.grab_static (imdata, v.width()/2, v.height()/2, 
		     v.width(), v.height());
      put_image(imdata,0,0,v.width(),v.height(),1);
    }
    else {
      put_image((u_char *)(v.direct_memptr()),0,0,v.width(),v.height(),1);
    }

    while (XCheckMaskEvent (c->display, ~0, &event)) {
      switch (event.type) {
      case ButtonPress:                      
        if (event.xbutton.button == Button1) {
          (*count)++;
          pos=(position *) realloc( pos, sizeof(position)*((*count)+1) );
          	//start next click/position. the old one is remembered
        }
        break;
      case ButtonRelease:
        if (event.xbutton.button == Button3) { //exit
	  if (!mmapped)
	    delete imdata;
          XSelectInput (c->display, winid, NoEventMask);
          return pos;
        }
        break;
      case MotionNotify:
	if ( (event.xmotion.state & Button1Mask) == 0 &&
	     (event.xmotion.state & Button3Mask) == 0 &&
	     (event.xmotion.state & Button2Mask) == 0 
	     )  {
	  v.clearline (pos[*count].x, pos[*count].y, 
		       length, pos[*count].angle);
	  pos[(*count)].x = event.xmotion.x;
	  pos[(*count)].y = event.xmotion.y;
	  v.line(pos[*count].x, pos[*count].y, 
		 length, pos[*count].angle);
	} 
        break;
      default:
        break;
      }
    }
  }
  if (!mmapped)
    delete imdata;
  flush();
  
}

position* 
XWindow::get_multi_pos(Video &v, char *name, int length,int dovidupdate)
{
#warning unimplemented: get_multi_pos
  return NULL;
}


// Short term hack by ADR to let me point to a region on an static Image.
void XWindow::get_region(XVImage<int> &i, char *name, position &p1, position &p2)
{
  p1.x = p2.x = i.width() / 2;
  p1.y = p2.y = i.height() / 2;
  p1.angle = p2.angle = 0;
  
  if (!is_open())
    open(i.width(),i.height(),name);
  else
    resize(i.width(),i.height(),name);

  Set_Name(name);  // why the name again? Doesn't open/resize do it?

  int *imdata;
  XEvent event;

  XSelectInput (c->display, winid,
                ButtonPressMask | ButtonReleaseMask | PointerMotionMask);

  while (1) { 
    // First, draw it on the display
    put_image(i.data(),0,0,i.width(),i.height(),1);
    rectangle( p1.x, p1.y, p2.x-p1.x, p2.y-p1.y, 0xffffff); // draw new rect

    while (XCheckMaskEvent (c->display, ~0, &event)) {
      switch (event.type) {
      case ButtonRelease:
        if (event.xbutton.button == Button1) { //exit
	  rectangle( p1.x, p1.y, p2.x-p1.x, p2.y-p1.y, 0); // erase old rect
          XSelectInput (c->display, winid, NoEventMask);
	  return;
        }
        break;
      case MotionNotify:
        if ( (event.xmotion.state & Button1Mask) == 0 )  {
	  p1.x = event.xmotion.x;
	  p1.y = event.xmotion.y;
	} else {
	  rectangle( p1.x, p1.y, p2.x-p1.x, p2.y-p1.y, 0); // erase old rect
	  p2.x = event.xmotion.x;
	  p2.y = event.xmotion.y;
	  rectangle( p1.x, p1.y, p2.x-p1.x, p2.y-p1.y, 0xffffff); // draw new rect
	}
        break;
      default:
        break;
      }
    }
  }
  flush();
}




//* Method : XWindow::TrySharedImage
//*
//* This code originally from the mesa (public domain OpenGL) source...
//* 
//* Not only does this source gracefully exit when a shared memory image
//* can't be created, but it also doesn't leave any shared memory segments
//* hanging around
//*
//* Also defined is a static int and static function for the case
//* where there is no Shared Memory
#ifndef noSHM
static int S_Xerror;
static int S_handleXerror(Display*,XErrorEvent*)
{
  S_Xerror = 1;
  return 0;
}
#endif

XImage*
XWindow::TrySharedImage()
{
#ifndef noSHM
   XImage *image;
   Display *display = c->display;

   image = XShmCreateImage(display,
                           c->visual,
			   c->depth,
                           ZPixmap,
			   NULL,
			   &IF,
			   width, height);
   if (image == NULL) {
      cerr << "XShmCreateImage problem" << endl;
      return 0;
   }

   // create the shared memory segment
  IF.shmid = shmget(IPC_PRIVATE,
                          image->bytes_per_line * height,
                          IPC_CREAT | 0777);
  IF.readOnly = False;
  if (IF.shmid < 0) {
     cerr << "shmget() problem" << endl;
     return 0;
  }

  data = (char *) (IF.shmaddr = image->data = (char *) shmat(IF.shmid, 0, 0));

  if (data  == (char *) -1) {
    cerr << "shmat() problem" << endl;
    return 0;
  }

  XSync(display, False);

  // set the x error handler 
  int (*old_handler)(Display *, XErrorEvent *);
  old_handler = XSetErrorHandler(S_handleXerror);

  // we need
  //   to be especially careful with this call, because it may fail
  //   if X is being used remotely

  // have the server attach to the shared memory segment 
  S_Xerror = 0;
  XShmAttach(display, &IF);
  XSync(display, False);

  if (S_Xerror) {
    // there was an error; can't use shared memory images
    XSetErrorHandler(old_handler);

    // detach from the segment
    shmdt(IF.shmaddr);

    // remove the shmid of the shared memory segment
    shmctl(IF.shmid, IPC_RMID, 0);

    return 0;
  }

  // remove the shmid of the shared memory segment; no one else
  //   needs to attach to it, so it will disappear once the client
  //   quits
  shmctl(IF.shmid, IPC_RMID, 0);

  // reset the X error handler
  XSetErrorHandler(old_handler);

  CompletionType = XShmGetEventBase(c->display) +  ShmCompletion;
  sent_image = 0;

  return image;
#else
  // We aren't doing shared memory, so return 0
  return 0;
#endif
}



 
