/*                                                                -*-c++-*-
    Copyright (C) 1991 Gregory D. Hager and Sidd Puri (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//-----------------------------------------------------------------------------
//
//  Console.cc
//
//  Definition of XConsole member functions.
//
//  7 - 1 - 93
//  Sidd Puri
//  8 - 1 - 94
//  Gregory D. Hager
//  7 - 5 - 95
//  Jonathan Wang
//
//  7 - 25 - 95
//  Gregory D. Hager
//  Added 16 bit color support.
//
//  11 - 25 -96
//  Loring Holden (lsh@cs.brown.edu)
//  Added support for 24 bit color when 24 bit color is not default visual
//  Fixed shared memory support so shared mem segments won't hang around
//
//-----------------------------------------------------------------------------

#include <site_config.h>
#include <X11/Xlib.h>
#ifndef noSHM
#include <X11/extensions/XShm.h>
#endif
#include <X11/cursorfont.h>
#include <X11/Xutil.h>
#include <string.h>	// for strcmp()
#include <stdlib.h>	// for abs()
#ifndef noSHM
#include <sys/ipc.h>
#include <sys/shm.h>
extern "C" int XShmGetEventBase(Display *display);
#endif
#include <unistd.h>

#include "XConsole.hh"
#include "Video.hh"
#include "Image.hh"

//-----------------------------------------------------------------------------
//  Helper functions
//-----------------------------------------------------------------------------

static int reversebits (int in) {
  int out = 0;
  for (int i = 0; i < 8; i++) {
    out <<= 1;
    out |= in & 1;
    in  >>= 1;
  }
  return out;
}

static int colorgain (int col) {
  float x = float (col) / 256;
  return int (256 * (-x*x + 2*x));
}



//* Method : XConsole::XConsole
//* Constructor for XConsole - makes the appropriate X calls
//* to create the necessary data structures (the colormap, display, gc...)
XConsole::XConsole(int w_color, int synced)
{
  display = XOpenDisplay (0);		     // open default display
  if (display == 0) {
    cerr << ("Can't open X window");
    exit(1);}

  if (synced)
    XSync(display, 0);

  is_color = w_color;
  screen = XDefaultScreen (display);	     // store defaults
  root = XDefaultRootWindow (display);

  XVisualInfo visInfo;
  int tryTrueColor = 0; // this controls the following if expression
  if (is_color && tryTrueColor &&
                  XMatchVisualInfo(display,screen, 24, TrueColor, &visInfo)) {
    visual = visInfo.visual;
    depth = 24;
    colormap = XCreateColormap(display, root, visual, AllocNone);
  } else {
     XWindowAttributes attributes;
     visual = XDefaultVisual (display, screen);
     colormap = XDefaultColormap (display, screen);
     
    
     gc = XDefaultGC (display, screen);
    


     XGetWindowAttributes (display, root, &attributes);
     depth = attributes.depth;
  }


  // there is  a conflict here, primarily because 24 bit depth attributes
  // do mean the device is going to be giving color information
  if (depth < 24)
    {
      cout << "depth = " <<  depth << " " <<  w_color << endl;
      fill_color_map(w_color, depth);
    } 
  else 
    {
      cout << "Making redpixel" << endl;
      if (!(redpixel = GetPixel(True, False, False)))
	redpixel = GetPixel(False, True, False);      
      if (!redpixel)
	redpixel = GetPixel(False, False, True);      
      if (redpixel == 0)
	cout << "Could not allocate red pixel" << endl << flush;
      cout << "True Color!!" << endl;
    };
}

//* Copy constructor for XConsole - merely sets *this = xconsole_in
XConsole::XConsole(XConsole& x) {
  cerr << "Warning -- using XConsole copy constructor.\n";
  *this = x;
}

//* Method : XConsole::GetPixel
//* Helper function for getting a pixel of a higlight color
//* which is usually something close to red
unsigned long
XConsole::GetPixel(Bool red, Bool green, Bool blue)
{
  unsigned long pixel = 0;
  int value = 255;
  XColor color;

  while ((pixel == 0) && (value > 0)) {
     color.red = color.green = color.blue = 0;
     if (red) color.red = value << 8;
     else if (green) color.green = value << 8;
     else if (blue) color.blue = value << 8;
     if (XAllocColor(display, colormap, &color) == 0) {
	  // Could not allocate color
	  value--;
     } else pixel = color.pixel;
  }
  return pixel;
}	

//* Method : XConsole::fill_color_map
//* Member which allocates the colors. This function determines what kind of
//* display the current hardware supports (e.g. TrueColor, 8-bit colormapping, etc.)
//* so that the best display gets created.
void
XConsole::fill_color_map(int w_color,int depth)
{
  cout << "XConsole::fill_color_map" << endl;
  

  XColor color;
  int ncolors = 0,i;

    // Allocate pure red as the "outlier" color
    // If we can't get pure red, try others until we get something
  if (!(redpixel = GetPixel(True, False, False)))
     redpixel = GetPixel(False, True, False);

  if (!redpixel)
    redpixel = GetPixel(False, False, True);

  if (redpixel == 0)
     cout << "Could not allocate red pixel" << endl << flush;

    // allocate the rest of the colors
  if (!w_color) {
    cout << "filling monochrome colormap" << endl;
    pixels = new unsigned long[256];

    for (i = 0; i < 256; i++) {
      int col = reversebits (i);
      color.red = color.green = color.blue = colorgain (col) << 8;
      if (XAllocColor (display, colormap, &color) == 0) {
	pixels[col] = (unsigned long)-1;
      }
      else {
	pixels[col] = color.pixel;
	ncolors++;
      }
    }

      // Fix up the ones that weren't allocated 
    for (i = 0; i < 256 && pixels[i] == -1; i++);
    unsigned long prevpixel = pixels[i];
    for (i = 0; i < 256; i++)
      if (pixels[i] == -1)
	pixels[i] = prevpixel;
      else
	prevpixel = pixels[i];
  }
  else
    {
      cout << "filling " << depth << " bit colormap" << endl;
      int i,r,g,b;
      XColor color;
      unsigned long  pix;  unsigned long* pixPointer = &pix;
      unsigned long prevpixel;

      switch (depth) {
      case 8:

	pixels = new unsigned long[256];

        prevpixel = redpixel;
	for (i=255; i>=0 ; i--) {
	  r = i>>5  ;  g = (i>>2)&0x7  ;  b = i&0x3;
	  color.red   = 256 * (r*32+30);
	  color.green = 256 * (g*32+30);
	  color.blue  = 256 * (b*64+50);
	  if (XAllocColor (display, colormap, &color) == 0) {
	    pixels[i] = (unsigned long)BlackPixel(display,screen);
//	    pixels[i] = prevpixel;
	  }
	  else {
//	    prevpixel = pixels[i] = color.pixel;
	    pixels[i] = color.pixel;
	    ncolors++;
	  }
	}


/*  Old code for fixing up unallocated pixels

	for (i = 0; i < 256 && pixels[i] == -1; i++);
	  prevpixel = pixels[i];
	for (i = 0; i < 256; i++)
	    if (pixels[i] == -1)
	      pixels[i] = prevpixel;
	    else
	      prevpixel = pixels[i];
*/
	break;
	
      case 16:
	  // Let's assume we can do true color
	if (visual->c_class != TrueColor) {
	  cerr << ("Can only do 16 bits of true color\n");
	  exit(1);
	}
	  // Just to make it easy to deallocate these at the end
	pixels = new unsigned long[2];
	break;

      default:
	cerr << ("Unknown display size in color allocation\n");
        exit(1);
      }
    }
    cout << "Got " << ncolors << " colors for the console window." << endl;
}

//* Method : XConsole::map_mono
//* Member which transforms image data into the format required by an XImage
//* monochrome version
void
XConsole::map_mono(int *imdata, XImage *im, int n, int depth)
{
  char* data = im->data;
  int *endim = imdata+n;
  short int *sdata;
  long  int *idata;
  unsigned char monocolor;

  switch (depth)
    {
    case 8:   
      for (; imdata<endim;imdata++) 
	{
#ifdef REDBOUNDS
	  if ((*imdata & 0xffffff00))
	    *data++=redpixel;
	  else
	    *data++=pixels[*imdata];
#else
	  *data++=pixels[*imdata&0xff];
#endif
	}
      break;

    case 16: 
      sdata = (short int *)data;
      for (; imdata<endim;imdata++) 
	{
#ifdef REDBOUNDS
	  if ((*imdata & 0xffffff00))
	    *sdata++=redpixel;
	  else
	    *sdata++=pixels[*imdata];
#else
	  *sdata++=pixels[*imdata&0xff];
#endif
	}
      break;
    case 24: 
      idata = (long int *)data;
      for (; imdata<endim;imdata++) 
	{
#ifdef REDBOUNDS
	  if ((*imdata & 0xffffff00)) {
	    *idata++  = 0x00ff0000;
	  }
	  else {
	    monocolor = *imdata&0xff;
	    *idata++  = monocolor | (monocolor<<8) | 
	      (monocolor<<16) | (monocolor<<24);
	  }
#else
	  monocolor = *imdata&0xff;
	  *idata++  = monocolor | (monocolor<<8) | 
	    (monocolor<<16) | (monocolor<<24);
#endif

	}      
      break;

    default:      
      cerr << "Not able to handle display with " << depth << " bits" << endl;
      cerr << ("X Error"); exit(1);
    }
}

//* Method : XConsole::map_color
//* Member which trqansforms image data into the format required by an XImage
//* color version
void
XConsole::map_color(int *imdata, XImage *im, int n, int depth)
{
  char* data = im->data;
  int* endim = imdata+n;
  unsigned short int *sdata;
  int r,g,b;
  unsigned int lv;

  switch (depth)
    {
    case 8:   
      for (; imdata<endim;imdata++) 
	{
	  explode((unsigned int)(*imdata),r,g,b);
	  lv = (((r & 0xE0) | ((g & 0xE0) >> 3) | (b >> 6)) & 0xFF);
	  *data++=pixels[lv];
	}
      break;

    case 16: 
      sdata = (unsigned short int *)data;
      for (; imdata<endim;imdata++) {
	*sdata++ = ( ((*imdata & 0xF8) << 8) |
		    ((*imdata & 0xFC00) >> 5) |
		    ((*imdata & 0xF80000) >> 19));
      }
      break;
      
    case 24:
    case 32: 
      memcpy(data, imdata, 4 * n);
      break;
      
    default:
      cerr << "Not able to handle display with " << depth << " bits" << endl;
      cerr << ("X Error"); exit(1);
    }
}

//* Method : XConsole::map
//* Overload of map which allows the first argument to be an unsigned char*
//* rather than an int* (i.e. a different XVision image representation)
void
XConsole::map(u_char *imdata, XImage *im, int n, int depth)
{
  char* data = im->data;
  u_char *endim = imdata+n;
  short int *sdata;
  long  int *ldata;
  switch (depth) 
    {
    case 8:   
      for (; imdata<endim;imdata++) {*data++=pixels[*imdata];}
      break;
    case 16: 
      sdata = (short int *)data;
      for (; imdata<endim;imdata++) {*sdata++=pixels[*imdata];}
      break;
    case 24: 
      ldata = (long int *)data;
      for (; imdata<endim;imdata++) 
	*ldata++  = *imdata | (*imdata<<8) | (*imdata<<16) ;
      break; 
    default:
	cerr << "Not able to handle display with " << depth << " bits" << endl;
	cerr << ("X Error"); exit(1);
    }
}


