/*
    Copyright (C) 1991 Gregory D. Hager and  (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//-----------------------------------------------------------------------------
//
//  BlobColor.cc
//
//  Definition of BlobColor member functions based on Christopher Rasmussen's
//  code
//
//  7 - 18 -96
//  Greg Hager
//
//-----------------------------------------------------------------------------

#include <stdlib.h> // for exit()
#include "CWindow.hh"
#include "Video.hh"
#include "BlobColor.hh"
#include "RGB2int.hh"


  // Two ways of creating a BlobColor. 

BlobColor::BlobColor (Video *v,
		      int height_in,
		      int width_in,
		      float range_in,
		      int hsamp, int wsamp,
		      float min_procent,
		      float saturation_thresh) :
		      BasicBlob(v,height_in, width_in, hsamp, wsamp, min_procent),
		      variance(3,3), pix(3), mean(3)
{
  saturation_threshold = saturation_thresh;
  sample_size = 16;
  sd_threshold = range_in;
}

BlobColor::BlobColor (BlobColor &l) :
  BasicBlob(l), variance(l.variance), pix(l.pix), mean(l.mean), 
  saturation_threshold(l.saturation_threshold) {
  }
 


template <class T>
inline int saturated(const T& r, const T& g, const T& b, 
		     float* saturation_threshold) 
{
  T maximum = ((r > g) ? r : g);
  maximum = (maximum > b) ? maximum : b;
  // different meanings of saturation
#if 1
  // this insures that pixels used are not 
  // near saturation on any ONE particular band
  return (maximum > (T)*saturation_threshold);
#else 
  // the alternative is to use the HSI version of saturation
  // in this model, a pure white  pixel would have a saturation of 0.5
  // whereas something with a lot of red, but no blue or green would be discarded.
  // this nonlinear model creates a curved space in the RGB cube where 
  // pixels are accepted
  T sum = r+b+g;
  if (sum == 0)
    return 0;
  return (( (float)maximum / (float)sum ) > 
	  *saturation_threshold);
#endif
}

// The match function

int 
BlobColor::member_test(int x) {

  explode((unsigned int) x,pix[0],pix[1],pix[2]);

  if (!saturated(pix[0],pix[1],pix[2], &saturation_threshold)) 
    {
      pix -= mean;
      pix = variance * pix;
      return (pix.ip() < sd_threshold);
    }
  else
    {
      return 0;
    };

}

int BlobColor::state_init (float x_in, float y_in) {
  Image im(sample_size,sample_size);
  source()->grab_aligned(im,x_in,y_in);
  if (pca_rgb(im,im.width() * im.height())) {
    set_x(x_in);
    set_y(y_in);
    return 1;
  }
  else
    return 0;
}

int BlobColor::interactive_init (CWindow &w) {
  position *pos;
  do 
    {
      pos = w.get_region (*source(), width(), height(),
			  "Position Blob");
    }
  while (!state_init( (pos[0].x+pos[1].x)/2, (pos[0].y+pos[1].y)/2 ));

  delete pos;
  return 1;
}


int BlobColor::pca_rgb(const Image &im, int npixels)
{
  int i,m0 = 0;
  unsigned int *data = (unsigned int *)im.data();
  ColVector vec(3);
  
  mean = 0.0;
  variance = 0.0;

  // Compute zeroeth, first and second order statistics

  int x = 0;
  for (i = 0; i < npixels; i++) 
    {
      explode(*data++,vec[0],vec[1],vec[2]);
      if (!saturated(vec[0],vec[1],vec[2], &saturation_threshold)) 
	{
	  m0 ++;
	  mean += vec;
	  variance += vec.op();
	}
      else 
	{
	  x++;         
	}
    }
  
  // information for debugging
  printf ("\n%d saturated pixels out of %d\n", x, m0);
  if (m0 < npixels*min_percent) {
    printf("Insufficient non-saturated data\n");
    return 0;
  }

  // Compute the real mean and variance
  mean /= m0;
  variance /= m0;
  variance -= mean.op();

  //cerr << "variance matrix" << endl;
  //cerr << variance << endl << endl;

  // Invert it
  variance = variance.i();
  //cerr << "inverse of variance matrix" << endl;
  //cerr << variance << endl << endl;
  
  // We're done.
  return 1;

}













